import datetime as dt

from six.moves import cPickle

import mongoengine as me

from sandbox.common import patterns

from . import base


class State(base.ConnectionSwitcherMixin, me.Document):
    """
    The class represents a mapping between Python object and database storage for "State" entity represents database
    instances
    """

    @patterns.singleton_property
    def shards(self):
        return dict((shard.name, shard) for shard in self.shards_info)

    def add_shard(self, shard):
        self.shards_info.append(shard)
        self.shards[shard.name] = shard

    #: The sub-document encapsulates all the discovered information about agent's hardware.
    class Shard(me.EmbeddedDocument):
        #: shard name
        name = me.StringField(min_length=1)
        #: Time of last information update.
        updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)
        #: Time of last information update.
        shard_cpu_updated = me.DateTimeField()
        #: cpu usage on host
        shard_cpu_usage = me.FloatField(default=0)
        # shard information
        information = me.BinaryField()
        # reloaded timestamp
        shard_reloaded_ts = me.DateTimeField()
        # replicaset id
        replicaset = me.StringField()

        @patterns.singleton_property
        def info(self):
            return cPickle.loads(self.information)

        def update_info(self, info):
            self.info.clear()
            self.info.update(info)
            self.information = cPickle.dumps(info)

    #: First part of a fully qualified domain name, for example, "sandbox1234" for sandbox1234.search.yandex.net
    name = me.StringField(min_length=1, primary_key=True)
    #: Time of last information update.
    updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)
    #: Time of last information update.
    cpu_updated = me.DateTimeField()
    #: Enabled api on host
    api_enabled = me.BooleanField(required=True, default=True)
    #: cpu usage on host
    cpu_usage = me.FloatField(default=0)
    #: shards information
    shards_info = me.ListField(me.EmbeddedDocumentField(Shard))
