import itertools
import collections
import datetime as dt
import mongoengine as me

from sandbox.common import enum
from sandbox.common import math

from . import base


class Statistics(base.ConnectionSwitcherMixin, me.Document):
    """
    The class represents a mapping between Python object and database storage for "Statistics" entity.
    """

    class Keys(enum.Enum):
        STATUS = None
        ENQUEUED_TASKS = None
        STORAGE = None
        ENQUEUE_TIME = None

    # Service key
    key = me.StringField(primary_key=True, choices=list(Keys))
    # Date of last document update
    updated = me.DateTimeField(required=True, default=dt.datetime.utcnow)
    # Corresponding data
    data = me.DictField(required=True)


class Weather(base.ConnectionSwitcherMixin, me.Document):

    def update(self, execute_data, eta_data, bound, percentile):
        """
        Merge `self` and `other` documents. If data.history of resulting document larger than bound items
        it will be cut to last bound ones.

        :param other: TaskWeather document to merge it to self
        :param bound: amount of last history items to save
        :return: modified self
        :rtype: mapping.TaskWeather
        """

        def tail(iterable, n):
            """Return an iterator over the last n items"""
            # tail('ABCDEFG', 3) --> E F G
            return iter(collections.deque(iterable, maxlen=n))

        def combine(old, new):
            return list(
                tail(itertools.chain(old, new), bound)
            )

        self.data.history = combine(self.data.history or [], execute_data)
        self.data.weather = sum(self.data.history)

        self.data.eta_history = combine(self.data.eta_history or [], eta_data)
        perc = math.percentile(sorted(self.data.eta_history), percentile)
        self.data.eta = round(perc if perc >= 1 else 0, 2)

        return self

    class Data(me.EmbeddedDocument):
        # Last execution results indicator calculated according to `history` field
        weather = me.IntField(min_value=0)
        # Last execution results
        history = me.ListField(me.IntField())
        # Estimated time of arrival of current task type calculated as median of last executions
        eta = me.FloatField(min_value=0)
        # Last execution durations
        eta_history = me.ListField(me.IntField())

    # Task type
    type = me.StringField(primary_key=True)
    # Corresponding statistics of current task type
    data = me.EmbeddedDocumentField(Data, required=True)
