import datetime as dt
import mongoengine as me

import sandbox.common.types.template as ctte

from . import base
from . import task


class TaskTemplate(base.ConnectionSwitcherMixin, me.Document):
    """ Task template document. """

    meta = {
        "indexes": [
            "task.type",
            "author",
            "shared_with",
            "favorites",
        ],
    }

    class Time(me.EmbeddedDocument):
        """ Time marks aggregate. """
        #: Date of template creation.
        created = me.DateTimeField(db_field="ct", default=dt.datetime.utcnow, required=True)
        #: Date of last template last update
        updated = me.DateTimeField(db_field="up", default=dt.datetime.utcnow, required=True)

    class Task(me.EmbeddedDocument):
        class ParameterMeta(task.ParametersMeta.ParameterMeta):
            # Field must be hide
            hide = me.BooleanField(default=False)
            # Filter for parameter
            filter = me.DictField()
            # Get default value from code
            default_from_code = me.BooleanField(default=True)
            # Parameter value
            value = task.ParameterValue()

        #: Custom parameters meta information
        input_parameters = me.ListField(me.EmbeddedDocumentField(ParameterMeta))
        #: Common parameters meta information
        common_parameters = me.ListField(me.EmbeddedDocumentField(ParameterMeta))
        #: Requirements meta information
        requirements = me.ListField(me.EmbeddedDocumentField(ParameterMeta))
        #: Task type.
        type = me.StringField(required=True)

    # Template alias
    alias = me.StringField(primary_key=True)
    # Template author
    author = me.StringField(required=True)
    # Template status
    status = me.StringField(required=True, choices=list(ctte.Status))
    # Template description
    description = me.StringField(default="")
    # List of users or groups, who can edit template
    shared_with = me.ListField(me.StringField(), default=[])
    # Template is in favorites for this users
    favorites = me.ListField(me.StringField())
    # Task information
    task = me.EmbeddedDocumentField(Task, default=Task())
    # Time properties
    time = me.EmbeddedDocumentField(Time, default=Time())


class TemplateAudit(me.Document):
    """ Template audit. """

    meta = {
        "indexes": [
            "template_alias", "date", "author"
        ],
    }

    class ChangedField(me.EmbeddedDocument):
        # Old field info
        old = me.EmbeddedDocumentField(TaskTemplate.Task.ParameterMeta)
        # New field info
        new = me.EmbeddedDocumentField(TaskTemplate.Task.ParameterMeta)

    class ChangedProperty(me.EmbeddedDocument):
        # old property info
        old = task.ParameterValue()
        # New property info
        new = task.ParameterValue()

    #: Alias of template
    template_alias = me.StringField(required=True, db_field="ta")
    #: Date and time of template update
    date = me.DateTimeField(required=True, db_field="dt", default=dt.datetime.utcnow)
    #: The entry author
    author = me.StringField(db_field="a")
    #: Changed input parameters
    input_parameters = me.ListField(me.EmbeddedDocumentField(ChangedField), default=[])
    #: Changed common parameters
    common_parameters = me.ListField(me.EmbeddedDocumentField(ChangedField), default=[])
    #: Changed requirements
    requirements = me.ListField(me.EmbeddedDocumentField(ChangedField), default=[])
    #: Changed template properties
    properties = me.MapField(me.EmbeddedDocumentField(ChangedProperty), default={})
