import datetime as dt
import mongoengine as me

from sandbox.common import hash
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from . import base


class TimeTrigger(base.ConnectionSwitcherMixin, me.Document):
    """ Wait time trigger """
    meta = {"indexes": ["time"]}

    # task id that owns the trigger
    source = base.ReferenceField(primary_key=True)
    # time in UTC at which the trigger will work
    time = me.DateTimeField()
    # creation time
    ctime = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    # task session token
    token = me.StringField()
    # set to true when source task switches to WAIT_TIME and ready to be restarted
    activated = me.BooleanField(default=False)


class TaskStatusTrigger(base.ConnectionSwitcherMixin, me.Document):
    """ Trigger for waiting of task status """
    meta = {"indexes": ["targets", "statuses"]}

    # task id that owns the trigger
    source = base.ReferenceField(primary_key=True)
    # task ids that will be checked by the trigger
    targets = me.ListField(base.ReferenceField())
    # task statuses that are waiting by the trigger
    statuses = me.ListField(me.StringField(choices=list(iter(ctt.Status))))
    # wait all of the targets or any of them
    wait_all = me.BooleanField()
    # creation time
    ctime = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    # task session token
    token = me.StringField()
    # set to true when source task switches to WAIT_TASK and ready to be restarted
    activated = me.BooleanField(default=False)
    # last check by TaskStateSwitcher wait_task garbage collector
    last_gc_check = me.DateTimeField(default=lambda: dt.datetime.min, required=True)
    # time the task was ready to execute again
    fired_at = me.DateTimeField()


class TaskOutputTrigger(base.ConnectionSwitcherMixin, me.Document):
    """ Trigger for waiting of tasks' output changes """
    meta = {"indexes": ["targets"]}

    class TargetField(me.EmbeddedDocument):
        # task id
        target = base.ReferenceField(required=True)
        # task output field
        field = me.StringField(required=True)

    # task id that owns the trigger
    source = base.ReferenceField(primary_key=True)
    # task ids that will be checked by the trigger
    targets = me.ListField(me.EmbeddedDocumentField(TargetField))
    # wait all of the targets or any of them
    wait_all = me.BooleanField()
    # creation time
    ctime = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    # task session token
    token = me.StringField()


class TaskStatusNotificationTrigger(base.ConnectionSwitcherMixin, me.Document):
    """ Trigger for task status notifications """
    meta = {"indexes": ["source"]}

    # Avoid duplicates when adding a notification for a task
    id = me.StringField(primary_key=True, db_field="nh")
    # task id that owns the trigger
    source = base.ReferenceField()
    # task statuses that are waiting by the trigger
    statuses = me.ListField(me.StringField(choices=list(iter(ctt.Status))))
    # send notification with specified transport param
    transport = me.StringField(choices=list(iter(ctn.Transport)), required=True)
    # list of addresses
    recipients = me.ListField(me.StringField(), required=True)
    # creation time
    ctime = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    # check status
    check_status = me.StringField(df_field="cs", choices=list(ctn.JugglerStatus))
    # tags for Juggler
    juggler_tags = me.ListField(me.StringField(min_length=1, max_length=128), db_field="jt")

    def save(self, **kwargs):
        self.id = hash.json_hash(
            dict(
                source=self.source,
                statuses=self.statuses,
                transport=self.transport,
                recipients=self.recipients,
            ),
            sort_lists=True
        )
        return super(TaskStatusNotificationTrigger, self).save(**kwargs)
