import datetime as dt
import mongoengine as me

import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu

from sandbox.common import patterns

from . import abcd
from . import base


class OAuthCache(base.ConnectionSwitcherMixin, me.Document):
    """
    Mapping class between Python objects and OAuth tokens cache collection in database.
    """
    meta = {
        "collection": "oauth_cache",
        "indexes": [
            "source",
            "app_id",
            {
                "fields": ["task_id"],
                "sparse": True,
                "unique": True,
            }
        ]
    }

    #: contains information to be updated for the current session
    class UpdateData(me.EmbeddedDocument):
        #: execution kill timeout
        kill_timeout = me.IntField(db_field="kt")

    class Properties(me.EmbeddedDocument):
        #: login of author for session creation request
        session_maker = me.StringField()
        #: Session for this service
        service = me.StringField()
        #: Associated with Sandbox task id
        sandbox_task_id = me.IntField()

    #: Token identifier
    token = me.StringField(primary_key=True)
    #: User's login, which can be identified by the token
    login = me.StringField(required=True)
    #: Task owner
    owner = me.StringField()
    #: Token creation time
    created = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    #: Token validation time
    validated = me.DateTimeField(default=dt.datetime.utcnow, required=True)
    #: Token validation limit in seconds
    ttl = me.IntField(min_value=1)
    #: Source of the token (sandbox client, ssh, passport, tasklet installation)
    source = me.StringField(required=True)
    #: Unique application identifier, specific to the source (task_id, bb source_id or tasklet_id)
    app_id = me.StringField()
    #: Task id associated with this token
    task_id = me.IntField()
    #: Task session state
    state = me.StringField(choices=list(ctt.SessionState))
    #: Task abort reason
    abort_reason = me.StringField(
        choices=[ctt.Status.STOPPING, ctt.Status.DELETED, ctt.Status.EXPIRED]
    )
    #: Vault encryption key
    vault = me.BinaryField()
    #: Information to be updated for the current session
    update_data = me.EmbeddedDocumentField(UpdateData)
    #: Session properties
    properties = me.EmbeddedDocumentField(Properties)

    @property
    def expired(self):
        return self.state == ctt.SessionState.EXPIRED

    @property
    def aborted(self):
        return self.state == ctt.SessionState.ABORTED

    @property
    def client_id(self):
        if ":" in self.source:
            return self.source.split(":", 1)[1]
        return None

    @property
    def client(self):
        """ For compatibility between web request.session and serviceapi request.session. Don't use it. """
        return self.client_id

    @property
    def task(self):
        """ For compatibility between web request.session and serviceapi request.session. Don't use it. """
        return self.task_id

    @property
    def is_task_session(self):
        return self.source.startswith(ctu.TokenSource.CLIENT)

    @property
    def is_external_session(self):
        return self.source == ctu.TokenSource.EXTERNAL_SESSION

    def get_update_options(self):
        if self.update_data is None:
            return None
        if self.update_data.kill_timeout:
            return {"kill_timeout": self.update_data.kill_timeout}

    @classmethod
    def client_sessions(cls, host):
        return cls.objects(source="{}:{}".format(ctu.TokenSource.CLIENT, host))

    @classmethod
    def locked_task_ids(cls):
        return list(cls.objects(task_id__exists=True).fast_scalar("task_id"))

    @classmethod
    def touch(cls, token):
        cls.objects(token=token).update_one(set__validated=dt.datetime.utcnow())


class User(base.ConnectionSwitcherMixin, me.Document):
    """
    Mapping class between Python objects and "users" collection in database.
    """

    meta = {"indexes": ["session.id", "uid"]}

    #: user login
    login = me.StringField(primary_key=True)

    #: super_user rights
    super_user = me.BooleanField()

    #: API methods that are allowed to execute
    allowed_api_methods = me.ListField(me.StringField())

    #: staff validate time
    staff_validate_timestamp = me.DateTimeField()

    class Session(me.EmbeddedDocument):
        """ Blackbox authentication cache container. """
        id = me.StringField(required=True)
        updated = me.DateTimeField(default=dt.datetime.utcnow, required=True)

    #: Blackbox authentication cache.
    session = me.EmbeddedDocumentField(Session)

    #: Whether user is robot
    robot = me.BooleanField(default=False)

    #: User preferences. Selected filters, options.
    preferences = me.DictField()

    #: User roles
    roles = me.ListField(me.StringField(choices=list(ctu.Role)))

    # User telegram login
    telegram_login = me.StringField()

    # User private chat id with bot
    messenger_chat_id = me.StringField()

    # Api quota
    api_quota = me.LongField(min_value=0)

    # User uid from Passport
    uid = me.StringField()

    def reload(self):
        super(User, self).reload()
        del self.groups

    @patterns.singleton_property
    def groups(self):
        return list(Group.objects(users=self.login))

    @patterns.singleton_property
    def groups_names(self):
        return list(Group.objects(users=self.login).fast_scalar("name"))


class Group(base.ConnectionSwitcherMixin, me.Document):
    """
    Users can unit to groups for tasks manage
    Additional feature - specific email address for group
    """

    meta = {
        "indexes": [
            "parent",
            {
                "fields": ["user_tags.name"],
                "sparse": True,
                "unique": True
            },
            "telegram_chat_id",
            "abc",
            "abcd_account.id",
            "abcd_account.folder_id",
            "bucket",
        ]
    }

    class PriorityLimits(me.EmbeddedDocument):
        """ Maximum allowed priorities for the group. """
        #: User Interface priority limits
        ui = me.IntField()
        #: API (REST, XMLRPC) priority limits
        api = me.IntField()

    class SyncSource(me.EmbeddedDocument):
        """ Source for syncing content of the group """
        class SyncSourceContent(me.EmbeddedDocument):
            name = me.StringField()
            logins = me.ListField(me.StringField())
        #: source name for syncing content of the group
        source = me.StringField(choices=list(iter(ctu.GroupSource)))
        #: names of groups in source from which to sync, plus users to be added independently. all comma-separated
        group = me.StringField()
        #: actual users per group in sync groups
        content = me.ListField(me.EmbeddedDocumentField(SyncSourceContent))

        def get_users(self):
            users = set()
            for source_content in self.content:
                users.update(source_content.logins)
            return users

    class UserTag(me.EmbeddedDocument):
        name = me.StringField()

    class JugglerSettings(me.EmbeddedDocument):
        class JugglerCheck(me.EmbeddedDocument):
            # Host check
            host = me.StringField()
            # Service check
            service = me.StringField()

        default_host = me.StringField()
        default_service = me.StringField()
        checks = me.MapField(me.EmbeddedDocumentField(JugglerCheck), default={})

    #: group name
    name = me.StringField(min_length=1, primary_key=True)
    #: group users
    users = me.ListField(me.StringField(), required=True)  # TODO: deprecated, delete in (SANDBOX-8919)
    #: ABC service identifier the group is linked with
    abc = me.StringField()
    #: group email address
    email = me.StringField()
    #: Maximum allowed priority for the group.
    priority_limits = me.EmbeddedDocumentField(PriorityLimits)
    #: computational quota value in QP divided by settings.serviceq.server.quota.external_scale
    quota = me.IntField(min_value=0)
    #: group used to account quota consumption
    parent = me.StringField()
    #: source for syncing content of the group
    sources = me.ListField(me.EmbeddedDocumentField(SyncSource))
    # Group private chat id with bot
    messenger_chat_id = me.StringField()
    # Telegram chat id
    telegram_chat_id = me.StringField()
    # Juggler settings
    juggler_settings = me.EmbeddedDocumentField(JugglerSettings)
    #: user tags
    user_tags = me.ListField(me.EmbeddedDocumentField(UserTag))
    #: group use mds for resources
    mds_strong_mode = me.BooleanField(default=False)
    #: transfer old resources to mds
    mds_transfer_resources = me.BooleanField(default=True)
    #: group is public
    public = me.BooleanField(default=False)
    #: ABCD account specific fields
    abcd_account = me.EmbeddedDocumentField(abcd.ABCDAccount, db_field="abcd")
    #: MDS-S3 bucket name (reference to the Bucket)
    bucket = me.StringField()

    def get_users(self):
        users = set()
        for source in self.sources:
            users.update(source.get_users())
        return users or set(self.users)

    def get_sources(self):
        """ Get all valid sources for the group """
        return [source for source in self.sources if source.source in ctu.GroupSource]


class RobotOwner(base.ConnectionSwitcherMixin, me.Document):
    """
    List of robots owned by the Sandbox user
    """
    #: user login
    login = me.StringField(primary_key=True)
    #: list of robots owned by the user
    robots = me.ListField(me.StringField(), required=True)
