"""
The module aggregates all the upgrade scripts, which can be applied to the database.
Generally saying, each script has `pre`, `main`, and `post` execution stages.
First and last ones are supposed to be executed on a working database, while the
`main` stage can only be executed while the whole service is in READONLY state.

`__pre__` and `__all__` global variables in this module determine the order, in which
appropriate upgrade scripts are executed.

Note: any usage of `yasandbox.database.model` classes, as well as `mongoengine` module
itself should be avoided inside upgrade scripts, because they can probably be executed
much later than the time model classes were developed, so the database scheme may
have already been significantly changed.

Also, keep in mind, while developing any stage of the upgrade script, that it must be
restartable, i.e., for example, if the database connection is lost in the middle of
execution process, the stage's code must be ready to be restarted to finish the stage
and do not leave the database in inconsistent state.
"""

from __future__ import absolute_import, print_function

import sys
import time

from sandbox import common

from . import tasks  # noqa
from . import resources  # noqa
from . import users  # noqa
from . import misc  # noqa
from . import schedulers  # noqa
from . import statistics  # noqa
from . import clients  # noqa
from . import vault  # noqa

# A list and an order of available pre-upgrade steps, which can be applied.
__pre__ = [
    'misc.RenameUpgradeScripts',
    'resources.LastUsageTime', 'tasks.UpdateTime', 'users.UseLoginAsID', 'tasks.UpdatePriority',
    'schedulers.UpdateSchedulerTaskCxt',
    'misc.DropIndexes', 'tasks.Audit',
    'statistics.UpdateEnqueueTimeStatistics',
    'tasks.UpdateStatuses', 'tasks.UpdateTriggerStatuses', 'tasks.UpdateAuditStatuses',
    'users.DropSessionCollection',
    'tasks.ExtractAuthorFieldFromCtx',
    'schedulers.UpdateSchedulerNotifications',
    'tasks.UpdateOldReleasedTasks',
    'clients.AddRevisionField',
    'schedulers.UpdateSchedulerCreateTime',
    'schedulers.AddWatchingSchedulersToCache',
    'resources.Expires',
    'tasks.UpdateDiskUsageMax',
    'resources.ReleasedTTL',
    'users.RenameGroups',
    'misc.DropExtraFields',
    'tasks.ExtractHostsChooserFieldsFromCtx',
    'tasks.DnsFieldAsEnum',
    'schedulers.MakeSchedulersSimilarToTasks',
    'tasks.FixNotificationsField',
    'vault.EncryptSecrets',
    'users.MultiSourceGroups'
]
# The order in which main- and post- upgrade steps are performed
__all__ = __pre__[:]


class Type(common.utils.Enum):
    """ Upgrade type enumeration. """
    PRE = 'pre'
    MAIN = 'main'
    POST = 'post'


def apply_missing(applied, pre_only=False):
    """
    Applies missing database upgrade script. Yields each applied script name.
    :param applied:     A collection of lists of already applied scripts per type.
    :param pre_only:    Flags that only pre-upgrade scripts should be applied.
    """
    for name in _apply_list(__pre__, applied.pre, Type.PRE):
        yield (name, Type.PRE)

    if pre_only:
        return

    for type in (Type.MAIN, Type.POST):
        for name in _apply_list(__all__, getattr(applied, type), type):
            yield (name, type)


def _apply_list(available, applied, type):
    counter = 0
    cz = common.console.AnsiColorizer()
    prefix = '[{}] '.format(cz.red(type.upper()))
    for name in available:
        skip = name in applied
        script = sys.modules[__name__]
        for attr in name.split('.'):
            script = getattr(script, attr)
        now = time.time()
        if not skip or counter:
            print('{}{} {} [{}] {}'.format(
                prefix,
                cz.yellow('Applying'),
                cz.white(name),
                cz.black(script.__doc__.strip()),
                cz.yellow('...')
            ))
        if skip:
            if counter:
                print('{}{}'.format(prefix, cz.yellow("Already applied.")))
            continue

        applier = script()
        getattr(applier, type)()
        print('{}{} [{}].'.format(
            prefix,
            cz.green("Successfully applied {}".format(cz.white(name))),
            cz.white(common.utils.td2str(time.time() - now)))
        )
        counter += 1
        yield name
