import abc


class UpgradeStep(object):
    """
    Single upgrade step object. Will be instantiated and executed automatically by calling `run` method.
    """
    __metaclass__ = abc.ABCMeta

    def pre(self):
        """
        Optional sub-step of the upgrade script. Can be executed while the service is operating in normal mode.
        It is a good place to pre-create some calculated document fields, build indexes, etc.
        Will be automatically executed before the `main` part.
        """
        pass

    @abc.abstractmethod
    def main(self):
        """
        The main part of the upgrade step. It is required to run this part exclusively, while the service
        will be automatically switched into READONLY mode.
        """
        pass

    def post(self):
        """
        Optional sub-step of the upgrade script. Can be executed when the service switched back to normal operation.
        It is a good place to remove some unused document attributes, drop indexes, etc.
        Will be automatically executed after the `main` part.
        """
        pass
