import copy

from pymongo import errors
from mongoengine import connection

from sandbox import common
from sandbox.yasandbox.database import mapping

from . import base


class RenameUpgradeScripts(base.UpgradeStep):
    """ Renames applied upgrade script in accordance with last naming changes. """

    def pre(self):
        need_save = False
        model = mapping.Settings
        for attr in model.updates.applied:
            lst = getattr(model.updates.applied, attr)
            orig = copy.copy(lst)
            for i, v in enumerate(lst):
                if v.startswith('user.'):
                    lst[i] = v.replace('user.', 'users.')
                if v.startswith('task_priority.'):
                    lst[i] = v.replace('task_priority.', 'tasks.')
            if lst != orig:
                need_save = True

        if need_save:
            model.save()

    def main(self):
        # Nothing to do here actually.
        pass


class DropIndexes(base.UpgradeStep):
    """ Drops unnecessary compound indexes. """
    def main(self):
        # Nothing to do here actually.
        pass

    def post(self):
        db = connection.get_db()
        for coll, idx in (
                ('resource', 'hosts.h_1_hosts.st_1'),
                ('task', 'exc.host_1_exc.cncl_1'),
                ('task', 'exc.st_1__id_-1'),
                ('task', 'exc.cncl_1'),
        ):
            with common.console.LongOperation("Dropping '{}' index from '{}' collection".format(idx, coll)):
                try:
                    db[coll].drop_index(idx)
                except errors.OperationFailure:
                    pass


class DropExtraFields(base.UpgradeStep):
    """ Drops extra documents' fields. """

    def main(self):
        # Nothing to do here actually.
        pass

    def post(self):
        db = connection.get_db()
        for field, coll in (
                ('trusted', 'user'),
                ('skynet_key_rights', 'user'),
                ('restarting', 'client'),
                ('trig', 'task'),
                ('next_run', 'service'),
        ):
            with common.console.LongOperation("Dropping '{}' field from '{}' collection documents".format(field, coll)):
                db[coll].update({field: {"$exists": True}}, {"$unset": {field: True}}, multi=True)
