import bson
import cPickle
import calendar
import datetime as dt
import mongoengine as me

from sandbox.common import utils
from sandbox.common import console
import sandbox.common.types.task as ctt
import sandbox.common.types.scheduler as cts

from . import base


class UpdateSchedulerTaskCxt(base.UpgradeStep):
    """
    Convert schedulers priority value in task_ctx to class-subclass scheme
    """

    priority_map_cache = []

    class Priority(object):
        """
        Task priority
        """
        class Class(utils.Enum):
            USER = 2
            SERVICE = 1
            BACKGROUND = 0

        class Subclass(utils.Enum):
            HIGH = 2
            NORMAL = 1
            LOW = 0

        def __init__(self, cls=None, scls=None):
            if cls is None:
                cls = self.Class.BACKGROUND
            if scls is None:
                scls = self.Subclass.LOW
            if cls not in self.Class or scls not in self.Subclass:
                raise ValueError("Incorrect priority values: class %s, subclass %s", cls, scls)
            self.cls = cls
            self.scls = scls

    @classmethod
    def map_int_priority(cls, int_prio):
        if not cls.priority_map_cache:
            cls.priority_map_cache = []
            for range_, priority in [
                ((-100, -50), cls.Priority(cls.Priority.Class.BACKGROUND, cls.Priority.Subclass.LOW)),
                ((-50, -25), cls.Priority(cls.Priority.Class.BACKGROUND, cls.Priority.Subclass.NORMAL)),
                ((-25, 0), cls.Priority(cls.Priority.Class.BACKGROUND, cls.Priority.Subclass.HIGH)),
                ((0, 5), cls.Priority(cls.Priority.Class.SERVICE, cls.Priority.Subclass.LOW)),
                ((5, 11), cls.Priority(cls.Priority.Class.SERVICE, cls.Priority.Subclass.NORMAL)),
                ((11, 25), cls.Priority(cls.Priority.Class.SERVICE, cls.Priority.Subclass.HIGH)),
                ((25, 50), cls.Priority(cls.Priority.Class.USER, cls.Priority.Subclass.LOW)),
                ((50, 75), cls.Priority(cls.Priority.Class.USER, cls.Priority.Subclass.NORMAL)),
                ((75, 101), cls.Priority(cls.Priority.Class.USER, cls.Priority.Subclass.HIGH)),
            ]:
                cls.priority_map_cache.extend([priority] * abs(range_[1] - range_[0]))

        return cls.priority_map_cache[min(max(-100, int_prio), 100) + 100]

    def main(self):
        collection = me.connection.get_db()['scheduler']
        pbar = console.ProgressBar('Updating data', collection.count())
        for i, row in enumerate(collection.find({}, {'_id': 1, 'task.context': 1})):
            tsk_ctx = cPickle.loads(row["task"]["context"])
            int_prio = tsk_ctx.pop("prio", 0)
            priority = self.map_int_priority(int_prio)
            tsk_ctx["prio_cls"] = priority.cls
            tsk_ctx["prio_scls"] = priority.scls
            collection.update(
                {"_id": row["_id"]},
                {"$set": {"task.context": bson.binary.Binary(cPickle.dumps(tsk_ctx))}}
            )
            pbar.update(i)
        pbar.finish()


class UpdateSchedulerNotifications(base.UpgradeStep):
    """
    Convert schedulers task context fields related to notifications to notification sub objects
    """
    NOTIFICATION_FIELDS = ("notify_via", "notify_if_finished", "notify_if_failed")

    @staticmethod
    def new_notifications(author, owner, groups, ctx):
        """ Convert old notification info format to database documents """
        notifications = []
        notify_via = ctx.get("notify_via")
        if notify_via:
            notify_if_finished = ctx.get("notify_if_finished")
            notify_if_finished_recipients = [author]
            if notify_if_finished is not None:
                notify_if_finished = notify_if_finished.strip()
                if notify_if_finished:
                    notify_if_finished_recipients.extend(map(lambda s: s.strip(), notify_if_finished.split(',')))
                else:
                    notify_if_finished_recipients = None
            if notify_if_finished_recipients:
                notifications.append({
                    "st": [ctt.Status.SUCCESS],
                    "tr": notify_via,
                    "rs": notify_if_finished_recipients
                })
            notify_if_failed = ctx.get("notify_if_failed")
            notify_if_failed_recipients = [author]
            if notify_if_failed is not None:
                notify_if_failed = notify_if_failed.strip()
                if notify_if_failed:
                    notify_if_failed_recipients.extend(map(lambda s: s.strip(), notify_if_failed.split(',')))
                else:
                    notify_if_failed_recipients = None
            if notify_if_failed_recipients:
                notifications.append({
                    "st": [ctt.Status.FAILURE],
                    "tr": notify_via,
                    "rs": notify_if_failed_recipients
                })
            notify_if_break_recipients = [author]
            if owner in groups:
                gr = groups[owner]
                if gr.get("email"):
                    notify_if_break_recipients.append(gr["email"])
            notifications.append({
                "st": [
                    ctt.Status.EXCEPTION,
                    ctt.Status.NO_RES,
                    ctt.Status.TIMEOUT,
                ],
                "tr": notify_via,
                "rs": notify_if_break_recipients
            })
        return notifications

    def main(self):
        collection = me.connection.get_db()["scheduler"]
        groups = {row['_id']: row for row in me.connection.get_db()["group"].find({})}
        pbar = console.ProgressBar('Updating data', collection.count())
        for i, row in enumerate(collection.find({}, {"_id": 1, "author": 1, "owner": 1, "task.context": 1})):
            tsk_ctx = cPickle.loads(row["task"]["context"])
            if not set(tsk_ctx).intersection(self.NOTIFICATION_FIELDS):
                continue
            notifications = self.new_notifications(row.get("author"), row.get("owner"), groups, tsk_ctx)
            for field in self.NOTIFICATION_FIELDS:
                tsk_ctx.pop(field, None)
            collection.update(
                {"_id": row["_id"]},
                {"$set": {
                    "task.notifications": notifications,
                    "task.context": bson.binary.Binary(cPickle.dumps(tsk_ctx))
                }}
            )
            pbar.update(i)
        pbar.finish()


class UpdateSchedulerCreateTime(base.UpgradeStep):
    """
    Convert schedulers created field to time.created
    """

    def main(self):
        collection = me.connection.get_db()["scheduler"]
        pbar = console.ProgressBar('Updating data', collection.count())
        for i, row in enumerate(collection.find({}, {"_id": True, "created": True})):
            collection.update(
                {"_id": row["_id"]},
                {
                    "$set": {"time.created": row["created"]},
                    "$unset": {"created": True}
                }
            )
            pbar.update(i)
        pbar.finish()


class AddWatchingSchedulersToCache(base.UpgradeStep):
    """
    Add watching schedulers to waiting task cache
    """

    def main(self):
        service_collection = me.connection.get_db()["service"]
        res = service_collection.find({"_id": "Scheduler"})
        if res.count() != 1:
            return
        srv_sched = res[0]
        ctx = srv_sched.setdefault("context", {})

        shed_collection = me.connection.get_db()["scheduler"]
        scheduler_ids = [sch["_id"] for sch in shed_collection.find({"status": "WATCHING"}, {"_id": True})]

        task_collection = me.connection.get_db()["task"]
        pipeline = [
            {'$project': {'_id': 1, 'sch': 1}},
            {'$sort': {'_id': 1}},
            {'$match': {'sch': {'$in': scheduler_ids}}},
            {'$group': {'_id': '$sch', 'task_id': {'$last': '$_id'}}},
        ]
        ret = task_collection.aggregate(pipeline)
        if ret.get('ok') != 1:
            raise me.OperationError('Unable to execute aggregation query: %r' % ret)

        ctx['watching_tasks_cache'] = [r['task_id'] for r in ret['result']]
        service_collection.update({"_id": srv_sched["_id"]}, {"$set": {"context": ctx}})


class MakeSchedulersSimilarToTasks(base.UpgradeStep):
    """ Make schedulers' fields similar to tasks' ones. In the name of inheritance. """

    @staticmethod
    def _upgrage_scheduler(row, scheduler_collection):
        if "plan" in row:
            return
        sch_ctx = cPickle.loads(row["context"])
        minute, hour, day, month, year = sch_ctx["start_date_time"]
        start_timestamp = calendar.timegm((year, month, day, hour, minute, 0, 0, 0, -1))
        task = row["task"]
        task_ctx = cPickle.loads(task["context"])
        task_ctx.pop("owner", None)
        execution_space = task_ctx.pop("execution_space", 0)
        priority = ctt.Priority(
            cls=int(task_ctx.pop("prio_cls")),
            scls=int(task_ctx.pop("prio_scls")),
        )

        fields_to_set = {
            "kt": task_ctx.get("kill_timeout", None),
        }
        for task_field, task_ctx_field in (
            ("flg", "important"),
            ("hid", "hidden"),
            ("fae", "fail_on_any_error"),
        ):
            fields_to_set[task_field] = bool(task_ctx.get(task_ctx_field, False))

        status = row["status"]
        time = row["time"]
        time["up"] = time["ct"] = time.pop("created")

        fields_to_set.update({
            "desc": row.get("description", ""),
            "status": cts.Status.STOPPED if status == "UNKNOWN" else status,
            "plan": {
                "days_of_week": sch_ctx["days_of_week"],
                "retry": sch_ctx["retry"],
                "interval": sch_ctx["success_interval"],
                "retry_interval": sch_ctx["retry_interval"],
                "start_mode": sch_ctx["start"],
                "start_time": dt.datetime.utcfromtimestamp(start_timestamp),
                "repetition": sch_ctx["repetition"],
                "sequential_run": bool(sch_ctx.get("sequential_run")),
            },
            "ctx": bson.binary.Binary(cPickle.dumps(task_ctx)),
            "noti": task["notifications"],
            "type": task["type"],
            "pr": int(priority),
            "req": {
                "plat": task["req"].get("plat"),
                "cpu": task["req"].get("cpu"),
                "host": task["req"].get("host"),
                "ram": task["req"].get("ram", 0),
                "disk": execution_space,
            },
            "time": time,
        })
        fields_to_unset = {
            "context": "",
            "description": "",
            "task": "",
        }

        scheduler_collection.update({"_id": row["_id"]}, {"$set": fields_to_set, "$unset": fields_to_unset})

    def main(self):
        scheduler_collection = me.connection.get_db()["scheduler"]
        progress = console.ProgressBar("Updating schedulers' fields", scheduler_collection.count())
        for i, row in enumerate(scheduler_collection.find({})):
            try:
                self._upgrage_scheduler(row, scheduler_collection)
            except:
                colorizer = console.AnsiColorizer()
                print(colorizer.colorize("\nError in scheduler:\n{}".format(row), "red"))
                raise
            progress.update(i)
        progress.finish()
