import Queue
import cPickle
import operator
import itertools
import bson.binary
import datetime as dt
import multiprocessing.pool
import threading as th

import pymongo
from mongoengine import connection

from sandbox.common import utils
from sandbox.common import console
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from . import base
from . import helpers


class UpdateTime(base.UpgradeStep):
    """ Adds `time.created` based on `time.updated` attribute, while the last one is calculated based on history. """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            tid, ctime, mtime = data
            collection.update(
                {'_id': tid},
                {'$set': {'time.ct': ctime, 'time.up': mtime}}
            )

    def _process(self, collection, pre):
        query = {'time.ct': {'$exists': False}}
        if pre:
            query['exc.st'] = {'$in': ['FINISHED', 'FAILURE', 'DELETED']}
        with console.LongOperation('Counting tasks amount to be updated'):
            amount = collection.find(query).count()

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)
        for i, row in enumerate(collection.find(query, {'_id': 1, 'time.up': 1, 'exc.hst.at': 1})):
            ctime = row['time']['up']
            mtime = max(h.get('at', ctime) for h in row['exc']['hst']) if row['exc']['hst'] else ctime
            tasks.append((row['_id'], ctime, mtime))
            pbar.update(i)
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for i, row in enumerate(tasks):
            queue.put(row)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def pre(self):
        collection = connection.get_db()['task']
        with console.LongOperation('Creating new index'):
            collection.create_index('time.up')
        self._process(collection, True)

    def main(self):
        self._process(connection.get_db()['task'], False)


class UpdatePriority(base.UpgradeStep):
    """
    Changes task integer priority in range -100, 100 to class-subclass based integer priority
    """
    THREADS = 20
    MAX_QUEUE_SIZE = 200

    class Priority(object):
        """
        Task priority
        """

        class Class(utils.Enum):
            USER = 2
            SERVICE = 1
            BACKGROUND = 0

        class Subclass(utils.Enum):
            HIGH = 2
            NORMAL = 1
            LOW = 0

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            tid, heavy_prio, light_prio = data
            collection.update(
                {'_id': tid},
                {'$set': {'pr': heavy_prio * 10 + light_prio}}
            )

    def _process(self, collection, pre):
        prio_map = list(itertools.product(
            sorted(self.Priority.Class, reverse=True),
            sorted(self.Priority.Subclass, reverse=True)
        ))
        prio_cache = []
        for prio in xrange(-100, 101):
            for j, x in enumerate(xrange(100, -100, -23)):
                if prio >= x:
                    prio_cache.append(prio_map[j])
                    break
            else:
                prio_cache.append((self.Priority.Class.BACKGROUND, self.Priority.Subclass.LOW))

        query = {'pr': {'$exists': False}}
        if pre:
            query['exc.st'] = {'$in': ['FINISHED', 'FAILURE', 'DELETED']}
        with console.LongOperation('Counting tasks amount to be updated'):
            amount = collection.find(query).count()

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)
        for i, row in enumerate(collection.find(query, {'_id': 1, 'prio': 1})):
            tasks.append((row["_id"],) + prio_cache[min(max(-100, row["prio"]), 100) + 100])
            pbar.update(i)
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for i, row in enumerate(tasks):
            queue.put(row)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def pre(self):
        collection = connection.get_db()['task']
        self._process(collection, True)

    def main(self):
        self._process(connection.get_db()['task'], False)

    def post(self):
        helpers.drop_field(connection.get_db()['task'], 'prio')


class Audit(base.UpgradeStep):
    """
    Move task history notes to 'Audit' collection
    """

    THREADS = 20
    MAX_QUEUE_SIZE = 200
    CHUNK_SIZE = 10000

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            fields = ('ti', 'dt', 'ct', 'a', 'h')
            collection.insert(dict(zip(fields, row)) for row in data)

    def _process(self, collection, pre=False):
        query = {'exc.hst': {'$exists': True}}
        audit_collection = connection.get_db()['audit']

        if not pre:
            applied = set()
            with console.LongOperation("Fetching audit log IDs") as op:
                count = 0
                audit_query = {}
                chunk_size = self.CHUNK_SIZE * 100
                while True:
                    chunk = list(
                        (item['_id'], item["ti"])
                        for item in audit_collection.find(
                            audit_query, {"ti": 1}
                        ).sort('_id', pymongo.ASCENDING).limit(chunk_size)
                    )
                    applied |= set(item[1] for item in chunk)
                    count += len(chunk)
                    op.intermediate("Fetched chunk of {} items. Totally fetched {} record, unique records: {}".format(
                        len(chunk), count, len(applied)
                    ))
                    if len(chunk) < chunk_size:
                        break
                    audit_query = {'_id': {'$gt': chunk[-1][0]}}

            with console.LongOperation("Fetching tasks IDs"):
                tids = set(item.get("_id") for item in collection.find({}, {"_id": 1}))
            with console.LongOperation("Calculating ids of documents to update") as op:
                ids = list(tids - applied)
                op.intermediate("{} not upgraded documents detected".format(len(ids)))
            amount = len(ids)
        else:
            query['exc.st'] = {"$in": ['FINISHED', 'FAILURE', 'DELETED']}
            amount = collection.find(query).count()
            ids = []

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)

        for x in xrange(len(ids) / self.CHUNK_SIZE + 1):
            if ids:
                query["_id"] = {"$in": ids[x * self.CHUNK_SIZE:(x + 1) * self.CHUNK_SIZE]}
            for i, row in enumerate(collection.find(query, {'_id': 1, 'exc.hst': 1})):
                task_id = row["_id"]
                tasks.extend(
                    (task_id, h.get('at'), h.get('cont'), h.get('auth'), h.get('src'))
                    for h in row['exc']['hst']
                )
                pbar.update(i * (x * self.CHUNK_SIZE + 1))
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, audit_collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for x in xrange(len(tasks) / self.CHUNK_SIZE):
            chunk = tasks[x * self.CHUNK_SIZE:(x + 1) * self.CHUNK_SIZE]
            queue.put(chunk)
            pbar.update(x * self.CHUNK_SIZE)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def pre(self):
        self._process(connection.get_db()['task'], True)

    def main(self):
        self._process(connection.get_db()['task'], False)

    def post(self):
        helpers.drop_field(connection.get_db()['task'], 'exc.hst')


class UpdateStatuses(base.UpgradeStep):
    """ Convert task statuses. """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            _id, st = data
            collection.update(
                {'_id': _id},
                {'$set': {'exc.st': st}}
            )

    def _process(self, collection, statuses, wait_time_tids=()):
        query = {'exc.st': {'$in': statuses}}
        with console.LongOperation('Counting tasks amount to be updated'):
            amount = collection.find(query).count()

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)
        for i, row in enumerate(collection.find(query, {'_id': 1, 'exc.st': 1})):
            tasks.append([row['_id'], row['exc']['st']])
            pbar.update(i)
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for i, row in enumerate(tasks):
            if row[0] in wait_time_tids and row[1] == 'WAIT_CHILD':
                row[1] = ctt.Status.WAIT_TIME
            else:
                row[1] = ctt.Status.new_status(row[1])
            queue.put(row)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def main(self):
        wait_time_tids = [row['_id'] for row in connection.get_db()['time_trigger'].find({}, {'_id': 1})]
        self._process(connection.get_db()['task'], ['WAIT_DEPS', 'WAIT_CHILD'], wait_time_tids)

    def post(self):
        self._process(connection.get_db()['task'], ['FINISHED', 'NOT_READY', 'UNKNOWN'])


class UpdateAuditStatuses(base.UpgradeStep):
    """ Convert statuses in task audit. """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            _id, st = data
            collection.update(
                {'_id': _id},
                {'$set': {'st': ctt.Status.new_status(st)}}
            )

    def _process(self, collection):
        query = {'st': {'$in': ['FINISHED', 'NOT_READY', 'WAIT_DEPS', 'WAIT_CHILD', 'UNKNOWN']}}
        with console.LongOperation('Counting tasks amount to be updated'):
            amount = collection.find(query).count()

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)
        for i, row in enumerate(collection.find(query, {'_id': 1, 'st': 1})):
            tasks.append((row['_id'], row['st']))
            pbar.update(i)
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for i, row in enumerate(tasks):
            queue.put(row)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def main(self):
        pass

    def post(self):
        self._process(connection.get_db()['audit'])


class UpdateTriggerStatuses(base.UpgradeStep):
    """ Convert statuses in task status triggers. """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, collection):
        while True:
            data = queue.get()
            if not data:
                break
            _id, statuses = data
            collection.update(
                {'_id': _id},
                {'$set': {'statuses': [ctt.Status.new_status(st) for st in statuses]}}
            )

    def _process(self, collection):
        query = {'statuses': {'$in': ['FINISHED', 'NOT_READY', 'WAIT_DEPS', 'WAIT_CHILD', 'UNKNOWN']}}
        with console.LongOperation('Counting tasks amount to be updated'):
            amount = collection.find(query).count()

        tasks = []
        pbar = console.ProgressBar('Fetching data', amount)
        for i, row in enumerate(collection.find(query, {'_id': 1, 'statuses': 1})):
            tasks.append((row['_id'], row['statuses']))
            pbar.update(i)
        pbar.finish()

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, collection)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(tasks))
        for i, row in enumerate(tasks):
            queue.put(row)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def main(self):
        self._process(connection.get_db()['task_status_trigger'])


class ExtractAuthorFieldFromCtx(base.UpgradeStep):
    """
    Move `author`, `current_action` and 'disk_usage' fields from task context to separate field in task document
    """
    THREADS = 20
    MAX_QUEUE_SIZE = 200
    CHUNK_SIZE = 1000

    @classmethod
    def worker(cls, queue, collection, broken_ctx):
        while True:
            data = queue.get()
            if not data:
                break
            for item in data:
                try:
                    tid, ctx, ldu = item[0], cPickle.loads(str(item[1].decode("utf-8"))), item[2]["disc"]
                except Exception:
                    broken_ctx.append(str(item[0]))
                    continue
                mdu, author = max(ctx.pop("__disk_space_usage", [0])), ctx.pop("__author", "sandbox")
                collection.update(
                    {"_id": tid},
                    {
                        "$set": {
                            "author": author,
                            "exc.du.m": mdu,
                            "exc.du.l": ldu,
                            "ctx": bson.binary.Binary(cPickle.dumps(ctx))
                        },
                        "$unset": {"exc.disc": True}
                    }
                )

    def _process(self, collection):
        query = {'author': {'$exists': False}}
        with console.LongOperation("Calculating amount of documents to update") as op:
            amount = collection.find(query).count()
            op.intermediate("Detected {} documents to update".format(amount))
        if not amount:
            return

        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        broken_ctx = []
        pool = [
            th.Thread(target=self.worker, args=(queue, collection, broken_ctx)) for _ in xrange(self.THREADS)
        ]
        map(th.Thread.start, pool)

        updated = 0
        id_bound = None
        pbar = console.ProgressBar('Updating data', amount)
        total = []
        while True:
            if id_bound is not None:
                query["_id"] = {"$lt": id_bound}
            chunk = map(
                operator.itemgetter("_id", "ctx", "exc"),
                collection.find(
                    query,
                    {"_id": 1, "ctx": 1, "exc.disc": 1}
                ).sort("_id", pymongo.DESCENDING).limit(self.CHUNK_SIZE)
            )
            if not chunk:
                break
            total.extend(chunk)
            updated += len(chunk)
            queue.put(chunk)
            pbar.update(updated)
            id_bound = chunk[-1][0]
        pbar.finish()

        with console.LongOperation("Waiting for workers"):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

        if broken_ctx:
            print(
                console.AnsiColorizer().colorize("Task ids with broken ctx:\n{}".format(" ".join(broken_ctx)), "red")
            )

    def post(self):
        with console.LongOperation("Updating schedulers collection"):
            collection = connection.get_db()["scheduler"]
            for doc in collection.find({"author": {"$exists": False}}, {"_id": 1, "context": 1}):
                ctx = cPickle.loads(doc["context"])
                author = ctx.pop("author", "sandbox")
                collection.update(
                    {"_id": doc["_id"]},
                    {
                        "$set": {
                            "author": author,
                            "context": bson.binary.Binary(cPickle.dumps(ctx))
                        }
                    }
                )
        self._process(connection.get_db()['task'])

    def main(self):
        pass


class UpdateOldReleasedTasks(base.UpgradeStep):
    """ Switch previously released tasks status from SUCCESS to RELEASED """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, coll_task, coll_audit):
        while True:
            task_id = queue.get()
            if task_id is None:
                break
            coll_task.update(
                {'_id': task_id},
                {'$set': {'exc.st': 'RELEASED'}}
            )
            coll_audit.insert({
                'ti': task_id,
                'st': 'RELEASED',
                'dt': dt.datetime.utcnow(),
                'sc': 'sys'
            })

    def _process(self, coll_task, coll_res, coll_audit):
        with console.LongOperation('Counting tasks to be updated'):
            task_ids = set(
                row['_id']
                for row in coll_task.find(
                    {
                        '_id': {
                            '$in': list(set(
                                row['tid']
                                for row in coll_res.find({'state': 'READY', 'attrs.k': 'released'}, {'tid': 1})
                            ))
                        },
                        'exc.st': 'SUCCESS'
                    },
                    {'_id': 1}
                )
            )
        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, coll_task, coll_audit)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar('Updating data', len(task_ids))
        for i, task_id in enumerate(task_ids):
            queue.put(task_id)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation('Waiting for workers.'):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def main(self):
        pass

    def post(self):
        db = connection.get_db()
        self._process(db['task'], db['resource'], db['audit'])


class UpdateDiskUsageMax(base.UpgradeStep):
    """ Increase disk_usage.max up to 1024 times (<< 10) """

    THREADS = 20
    MAX_QUEUE_SIZE = 200

    @classmethod
    def worker(cls, queue, coll_task):
        while True:
            du = queue.get()
            if du is None:
                break
            coll_task.update(
                {"_id": du[0]},
                {"$set": {"exc.du.m": du[1] << 10}}
            )

    def _process(self, coll_task, last_task_id):
        with console.LongOperation("Counting tasks to be updated"):
            dus = {
                row["_id"]: row["exc"]["du"]["m"]
                for row in coll_task.find(
                    {"_id": {"$lte": last_task_id}, "exc.du.m": {"$gt": 0}},
                    {"_id": 1, "exc.du.m": 1}
                )
            }
        queue = Queue.Queue(self.MAX_QUEUE_SIZE)
        pool = [th.Thread(target=self.worker, args=(queue, coll_task)) for _ in xrange(self.THREADS)]
        map(th.Thread.start, pool)

        pbar = console.ProgressBar("Updating data", len(dus))
        for i, du in enumerate(dus.iteritems()):
            queue.put(du)
            pbar.update(i)
        pbar.finish()

        with console.LongOperation("Waiting for workers."):
            map(queue.put, [None] * len(pool))
            map(th.Thread.join, pool)

    def main(self):
        db = connection.get_db()
        last_task_id = db["task"].find({}, {"_id": 1}).sort("_id", pymongo.DESCENDING).limit(1)[0]["_id"]
        db["settings"].update({}, {"$set": {"_last_task_id": last_task_id}})

    def post(self):
        db = connection.get_db()
        last_task_id = db["settings"].find({}, {"_last_task_id": 1})[0]["_last_task_id"]
        self._process(db["task"], last_task_id)
        db["settings"].update({}, {"$unset": {"_last_task_id": 1}})


class ExtractHostsChooserFieldsFromCtx(base.UpgradeStep):
    """
    Move `__hosts_chooser_hosts`, `__hosts_chooser_os` and '__hosts_chooser_cpu_model' fields from task context
    to separate fields in task document
    """
    THREADS = 20
    MAX_QUEUE_SIZE = 200
    CHUNK_SIZE = 1000

    @classmethod
    def _process_tasks_worker(cls, queue, collection, broken_tasks_ids):
        tasks_chunk = queue.get()
        while tasks_chunk:
            for task in tasks_chunk:
                try:
                    context = cPickle.loads(task["ctx"])
                except Exception:
                    broken_tasks_ids.append(task["_id"])
                    context = {}
                collection.update(
                    {"_id": task["_id"]},
                    {
                        "$set": {
                            "req.host": context.get("__hosts_chooser_hosts", ""),
                        }
                    }
                )
            tasks_chunk = queue.get()

    def _process_tasks(self, collection):
        query = {"req.host": {"$exists": False}}

        with console.LongOperation("Updating tasks collection") as op:
            amount = collection.find(query).count()
            op.intermediate("Detected {} documents to update".format(amount))
        if not amount:
            return

        pool = multiprocessing.pool.ThreadPool(self.THREADS)
        queue = multiprocessing.Queue(self.MAX_QUEUE_SIZE)
        broken_tasks_ids = []
        for _ in xrange(self.THREADS):
            pool.apply_async(self._process_tasks_worker, args=(queue, collection, broken_tasks_ids))

        updated = 0
        progress_bar = console.ProgressBar("Updating data", amount)
        while True:
            tasks_chunk = list(collection.find(query).sort("_id", pymongo.DESCENDING).limit(self.CHUNK_SIZE))
            if not tasks_chunk:
                break
            updated += len(tasks_chunk)
            queue.put(tasks_chunk)
            query["_id"] = {"$lt": tasks_chunk[-1]["_id"]}
            progress_bar.update(updated)
        progress_bar.finish()

        with console.LongOperation("Waiting for workers"):
            map(queue.put, [None] * self.THREADS)
            pool.close()
            pool.join()

        if broken_tasks_ids:
            colorizer = console.AnsiColorizer()
            broken_tasks_ids = " ".join(map(str, broken_tasks_ids))
            print(colorizer.colorize("Broken tasks ids:\n" + broken_tasks_ids, "red"))

    @staticmethod
    def _process_scheduler(collection):
        with console.LongOperation("Updating schedulers collection"):
            for doc in collection.find({'task.req': {'$exists': False}}):
                task = doc["task"]
                task_context = cPickle.loads(task["context"])
                requirements = {
                    req_key: task_context.pop(context_key)
                    for req_key, context_key in (
                        ("plat", "__hosts_chooser_os"),
                        ("cpu", "__hosts_chooser_cpu_model"),
                        ("host", "__hosts_chooser_hosts"),
                    ) if context_key in task_context
                }
                collection.update(
                    {"_id": doc["_id"]},
                    {
                        "$set": {
                            "task.req": requirements,
                            "task.context": bson.binary.Binary(cPickle.dumps(task_context)),
                        }
                    }
                )

    def post(self):
        db = connection.get_db()
        self._process_scheduler(db["scheduler"])
        self._process_tasks(db["task"])

    def main(self):
        pass


class DnsFieldAsEnum(base.UpgradeStep):
    """ Convert `dns` field to enum. Old `dns` move to `dns64`. """

    THREADS = 20
    MAX_QUEUE_SIZE = 200
    CHUNK_SIZE = 1000

    @classmethod
    def _worker(cls, queue, collection):
        chunk = queue.get()
        while chunk:
            min_id, max_id = chunk
            for dns64 in (False, True):
                collection.update(
                    {"_id": {"$gte": min_id, "$lte": max_id}, "req.dns": dns64},
                    {"$set": {"req.dns": ctm.DnsType.DEFAULT, "req.dns64": dns64}},
                    multi=True,
                )
            chunk = queue.get()

    def _process_tasks(self, start_time=None):
        collection = connection.get_db()["task"]
        min_id = 0
        if start_time:
            for task in collection.find({"exc.time.st": {"$gt": start_time}}).sort("_id").limit(1):
                min_id = task["_id"]
        for task in collection.find().sort("_id", pymongo.DESCENDING).limit(1):
            max_id = task["_id"]

        pool = multiprocessing.pool.ThreadPool(self.THREADS)
        queue = multiprocessing.Queue(self.MAX_QUEUE_SIZE)
        for _ in xrange(self.THREADS):
            pool.apply_async(self._worker, args=(queue, collection))

        progress_bar = console.ProgressBar("Updating tasks collection", max_id - min_id + 1)
        processed = 0
        chunk_finish_id = max_id
        while chunk_finish_id >= min_id:
            chunk_start_id = max(chunk_finish_id - self.CHUNK_SIZE + 1, min_id)
            queue.put((chunk_start_id, chunk_finish_id))
            processed += chunk_finish_id - chunk_start_id + 1
            progress_bar.update(processed)
            chunk_finish_id = chunk_start_id - 1
        progress_bar.finish()

        with console.LongOperation("Waiting for workers"):
            map(queue.put, [None] * self.THREADS)
            pool.close()
            pool.join()

    def main(self):
        self._process_tasks(start_time=dt.datetime.utcnow() - dt.timedelta(days=1))

    def post(self):
        self._process_tasks()


class FixNotificationsField(base.UpgradeStep):
    """ notifications => noti """

    THREADS = 20
    MAX_QUEUE_SIZE = 200
    CHUNK_SIZE = 1000

    @classmethod
    def _worker(cls, queue, collection):
        tasks_chunk = queue.get()
        while tasks_chunk:
            for task in tasks_chunk:
                collection.update(
                    {"_id": task["_id"]},
                    {
                        "$unset": {"notifications": 1},
                        "$set": {"noti": task["notifications"]},
                    }
                )
            tasks_chunk = queue.get()

    def post(self):
        collection = connection.get_db()["task"]
        query = {"notifications": {"$exists": True}}

        with console.LongOperation("Updating tasks collection") as op:
            amount = collection.find(query).count()
            op.intermediate("Detected {} documents to update".format(amount))
        if not amount:
            return

        pool = multiprocessing.pool.ThreadPool(self.THREADS)
        queue = multiprocessing.Queue(self.MAX_QUEUE_SIZE)
        for _ in xrange(self.THREADS):
            pool.apply_async(self._worker, args=(queue, collection))

        updated = 0
        progress_bar = console.ProgressBar("Updating data", amount)
        while True:
            tasks_chunk = list(collection.find(query).sort("_id", pymongo.DESCENDING).limit(self.CHUNK_SIZE))
            if not tasks_chunk:
                break
            updated += len(tasks_chunk)
            queue.put(tasks_chunk)
            query["_id"] = {"$lt": tasks_chunk[-1]["_id"]}
            progress_bar.update(updated)
        progress_bar.finish()

        with console.LongOperation("Waiting for workers"):
            map(queue.put, [None] * self.THREADS)
            pool.close()
            pool.join()

    def main(self):
        pass
