import re

from pymongo import errors
from mongoengine import connection

from sandbox.common import console

from . import base


class UseLoginAsID(base.UpgradeStep):
    """
    Use login as _id for User records
    """

    def pre(self):
        collection = connection.get_db()['user']
        with console.LongOperation('Create new scheme records based on old records') as op:
            records = []
            # collect all existent
            for rec in collection.find():
                if 'login' in rec:
                    records.append((rec['login'], rec.get('super_user'), rec.get('allowed_api_methods')))
            # insert as new
            for login, super_user, allowed_api_methods in records:
                rec = {'_id': login}
                if super_user:
                    rec['super_user'] = super_user
                if allowed_api_methods:
                    rec['allowed_api_methods'] = allowed_api_methods
                collection.insert(rec)
            #
            op.intermediate('Updated {} records'.format(len(records)))

        with console.LongOperation("Dropping 'login' index"):
            try:
                collection.drop_index('login_1')
            except errors.OperationFailure:
                pass

    def main(self):
        pass

    def post(self):
        collection = connection.get_db()['user']

        with console.LongOperation('Drop old-style records'):
            collection.remove({'login': {'$exists': True}})

        with console.LongOperation("Dropping 'login' index"):
            try:
                collection.drop_index('login_1')
            except errors.OperationFailure:
                pass


class DropSessionCollection(base.UpgradeStep):
    """
    Use session collection is not needed any more.
    """

    def pre(self):
        db = connection.get_db()
        db.drop_collection('session')

    def main(self):
        pass

    def post(self):
        pass


class RenameGroups(base.UpgradeStep):
    """ Rename groups and owners with spaces in their names. """

    def main(self):
        pass

    def post(self):
        db = connection.get_db()
        rex = re.compile(r"[^A-Z0-9_-]")
        rex_chk = re.compile(r"^[A-Z0-9_-]+$")
        with console.LongOperation("Determining groups to rename"):
            groups = {
                _["_id"]: _
                for _ in db["group"].find()
                if not rex_chk.match(_["_id"])
            }
            for _ in groups.itervalues():
                _["_id"] = rex.sub("_", _["_id"])

        with console.LongOperation("Updating groups"):
            for n in sorted(groups):
                g = groups[n]
                db["group"].insert(g)
            db["group"].remove({"_id": {"$in": list(groups)}})

        with console.LongOperation("Updating schedulers collections") as op:
            for n in sorted(groups):
                nr = groups[n]["_id"]
                up = db["scheduler"].update(
                    {"owner": n},
                    {"$set": {"owner": nr}},
                    multi=True
                )["nModified"]
                op.intermediate("{} -> {} {} schedulers updated".format(n, nr, up))

        with console.LongOperation("Updating tasks collections") as op:
            for n in sorted(groups):
                nr = groups[n]["_id"]
                up = db["task"].update(
                    {"owner": n},
                    {"$set": {"owner": nr}},
                    multi=True
                )["nModified"]
                op.intermediate("{} -> {} {} tasks updated".format(n, nr, up))

        with console.LongOperation("Updating resources collections") as op:
            for n in sorted(groups):
                nr = groups[n]["_id"]
                up = db["resource"].update(
                    {"owner": n},
                    {"$set": {"owner": nr}},
                    multi=True
                )["nModified"]
                op.intermediate("{} -> {} {} resources updated".format(n, nr, up))


class MultiSourceGroups(base.UpgradeStep):
    """ A group can have a list of SyncSources """

    def main(self):
        pass

    def post(self):
        collection = connection.get_db()["group"]
        with console.LongOperation("Listifying group sync sources"):
            for grp in collection.find({"sync": {"$exists": True}, "sources": {"$exists": False}}):
                collection.update(
                    {"_id": grp["_id"]},
                    {"$set": {"sources": [grp["sync"]]}}
                )

        with console.LongOperation("Erasing old group sync sources"):
            for grp in collection.find({"sync": {"$exists": True}}):
                collection.update(
                    {"_id": grp["_id"]},
                    {"$unset": {"sync": ""}}
                )
