from __future__ import absolute_import

import sys
import logging
import xmlrpclib
import cPickle as pickle

from sandbox import common

from . import task
from . import resource
from . import client
from . import release
from . import notification


logger = logging.getLogger(__name__)


class ManagerDispatchWrapper(object):
    def __init__(self, manager):
        self.manager = manager
        self.name = getattr(manager, 'remoteName', None)

    def __getattr__(self, attr):
        def wrapper(*args, **kwargs):
            settings = common.config.Registry()
            func_name = '%s_%s' % (self.name, attr)
            logger.debug("remote manager call %s.%s(%r, %r)", self.name, attr, args, kwargs)
            remote = common.proxy.ThreadLocalCachableServerProxy(settings.client.xmlrpc_url)
            args = xmlrpclib.Binary(pickle.dumps(args))
            kwargs = xmlrpclib.Binary(pickle.dumps(kwargs))
            result = remote.managerCall(func_name, args, kwargs)
            return pickle.loads(result.data)

        if not self.name:
            raise UnboundLocalError('{!r} is not supposed to be called remotely.'.format(self.manager))
        setattr(self, attr, wrapper)
        return wrapper


(
    task_manager,
    resource_manager,
    client_manager,
    release_manager,
    notification_manager,
) = tuple(
    ManagerDispatchWrapper(cls)
    for cls in (
        task.TaskManager,
        resource.ResourceManager,
        client.ClientManager,
        release.ReleaseManager,
        notification.NotificationManager,
    )
)


# noinspection PyUnreachableCode
if False:
    # Enable code navigation in IDE
    task_manager = task.TaskManager
    resource_manager = resource.ResourceManager
    client_manager = client.ClientManager
    release_manager = release.ReleaseManager
    notification_manager = notification.NotificationManager


def use_locally(detain_init=False):
    """
    Switch locally declared manager objects from remote mode (which is default) to local.
    This means that each call to the appropriate manager object will not automatically
    enveloped into XML-RPC request but will executed directly.
    This also establishes database connection implicitly.
    """
    import sandbox.yasandbox.database.mapping
    sandbox.yasandbox.database.mapping.ensure_connection()

    detained = []
    orig_init = None
    empty_init = lambda _: None
    for name, symbol in globals().iteritems():
        if not isinstance(symbol, ManagerDispatchWrapper):
            continue
        if detain_init:
            orig_init = symbol.manager.__init__
            symbol.manager.__init__ = empty_init
        instance = symbol.manager()
        if detain_init:
            detained.append((orig_init, instance))
        globals()[name] = instance

    self = sys.modules[__name__]
    if detained:
        self.initialize_locally = lambda: [init(instance) for init, instance in detained]
    return self


def initialize_locally():
    """
    This function will be overriden in case of `use_locally()` called with detained initialization.
    In this case this one will contain all the collected initialization methods (constructors) of all
    managed object, which will be called on this method call.
    """
    raise UnboundLocalError('Initialization was not detained.')
