import logging
import calendar
import itertools

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


class ReleaseManager(object):
    Status = mapping.Task.Release.Status

    logger = logging.getLogger(__name__)

    @classmethod
    def add_release_to_db(
        cls,
        task_id, author, status,
        message_subject, message_body=None, changelog=None,
        creation_time=None,
    ):
        cls.logger.info("Adding release record for task #%r", task_id)
        mapping.Task.objects(id=task_id).update_one(
            set__release=mapping.Task.Release(
                creation_time=creation_time,
                author=author,
                status=status,
                message=mapping.Task.Release.Message(
                    subject=message_subject,
                    body=message_body
                ),
                changelog=changelog,
            )
        )

        # In additional to release record, forcedly backup released resources
        resources = list(mapping.Resource.objects(
            task_id=task_id, state=ctr.State.READY, attributes__key="released",
            hosts_states__host__nin=list(common.config.Registry().server.storage_hosts)
        ))
        cls.logger.debug("There are %d task's #%r released resources needs forced backup.", len(resources), task_id)
        if not resources:
            return

    @classmethod
    def release_task(
        cls,
        task_id, author, status=None,
        message_subject=None, message_body=None, changelog=None,
        addresses_to=None, addresses_cc=None,
        additional_parameters=None,
        request=None
    ):
        """
        Release task with id `task_id`

        :param task_id: id of task to release
        :param author: author of release
        :param status: release status (stable, testing etc)
        :param message_subject: subject of release message
        :param message_body: body of release message
        :param changelog: description of changes in release
        :param addresses_to: recipients of release message
        :param addresses_cc: additional recipients of release message
        :param additional_parameters: parameters to pass in on_release method
        :param request: sandbox-request object to pass in audit method
        :return: id of released task
        :rtype: int
        """
        from sandbox.yasandbox.manager import task_manager

        settings = common.config.Registry()

        task = task_manager.load(task_id)

        if task is None:
            raise common.errors.ReleaseError("Task #{} not found".format(task_id))

        if task.status == ctt.Status.RELEASING:
            if request and mapping.Audit.objects(
                task_id=task_id,
                status=ctt.Status.RELEASING,
                request_id=request.id
            ).first():
                return task_id
            raise common.errors.ReleaseError("Releasing of task #{} is already in progress".format(task_id))
        if not task.Status.can_switch(task.status, ctt.Status.RELEASING):
            raise common.errors.ReleaseError("Task #{} is not finished successfully".format(task_id))

        if status not in cls.Status:
            raise common.errors.ReleaseError("Invalid release status {!r}".format(status))

        task_resources = [r for r in task.list_resources() if r.type.releasable]
        if not task_resources:
            raise common.errors.ReleaseError("No resources for release")

        bad_resources = [r for r in task_resources if r.state in (ctr.State.BROKEN, ctr.State.DELETED)]
        if bad_resources:
            raise common.errors.ReleaseError(
                "Cannot release task #{}: the following resources are broken/deleted: {}".format(
                    task_id, ", ".join((str(res.id) for res in bad_resources))
                )
            )

        for resource in task_resources if settings.server.auth.enabled else []:
            user = controller.User.get(author)
            releasers = resource.type.releasers
            if not (user and user.super_user) and releasers and author not in controller.Group.unpack(releasers):
                raise common.errors.ReleaseError(
                    "Access denied. User {} cannot release resource #{}, type {}. "
                    "To fix this, you should add this user to 'releasers' property of this resource".format(
                        author, resource.id, resource.type
                    )
                )
        cls.logger.info("Creating release for #%s author %s with status %s", task_id, author, status)

        if not addresses_to:
            addresses_to = task.release_to or []
        if not addresses_cc:
            addresses_cc = []
        for resource in task_resources:
            if resource.type.release_subscribers:
                addresses_cc.extend(resource.type.release_subscribers)

        additional_parameters = additional_parameters or {}
        additional_parameters.setdefault("release_status", status)
        additional_parameters.setdefault("release_subject", message_subject)
        additional_parameters.setdefault("release_changelog_entry", changelog)
        additional_parameters.setdefault("release_comments", message_body)
        additional_parameters["releaser"] = author
        additional_parameters["email_notifications"] = {"to": addresses_to, "cc": addresses_cc}
        task.release_params = additional_parameters
        task_manager.update(task)
        try:
            task.set_status(ctt.Status.RELEASING, request=request, set_last_action_user=True)
        except common.errors.UpdateConflict:
            if request and mapping.Audit.objects(
                task_id=task_id,
                status=ctt.Status.RELEASING,
                request_id=request.id
            ).first():
                return task_id
            raise
        return task_id

    @staticmethod
    def list_releases(
        resource_type="", release_status="",
        task_id=0, arch="",
        resource_as_dict=False,
        order_by=None,
        creation_ts_bound=None,
        include_broken=False,
        limit=0, offset=0,
    ):
        """
        build filtered list of releases

        paginator using inspect for default parameter type - so using 0 and "" insdead of None
        """
        from sandbox.yasandbox.manager import resource_manager

        # fix params
        task_id = mapping.ObjectId(task_id) if task_id else None
        order_by = order_by or "-id"

        released_resources = mapping.Resource.released_resources(
            resource_type=resource_type,
            release_status=release_status,
            include_broken=include_broken,
            task_id=task_id,
            arch=arch,
            limit=limit,
            offset=offset
        )

        rids = itertools.chain.from_iterable(ids for _, ids in released_resources)
        all_resources = {r.id: r for r in resource_manager.fast_load_list(rids)}

        # map resources to tasks
        task_id_resource_map = {
            tid: filter(None, (all_resources.get(rid) for rid in sorted(resources)))
            for tid, resources in released_resources
        }

        # load released tasks
        cond = {
            "id__in": [x[0] for x in released_resources],
            "execution__status": ctt.Status.RELEASED,
            "release__exists": True,
        }
        if creation_ts_bound and len(creation_ts_bound) == 2:
            cond["release__creation_time__gte"] = creation_ts_bound[0]
            cond["release__creation_time__lte"] = creation_ts_bound[1]
        released_tasks = mapping.Task.objects(**cond).order_by(order_by).scalar(
            "id", "release", "type"
        )
        if order_by[1:] == "id":
            released_tasks = released_tasks.hint([("_id", 1)])

        # calculate result
        releases = []
        for tid, release, _ in released_tasks:
            resources = task_id_resource_map[tid]
            if resource_as_dict:
                resources = [r.to_dict() for r in resources]
            release_obj = {
                "id": tid,
                "task_id": tid,
                "timestamp": calendar.timegm(release.creation_time.timetuple()),
                "status": release.status,
                "author": release.author,
                "subject": release.message.subject,
                "comment": release.message.body,
                "changelog": list(release.changelog),
                "resources": resources,
            }
            releases.append(release_obj)
        return releases

    @staticmethod
    def count_releases(
        resource_type="", release_status="",
        task_id=0, arch="",
        limit=0, offset=0,
    ):
        """
        paginator using inspect for default parameter type - so using 0 and "" insdead of None
        """
        # fix params
        task_id = mapping.ObjectId(task_id) if task_id else None

        # load releades resources
        return len(mapping.Resource.released_resources(
            resource_type=resource_type,
            release_status=release_status,
            task_id=task_id,
            arch=arch,
            limit=limit,
            offset=offset
        ))
