# coding: utf-8

import time
import collections
import datetime as dt

from sandbox import common
import sandbox.common.types.database as ctd
import sandbox.common.types.resource as ctr

from sandbox import sdk2

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


class ResourceManager(object):
    """
        Менеджер для работы с ресурсами.
    """

    remoteName = 'resource'
    # Class-wide logging facility
    logger = None
    Model = mapping.Resource

    def __init__(self):
        self.Model.ensure_indexes()
        self.__class__.logger = common.log.get_core_log("resources_audit")

    @classmethod
    def create(cls, resource):
        author, owner = next(iter(mapping.Task.objects(id=resource.task_id).scalar("author", "owner")), (None, None))
        if (
            owner and
            owner != resource.owner and
            not controller.user_has_permission(author or resource.owner, (resource.owner,))
        ):
            raise ValueError(
                "User {!r} not allowed to create resource owned by {!r} for task #{}".format(
                    author, resource.owner, resource.task_id
                )
            )

        # create db obj from resource
        if resource.type.auto_backup:
            resource.attrs["backup_task"] = True
        resource.attrs = {cls._cast(k).strip(): cls._cast(v).strip() for k, v in resource.attrs.iteritems()}
        obj = resource._create()
        # update resource
        resource.id = obj.id
        cls.logger.info(
            "Created resource #%s of type '%s' in state '%s' with attrs: %s",
            resource.id, resource.type, resource.state, resource.attrs
        )
        return resource

    @classmethod
    def load(cls, resource_id):
        try:
            resource_id = mapping.ObjectId(resource_id)
        except (ValueError, TypeError):
            raise common.errors.TaskError("invalid resource id value '{}'".format(resource_id))
        obj = cls.Model.objects.with_id(mapping.ObjectId(resource_id))
        if not obj:
            return None
        from sandbox.yasandbox.proxy.resource import Resource
        resource = Resource._restore(obj)
        return resource

    @classmethod
    def fast_load_list(cls, ids):
        ids = map(mapping.ObjectId, ids)
        objs = cls.Model.objects.exclude('hosts_states').in_bulk(ids)
        from sandbox.yasandbox.proxy.resource import Resource
        return map(Resource._restore, map(objs.get, ids))

    @classmethod
    def update(cls, resource):
        resource.attrs = {cls._cast(k).strip(): cls._cast(v).strip() for k, v in resource.attrs.iteritems()}
        cls.logger.info(
            "Updating resource #%s in state '%s' with attrs: %r",
            resource.id, resource.state, resource.attrs
        )
        resource._update()

    @staticmethod
    def _cast(value):
        return value.encode("utf-8") if isinstance(value, unicode) else str(value)

    @classmethod
    def list_query(
        cls,
        id=0, resource_type=None, name_mask=None, owner=None, task_id=None, arch=None, date=None,
        state=None, skynet_id='', any_attrs=None, all_attrs=None, host=None, omit_failed=False,
        hidden=True, accessed=None, created=None,
    ):
        # name_mask: not supported (resource.name LIKE)
        # hidden: not supported

        if resource_type:
            if isinstance(resource_type, (list, tuple)):
                resource_type = [str(rt) for rt in resource_type]
            else:
                resource_type = str(resource_type)
        return controller.Resource.list_query(
            id=id,
            resource_type=resource_type,
            owner=owner,
            task_id=task_id,
            arch=arch,
            date=date,
            state=state,
            skynet_id=skynet_id,
            any_attrs=any_attrs,
            all_attrs=all_attrs,
            host=host,
            omit_failed=omit_failed,
            accessed=accessed,
            created=created
        )

    @classmethod
    def list(
        cls,
        resource_type='', omit_failed=False, hidden=0, owner='', limit=0, offset=0, host='',
        order_by='-id', date='', load=True
    ):
        return cls.list_task_resources(
            resource_type=resource_type, omit_failed=omit_failed, hidden=hidden,
            owner=owner, limit=limit, offset=offset, host=host, order_by=order_by,
            date=date, load=load
        )

    @classmethod
    def list_task_resources(
        cls,
        task_id=0, resource_type='', state='', omit_failed=False, hidden=True, owner='', arch='',
        attr=None, attr_name='', attr_value=None, any_attrs=None, all_attrs=None, id=0,
        name_mask='', limit=0, offset=0, skynet_id='', host='', order_by='-id', date='', load=True
    ):
        from sandbox.yasandbox.proxy.resource import Resource

        if attr:
            attr_name, attr_value = attr
        if not any_attrs and (attr_name or attr_value):
            any_attrs = {attr_name: attr_value}

        query = cls.list_query(
            id=id,
            resource_type=resource_type,
            name_mask=name_mask,
            owner=owner,
            task_id=task_id,
            arch=arch,
            date=date,
            state=state,
            skynet_id=skynet_id,
            any_attrs=any_attrs,
            all_attrs=all_attrs,
            host=host,
            omit_failed=omit_failed,
            hidden=hidden,
        )
        result = cls.Model.objects(**query).skip(int(offset))
        if limit:
            result = result.limit(int(limit))
        if order_by:
            result = result.order_by(order_by)
        return list(result.scalar('id')) if not load else map(Resource._restore, result.exclude("hosts_states"))

    @classmethod
    def last_resources(cls, resource_type=None, state=None):
        if not resource_type:
            raise ValueError("resource_type parameter should be non-empty.")
        return cls.fast_load_list(r[1] for r in cls.Model.last_resources('type', resource_type, state))

    @staticmethod
    def bulk_fields(ids, fields, safe_xmlrpc=False):
        """
        Fetch list of resources from db
        """
        from sandbox.yasandbox.proxy import resource as resource_proxy

        # probably mongo have some lag in shards request
        # so we make query second time for missed items

        ids = map(mapping.ObjectId, ids)

        def fetch_data(_ids):
            data = {}
            type_index = fields.index('type') if 'type' in fields else None
            objects = mapping.Resource.objects.in_bulk(_ids)
            for mp in objects.itervalues():
                resource = resource_proxy.Resource._restore(mp)
                values = [getattr(resource, attr) for attr in fields]
                if safe_xmlrpc and type_index is not None:
                    values[type_index] = str(values[type_index])
                _id = str(resource.id) if safe_xmlrpc else resource.id
                data[_id] = common.proxy.safe_xmlrpc_list(values) if safe_xmlrpc else values
            return data

        # collect results from db first time
        data = fetch_data(ids)

        # if some ids are missed - collect again and update results
        if len(ids) != len(data):
            data_ids = map(mapping.ObjectId, data)
            missed_ids = set(ids) - set(data_ids)
            missed_data = fetch_data(list(missed_ids))
            data.update(missed_data)

        return data

    @classmethod
    def count_task_resources(
        cls,
        task_id=0, resource_type='', state='', omit_failed=False, hidden=True, owner='', arch='',
        attr=None, attr_name='', attr_value=None, any_attrs=None, all_attrs=None, id=0,
        name_mask='', host='', date='', order_by=None
    ):

        if attr:
            attr_name, attr_value = attr
        if not any_attrs and (attr_name or attr_value):
            any_attrs = {attr_name: attr_value}

        query = cls.list_query(
            id=id,
            resource_type=resource_type,
            name_mask=name_mask,
            owner=owner,
            task_id=task_id,
            arch=arch,
            date=date,
            state=state,
            any_attrs=any_attrs,
            all_attrs=all_attrs,
            host=host,
            omit_failed=omit_failed,
            hidden=hidden,
        )
        return cls.Model.objects(**query).count()

    @classmethod
    def delete_resource(cls, resource_id, ignore_last_usage_time=False):
        """
            Удалить ресурс с идентификатором resource_id

            :param resource_id: идентификатор ресурса
            :param ignore_last_usage_time: принудительно удалить ресурс, даже если он недавно использовался
            :return: None, если ресурс успешно удалён, строка с сообщением об ошибке — в противном случае
        """
        settings = common.config.Registry()
        resource = cls.load(resource_id)
        last_usage_time = resource.last_usage_time

        if resource.attrs.get(ctr.ServiceAttributes.TTL, '') == 'inf':
            return 'Resource has ttl=inf attribute.'
        if not ignore_last_usage_time:
            cls.logger.debug('Try to delete resource %s with last usage time %s', resource.id, last_usage_time)
            delete_threshold_time = time.time() - settings.server.services.clean_resources.default_ttl * 24 * 60 * 60

            # проверяем, можем ли удалить ресурс
            if last_usage_time > delete_threshold_time:
                cls.logger.info('Cannot delete resource %s with last usage time %s', resource.id, last_usage_time)
                return 'Resource has not expired yet!'
        else:
            cls.logger.debug('Try to delete resource %s', resource.id)

        # удаляем все копии ресурса на хостах
        for host in cls.get_hosts(resource.id):
            cls.logger.info('Delete resource %s from host %s', resource.id, host)
            cls.mark_host_to_delete(resource.id, host)

        cls.logger.info('Mark resource %s as %s', resource.id, cls.Model.State.DELETED)
        # помечаем ресурс как DELETED
        cls.Model.objects(id=mapping.ObjectId(resource.id)).update(
            set__state=cls.Model.State.DELETED,
            set__time__updated=dt.datetime.utcnow()
        )

        cls.logger.info('Resource #%s with last usage time %s was deleted successfully.', resource.id, last_usage_time)
        return

    @classmethod
    def reset_resource(cls, id_, state=None):
        """
        Resets the resource by given ID - erases its skynet ID and hosts list, sets state to 'NOT_READY' or 'DELETED'.
        :param state:   new resource state to be set.
        """
        state = state or cls.Model.State.NOT_READY
        cls.logger.info("Resetting resource #%s - set state '%s'", id_, state)
        update = dict(unset__skynet_id=True, unset__mds=True, set__state=state, set__time__updated=dt.datetime.utcnow())
        if state == cls.Model.State.NOT_READY:
            update["unset__hosts_states"] = True

        cls.Model.objects(id=id_).update_one(**update)

    @classmethod
    def reset_resource_hosts(cls, id_):
        """ Resets the given resource's hosts list. """
        cls.logger.info("Resetting resource's #%s hosts list.", id_)
        cls.Model.objects(id=id_).update_one(unset__hosts_states=True)

    ########################################
    # Hosts
    ########################################

    # DEPRECATED
    @classmethod
    def get_host(cls, resource_id):
        hosts = cls.get_hosts(resource_id)
        if not hosts:
            return None
        return hosts[0]

    @classmethod
    def add_host(cls, resource_id, host):
        """
        kind of strange logic:
        push new host
        or
        update time of existed OK host
        """
        if cls.Model.objects(id=mapping.ObjectId(resource_id), hosts_states__host=host).first():
            # update last_usage_time only for OK hosts
            cls.touch(resource_id, host)
        else:
            # push new item
            controller.Resource.hard_add_host(resource_id, host)
        cls.logger.info("Added host '%s' for resource #%s", host, resource_id)

    @classmethod
    def remove_host(cls, resource_id, host):
        cls.Model.objects(
            id=mapping.ObjectId(resource_id)
        ).update_one(pull__hosts_states__host=host)
        cls.logger.info("Removed host '%s' for resource #%s", host, resource_id)

    @classmethod
    def touch(cls, resource_id, host=None):
        """
        Update given resource's usage time.
        If optional parameter `host` is passed mark this host state as OK

        :param resource_id: id of resource to touch
        :param host: hostname to mark its state as Ok
        """
        controller.Resource.touch(resource_id, host)

    @classmethod
    def mark_host_to_delete(cls, resource_id, host):
        # priority: not implemented
        cls.Model.objects(
            id=mapping.ObjectId(resource_id),
            hosts_states__host=host
        ).update_one(
            set__hosts_states__S__state=cls.Model.HostState.State.MARK_TO_DELETE
        )
        cls.logger.info("Host '%s' marked to delete for resource #%s", host, resource_id)

    @classmethod
    def get_hosts(cls, resource_id, all=False):
        """
            Получить список хостов

            :param resource_id: идентификатор ресурса
            :param all: вернуть все ресурсы, в том числе и те, что были удалены
            :return: список из названий хостов
        """
        return controller.Resource.get_hosts(resource_id, all)

    ########################################
    # Attributes
    ########################################

    @classmethod
    def set_attr(cls, resource_id, name, value):
        attribute = cls.Model.Attribute(key=cls._cast(name).strip(), value=cls._cast(value).strip())
        updated = cls.Model.objects(
            id=mapping.ObjectId(resource_id),
            attributes__key=attribute.key
        ).update_one(set__attributes__S=attribute)
        if not updated:
            updated = cls.Model.objects(
                id=mapping.ObjectId(resource_id)
            ).update_one(
                push__attributes=attribute,
                set__time__updated=dt.datetime.utcnow()
            )
        if updated:
            cls.logger.info("Set resource #%s attribute {'%s': '%s'}", resource_id, name, value)
        return bool(updated)

    @classmethod
    def drop_attr(cls, resource_id, name):
        res = cls.Model.objects(
            id=mapping.ObjectId(resource_id),
            attributes__key=name
        )
        attr = res.only('attributes').first()
        if not attr:
            return False
        value = filter(lambda x: x.key == name, attr.attributes)[0].value
        res.update_one(pull__attributes__key=name, set__time__updated=dt.datetime.utcnow())
        cls.logger.info("Drop resource #%s attribute {'%s': '%s'}", resource_id, name, value)
        return True

    @classmethod
    def get_attr(cls, resource_id, name):
        obj = cls.Model.objects(
            id=mapping.ObjectId(resource_id),
            attributes__key=name
        ).only('attributes').first()
        if not obj:
            return None
        return filter(lambda x: x.key == name, obj.attributes)[0].value

    @classmethod
    def has_attr(cls, resource_id, name):
        obj = cls.Model.objects(
            id=mapping.ObjectId(resource_id),
            attributes__key=name
        ).only('attributes').first()
        if not obj:
            return False
        return True

    ########################################
    # Dependent
    ########################################

    @staticmethod
    def list_task_dep_resources_id(task_id):
        """
            Вернуть идентификаторы ресурсов, от которых зависит задача

            :param task_id: идентификатор задачи
            :return: список из идентификаторов ресурсов
        """
        task = mapping.Task.objects().only('requirements.resources').with_id(task_id)
        return task.requirements.resources if task else []

    @classmethod
    def list_task_dep_resources(cls, task_id):
        """
            Вернуть объекты ресурсов, от которых зависит задача

            :param task_id: идентификатор задачи
            :return: список из объектов ресурсов
        """
        result = cls.fast_load_list(cls.list_task_dep_resources_id(task_id))
        return result

    @classmethod
    def get_dependent_tasks(cls, resource_id):
        return cls.list_dependent(resource_id)

    @staticmethod
    def list_dependent(resource_id=0, limit=0, offset=0):
        query = mapping.Task.objects(
            requirements__resources=mapping.ObjectId(resource_id)
        ).order_by('-id').skip(offset)
        return list((query.limit(limit) if limit else query).scalar('id'))

    @staticmethod
    def count_dependent(resource_id):
        return mapping.Task.objects(requirements__resources=mapping.ObjectId(resource_id)).count()

    @classmethod
    def get_dependent_list(cls, resource_id=0, limit=0, offset=0):
        from sandbox.yasandbox.manager import task_manager
        task_id_list = cls.list_dependent(resource_id, limit, offset)
        task_list = task_manager.fast_load_list(task_id_list)
        return zip(task_id_list, task_list)

    @classmethod
    def get_dependent_count(cls, resource_id=0):
        return cls.count_dependent(resource_id)

    ########################################
    # Misc
    ########################################

    @classmethod
    def storage_insufficient_redundancy(cls, host=None):
        """
        Lists resources for the given storage host, which has redundancy problems, i.e.,
        has insufficient copies on `H@SANDBOX_STORAGE` hosts group.
        Notice: the request can take significant amount of time (up to several minutes).

        :param host:    Host to be checked, if it's None, check not replicated resources in MDS.
        :return:        A list of tuples with resource ID, resource type and resource size in KiB.
        """

        settings = common.config.Registry()
        hosts = set(settings.server.storage_hosts)
        if host is not None:
            hosts.discard(host)
        items = []
        kws = dict(
            type__ne=str(sdk2.service_resources.TaskLogs),
            state=cls.Model.State.READY,
            hosts_states__state=cls.Model.HostState.State.OK,
            hosts_states__host__nin=hosts,
        )
        if host is None:
            kws["mds"] = None
        else:
            kws["hosts_states__host__in"] = [host]
        query = cls.Model.objects(**kws).fast_scalar("id", "type", "size", "mds")
        for item in query:
            if host is not None and item[-1]:
                continue
            items.append(controller.Resource.ListEntry(*item[:-1]))
        return items

    @classmethod
    def storage_statistics(cls, host):
        cnt = collections.Counter()
        with mapping.switch_db(mapping.Resource, ctd.ReadPreference.SECONDARY) as Resource:
            resources = Resource.host_resources_stats(host)
        rs = cls.storage_insufficient_redundancy(host)
        client = mapping.Client.objects.with_id(host)
        info_system = client.info["system"]
        cnt["disk_total"] += info_system.get("total_space", 0) << 20
        cnt["disk_free"] += info_system.get("free_space", 0) << 20
        cnt["amount_total"] += sum(v[0] for v in resources.itervalues())
        cnt["amount_pure"] += sum(v[0] for k, v in resources.iteritems() if k == cls.Model.State.READY)
        cnt["amount_unique"] = len(rs)
        cnt["size_total"] += int(sum(v[1] for v in resources.itervalues()) * 1024)
        cnt["size_pure"] += int(sum(
            v[1] for k, v in resources.iteritems() if k == cls.Model.State.READY
        ) * 1024)
        cnt["size_unique"] = sum(r[-1] for r in rs) << 10
        cnt["disk_used"] = cnt["disk_total"] - cnt["disk_free"]
        cnt["delta"] = cnt["disk_used"] - cnt["size_pure"]
        return cnt

    @classmethod
    def drop_host_resources(cls, host, resources):
        resources = sorted(map(int, common.utils.chain(resources)))
        cls.logger.info("Remove host '%s' for resources %r", host, resources)
        cls.Model.objects(id__in=resources).update(pull__hosts_states__host=host)

    @classmethod
    def backuped_resources(cls, ids=None, on_hosts=None, not_on_hosts=None, types=None, exclude_types=None):
        """
        Returns ready resources that have backups.

        :param on_hosts: list of hosts the resources are storing on
        :param not_on_hosts: list of hosts the resources aren't storing on
        :param ids: ids of resources
        :param types: list of target resource types
        :param exclude_types: list of resource types to exclude
        :return: resources
        :rtype: `mongoengine.queryset.QuerySet`
        """

        query = {
            "attributes__key": "backup_task",
            "state": cls.Model.State.READY,
        }
        if ids is not None:
            query["id__in"] = ids
        if on_hosts is not None:
            query["hosts_states__host__in"] = on_hosts
        if not_on_hosts is not None:
            query["hosts_states__host__not__in"] = not_on_hosts
        if types is not None:
            query["type__in"] = types
        if exclude_types is not None:
            query["type__not__in"] = exclude_types
        return cls.Model.objects(**query)
