import os
import time
import logging

from sandbox.common import config as common_config
from sandbox.common import patterns
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

logger = logging.getLogger(__name__)


class Task:
    """ Base sandbox task class. """

    Status = ctt.Status
    Priority = ctt.Priority
    SessionState = ctt.SessionState
    # Task's RAM drive requirements specification type. Path to the mount point will be provided at task execution.
    RamDrive = ctm.RamDrive

    # Service task, for internal usage
    SERVICE = False

    # Task that want to run with root privileges inside LXC-containers
    privileged = False

    # Task form declaration.
    input_parameters = []

    # Task type, should be specified in derived class
    type = None

    # Default disk space requirements in megabytes
    @patterns.singleton_classproperty
    def execution_space(self):
        return common_config.Registry().common.task.execution.disk_required

    # Required RAM in MiB
    @patterns.singleton_classproperty
    def required_ram(self):
        return common_config.Registry().common.task.execution.required_ram

    # RAM drive requirements specification
    ramdrive = None
    # Required number of CPU cores
    cores = None
    # Switch task state to FAILURE for any exception occured during task's execution
    fail_on_any_error = False
    # Enable detailed disk usage statistics
    dump_disk_usage = True
    # Tcpdump arguments that are used for network packets logging
    tcpdump_args = ""
    # Task type responsible people.
    owners = []
    # Default recipients list for task release letter
    release_to = None
    # Simultaneous execution limit tag names and amounts.
    se_tags = {}
    # Suspend on statuses
    suspend_on_status = []
    # Task priority score
    score = 0

    # Maximum allowed automatic restart attempts before switching the task to "EXCEPTION" state.
    @patterns.singleton_classproperty
    def max_restarts(self):
        return common_config.Registry().common.task.execution.max_restarts

    # Query describing allowed clients by tags, must be string or common.types.client.Tag.Query object
    client_tags = ctc.Tag.GENERIC
    # Type of name servers in DNS config
    dns = ctm.DnsType.DEFAULT
    # Enable access to Yandex Vault
    enable_yav = False

    def __init__(self, task_id=0):
        self.id = task_id

        self.status = self.Status.DRAFT
        self.descr = ""
        self.info = ""
        self.model = ""
        self.host = ""
        self.lock_host = ""
        self.arch = ""
        self.required_host = ""
        self.parent_id = 0
        self.priority = self.Priority(0, 0)
        self.important = False
        self.hidden = False
        self.owner = ""
        self.author = ""
        self.timestamp = time.time()
        self.updated = self.timestamp
        self.timestamp_start = 0
        self.timestamp_finish = 0
        self.scheduler = None
        self.release_params = None
        self.se_tag = None
        self.tags = []
        self.last_action_user = None
        # Here will be a path to resources synchronization scripts for external tools.
        self.synchrophazotron = None
        # Here will be a path to script that behaves as svn but Arcadia-specific
        self.arcaphazotron = None
        # Task notifications
        self.notifications = []

        self._subproc_list = []

        self.__hosts_by_task_type = None

        self.ctx = {}
        self._mapping = None

        self.taskLogger = None
        self._download_timeout = 3 * 60 * 60

        self.enable_yav = self.__class__.enable_yav

        self._log_resource = None

    def get_default_parameters(self):
        """ Returns default value for the task """
        return {}

    def init_context(self):
        raise NotImplementedError

    def init_as_copy(self, task, request=None):
        raise NotImplementedError

    def title(self):
        return self.descr

    def can_restart(self):
        """ Check whether task can be restarted """
        return all((
            self.Status.can_switch(self.status, self.Status.ENQUEUING),
            self.status not in self.Status.Group.WAIT
        ))

    def copied_from(self):
        return self.ctx.get("copy_of")

    @property
    def tasks_archive_resource(self):
        return self.ctx.get("tasks_archive_resource")

    def __contains__(self, key):
        return hasattr(self, key)

    @property
    def footer(self):
        return None

    def get_release_info(self):
        raise NotImplementedError

    def skynet_get(self, resource):
        raise NotImplementedError

    def rsync_get(self, resource, host):
        raise NotImplementedError

    def is_subtask(self):
        return self.parent_id is not None

    def list_subtasks(self, load=False, completed_only=False, hidden=True):
        raise NotImplementedError

    def send_release_info_to_email(self, additional_parameters):
        raise NotImplementedError

    def mark_released_resources(self, status, ttl="inf"):
        raise NotImplementedError

    def list_resources(self, resource_type="", file_mask=""):
        raise NotImplementedError

    def invalidate_resources(self, with_hosts=False):
        raise NotImplementedError

    def get_common_log_view(self):
        raise NotImplementedError

    def get_debug_log_view(self):
        raise NotImplementedError

    def initLogger(self):
        raise NotImplementedError

    def set_info(self, info, do_escape=True):
        raise NotImplementedError

    def is_finished(self):
        return self.status in (
            self.Status.SUCCESS, self.Status.RELEASING, self.Status.RELEASED, self.Status.NOT_RELEASED
        )

    def is_completed(self):
        return self.status in self.Status.Group.FINISH

    def can_be_restarted(self):
        return self.status in (
            self.Status.EXCEPTION, self.Status.NO_RES, self.Status.TIMEOUT,
            self.Status.STOPPED, self.Status.TEMPORARY, self.Status.EXPIRED,
        )

    def on_enqueue(self):
        """
        This method will be called on first task enqueuing on the server side, right after DRAFT state.
        This means that `create_task` XML-RPC request will __block__ till this method
        completion so think twice before overriding it. It should only used for
        tasks chaining via pre-created resources. In other cases you should use
        `memoize_stage` method.
        """

    def on_prepare(self):
        """
        The method should be overridden to perform some action(s) before switch task to "EXECUTING" state.
        """

    def on_execute(self):
        """ Task logic implementation """
        raise NotImplementedError("Method on_execute is not implemented in Sandbox task {}".format(self.type))

    def on_release(self, additional_parameters):
        """
        Task specific actions.
        Executed when release has submitted.

        :param additional_parameters: release specific additional params
        """
        logging.debug("Release parameters: %r", additional_parameters)
        self.send_release_info_to_email(additional_parameters)
        self.mark_released_resources(additional_parameters["release_status"])

    def postprocess(self):
        """ Called for executing task at the last step, regardless of next task's state and exceptions handling. """
        raise NotImplementedError

    def on_failure(self):
        """ Called when switching task to status FAILURE """
        raise NotImplementedError

    def on_success(self):
        """ Called when switching task to status SUCCESS """
        raise NotImplementedError

    def on_finish(self):
        """ Called when switching to statuses SUCCESS, FAILURE """

    def on_break(self):
        """ Called when switching to statuses group BREAK """
        raise NotImplementedError

    def on_timeout(self):
        """ Called when switching task to status TIMEOUT """

    def on_wait(self):
        """ Called when switching to statuses group WAIT """
        raise NotImplementedError

    def on_terminate(self):
        """
        Called in signal handler when task executing is being stopped forcibly.
        """

    def __repr__(self):
        return "<Task {}/{} (owner: {}, host: {}, current status: {})>".format(
            self.type,
            self.id,
            self.owner,
            self.host,
            self.status,
        )

    def abs_path(self, *args):
        settings = common_config.Registry()
        return os.path.join(settings.client.tasks.data_dir, self.local_path(*args))

    def local_path(self, *args):
        return os.path.join(*(ctt.relpath(self.id) + list(map(str, args))))

    @staticmethod
    def get_task_abs_path(task_id):
        settings = common_config.Registry()
        return os.path.join(settings.client.tasks.data_dir, *ctt.relpath(task_id))

    def log_path(self, *args):
        """
        This method cannot be executed on server-side, but `_log_resource` attribute will be provided by `executor.py`.
        """
        return self._log_resource.abs_path(*args)

    def _register_dep_resource(self, resource_id):
        raise NotImplementedError
