import time
import logging
import cPickle
import calendar
import urlparse
import datetime as dt

from sandbox import common
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc

from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping

logger = logging.getLogger(__name__)


class Client(object):
    """ Encapsulate current client state """

    TagsOp = controller.Client.TagsOp

    Model = mapping.Client

    def __init__(
        self, hostname='', freespace=0, update_ts=0, arch='',
        model='', ncpu=0, ram=0, info=None, reloading=None, pending_commands=None,
    ):
        self.hostname = hostname.lower() if hostname else ''
        self.freespace = int(freespace)
        self.update_ts = update_ts
        self.arch = arch.lower()
        self.model = model.lower()
        self.ncpu = ncpu
        self.ram = ram
        self.info = info if info is not None else {'system': {}}
        self.reloading = reloading or set()
        self.pending_commands = pending_commands or []
        self._mapping = None
        self.protocol_version = None
        self._tags = None

    @property
    def tags(self):
        if self._tags is None and self._mapping:
            self._tags = set(self._mapping.tags)
        return self._tags or set()

    @property
    def os_version(self):
        """
        :return: OS version as string
        """
        return self.get('system', {}).get('os_version', '')

    @property
    def fqdn(self):
        url = self.info.get('system', {}).get('fileserver')
        return urlparse.urlparse(url).hostname if url else None

    def update(self, client_status):
        """
        Update client state from the given dictionary

        :param client_status: dictionary with new state
        """
        self.update_ts = int(time.time())

        sys_info = client_status['system']
        if 'free_space' in sys_info:
            self.freespace = int(sys_info['free_space'])
        if 'arch' in sys_info:
            self.arch = sys_info['arch']
        if 'cpu_model' in sys_info:
            self.model = sys_info['cpu_model']
        if 'ncpu' in sys_info:
            self.ncpu = int(sys_info['ncpu'])
        if 'physmem' in sys_info:
            self.ram = int(sys_info['physmem']) // (1024 ** 2)
        self.info.update(client_status)
        self.info['msg'] = ''
        return self.save()

    def reload(self, cmd=Model.Reloading.RESTART, author=ctu.ANONYMOUS_LOGIN, comment=""):
        """
        Reload client

        :param cmd: string with appropriate command. See Model.Reloading enum for available values
        :param author: user started reload
        :param comment: comment for command
        """
        if ctc.Tag.SERVER in self.tags and cmd in (self.Model.Reloading.REBOOT, self.Model.Reloading.POWEROFF):
            logger.warning("Attempt to reboot or power off a server %r", self.hostname)
            return
        if cmd not in [command.command for command in self.pending_commands]:
            self.pending_commands.append(self.Model.Command(command=cmd, author=author, comment=comment))
        if cmd == self.Model.Reloading.SHUTDOWN:
            self.info.update({"msg": "<b style='color: red'>Turned off by {author} at {datetime}</b>".format(
                author=author,
                datetime=dt.datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S")
            )})
        self.save()

    def pending_service_commands(self, request_commands=None):
        """
        Return set of pending service commands

        :param request_commands: list of available commands
        :rtype: set of str
        """

        return controller.Client.pending_commands(self.pending_commands, request_commands)

    def next_service_command(self, reset=False, request_commands=None):
        """
        Checks whether client must be reloaded

        :param reset: update client service commands
        :param request_commands: list of available commands
        :rtype: yasandbox.database.mapping.client.Client.Command
        """
        commands = self.pending_service_commands(request_commands)
        cmd = next((_ for _ in ctc.ReloadCommand if _ in commands), None)
        if cmd is not None:
            index = next((_ for _, value in enumerate(self.pending_commands) if value.command == cmd), None)

            if reset:
                self.reloading = set(self.reloading) - {cmd}

            if index is not None:
                if reset:
                    ret = self.pending_commands.pop(index)
                else:
                    ret = self.pending_commands[index]
            else:
                ret = self.Model.Command(command=cmd)

            if cmd == self.Model.Reloading.SHUTDOWN:
                self.update_ts = time.time() - common.config.Registry().server.web.mark_client_as_dead_after
            if reset:
                self.save()
            return ret

    def is_alive(self, offset=0):
        """
        Checks last ping time.

        :param offset: used to provide some extra time (in seconds) for client to update itself
        """
        return time.time() - self.update_ts <= common.config.Registry().server.web.mark_client_as_dead_after + offset

    def get_availability(self, offset=0):
        if self.is_alive(offset=offset):
            return ctc.ClientAvailability.ALIVE

        if ctc.Tag.MAINTENANCE in self.tags:
            return ctc.ClientAvailability.DEAD
        else:
            return ctc.ClientAvailability.UNKNOWN

    def _update_unavailability(self, value):
        if self.info.get("net_unreach_ts") != value:
            self.info["net_unreach_ts"] = value
            if value and not self.info.get("msg"):
                self.info["msg"] = "<b style='color: red'>Unavailable via network since {datetime} UTC</b>".format(
                    datetime=dt.datetime.utcfromtimestamp(value).strftime("%Y-%m-%d %H:%M:%S")
                )
            self.save()

    @property
    def available(self):
        return not bool(self.info.get("net_unreach_ts"))

    @available.setter
    def available(self, value):
        if value:
            self._update_unavailability(None)
        else:
            self._update_unavailability(time.time())

    def __str__(self):
        return 'Client "{0}"'.format(self.hostname)

    def __unicode__(self):
        return unicode(self.__str__())

    def __repr__(self):
        return self.__str__()

    def __getitem__(self, key):
        if hasattr(self, key):
            return getattr(self, key)
        else:
            return self.info.get(key)

    def __setitem__(self, key, value):
        if hasattr(self, key):
            setattr(self, key, value)
        else:
            self.info[key] = value

    def __contains__(self, key):
        return hasattr(self, key) or (key in self.info)

    def get(self, key, default_value=None):
        if hasattr(self, key):
            return getattr(self, key)
        else:
            return self.info.get(key, default_value)

    def __eq__(self, other):
        return self.hostname == other.hostname

    def get_info(self):
        """
        Get client info
        * ncpu: number of CPU's cores
        * ram: size of physical memory in Mb

        :return: dictionary with client info
        :rtype: dict
        """
        return {
            'hostname': self.hostname,
            'arch': self.arch,
            'freespace': self.freespace,
            'diskspace': self.info['system'].get('total_space', 0),
            'model': self.model,
            'ncpu': self.ncpu,
            'ram': self.ram,
            'os_version': self.os_version,
            'platform': self.platform,
            'lxc': bool(self.info.get('system', {}).get('lxc')),
            'alive_now': self.is_alive(),
            'inetfs': self.info['system'].get('inetfs', []),
            'storage_device_type': self.info['system'].get('storage_device_type'),
            'tags': tuple(self.tags)
        }

    @property
    def platform(self):
        return self.info['system'].get('platform', '')

    @property
    def lxc(self):
        return ctc.Tag.LXC in self.tags

    @property
    def porto(self):
        return ctc.Tag.PORTOD in self.tags

    @property
    def multislot(self):
        return ctc.Tag.MULTISLOT in self.tags

    @common.utils.singleton_property
    def platforms(self):
        # TODO: SANDBOX-2869: Drop this method somewhen
        if not self.lxc and not self.porto:
            return
        # import sandbox.yasandbox.services.update_sandbox_resources as up
        # return sorted(up.UpdateSandboxResources.lxc_resources)
        return list(ctc.ContainerPlatforms)  # TODO: SANDBOX-5127: Do not fetch the list from database - hardcode it

    def container(self, platform):
        """
        Returns best client's container resource form the given platform selector (`__host_chooser_os` context key).
        """
        return controller.Client.container(self, platform)

    @property
    def revision(self):
        return self._mapping.revision

    @property
    def idle(self):
        """
        Returns a time the client was switched into idle, if any, otherwise `None`.
        """
        idle = self.info.get('idle')
        return (None, None) if not idle else (idle['last_update'], idle['since'])

    def update_tags(self, tags, op):
        client_tags = controller.Client.update_tags_impl(self.hostname, tags, op)
        if client_tags is not None:
            self._tags = client_tags

    def mapping(self):
        """
        Performs internally stored mapping object update and returns it.

        :rtype `yasandbox.database.mapping.client.Client`
        """
        mp = self._mapping
        if not mp:
            mp = self._mapping = self.Model()

        mp.hostname = self.hostname
        mp.updated = dt.datetime.utcfromtimestamp(self.update_ts)
        mp.reloading = list(common.utils.chain(self.reloading))
        mp.pending_commands = self.pending_commands
        mp.platform = self.arch
        mp.context = cPickle.dumps(self.info)
        mp.info.clear()
        mp.info.update(self.info)

        if not mp.hardware:
            mp.hardware = self.Model.Hardware()
            mp.hardware.cpu = self.Model.Hardware.CPU()

        mp.hardware.disk_free = max(self.freespace, 0)
        mp.hardware.ram = self.ram
        mp.hardware.cpu.model = self.model
        mp.hardware.cpu.cores = self.ncpu

        return mp

    def save(self, increase_revision=False):
        """
        Updates internally stored mapping objects and updates the mapping in the database.

        :param increase_revision: In case of `True`, perform client's revision control to avoid
          possible raise conditions. In case on `None`, reset the revision number.
        :rtype: `yasandbox.clients.Client`
        """

        if not self.hostname:
            logger.warn('Attempt to save fake client object.')
            return self
        prev = self._mapping.revision if self._mapping else None
        mp = self.mapping()
        if increase_revision:
            prev = mp.revision
            mp.revision += 1
        elif increase_revision is None:
            mp.revision = 0

        try:
            mp.save(**({"save_condition": {"revision": prev}} if prev is not None else {}))
        except mapping.SaveConditionError as e:
            logger.warning("Conflict during save: %r", e)

        logging.info(
            "Updated client '%s', revision %s->%s",
            self.hostname, prev, mp.revision
        )
        return self

    @classmethod
    def restore(cls, mp):
        """
        Restores `Client` class object based on the given database mapping object.
        :rtype `yasandbox.clients.Client`
        """
        if not mp:
            return None
        obj = cls()
        obj._mapping = mp
        obj.hostname = mp.hostname
        obj.update_ts = calendar.timegm(mp.updated.timetuple())
        obj.reloading = set(common.utils.chain(mp.reloading))
        obj.pending_commands = mp.pending_commands
        obj.arch = mp.platform
        obj.info = cPickle.loads(mp.context)
        obj.freespace = mp.hardware.disk_free
        obj.ram = mp.hardware.ram
        obj.model = mp.hardware.cpu.model
        obj.ncpu = mp.hardware.cpu.cores
        return obj

    def match_tags(self, tags, task_platform=None, only_detect_platform=False):
        return controller.Client.match_tags(
            client=self,
            tags=tags,
            task_platform=task_platform,
            only_detect_platform=only_detect_platform,
        )
