from __future__ import print_function

import os
import time
import math
import pprint
import random
import shutil
import urllib
import logging
import urlparse
import itertools
import datetime as dt
import calendar

logger = logging.getLogger(__name__)

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox import sdk2

import sandbox.agentr.utils
from sandbox.yasandbox import manager
from sandbox.yasandbox import controller
from sandbox.yasandbox.database import mapping


class Resource(object):
    Model = mapping.Resource
    # TODO: This is only used in legacy UI templates.
    State = Model.State

    def __init__(
        self,
        resource_id, name, file_name, file_md5, resource_type, task_id, owner,
        state=None, timestamp=0, dummy=0, skynet_id=None, attrs=None,
        arch=ctm.OSFamily.ANY, size=0, mds=None, multifile=None, executable=None
    ):
        self.id = resource_id
        self.name = name
        self.file_name = file_name
        self.file_md5 = file_md5
        try:
            self.type = sdk2.Resource[str(resource_type)]
        except common.errors.UnknownResourceType:
            self.type = sdk2.service_resources.UnknownResource
        self.task_id = task_id
        self.state = state if state else self.Model.State.NOT_READY
        self.timestamp = timestamp or time.time()
        self.dummy = dummy
        self.last_usage_time = timestamp
        self.last_updated_time = timestamp
        self.attrs = attrs if isinstance(attrs, dict) else (attrs or {})
        self.skynet_id = skynet_id

        self.arch = (
            ctm.OSFamily.ANY
            if self.type.any_arch or not arch else
            common.platform.get_arch_from_platform(arch)
        )

        self.size = size
        self.host = None
        self.owner = owner
        self.hosts = None
        self.mds = mds
        self.force_cleanup = None
        self.multifile = multifile
        self.executable = executable

    def __getstate__(self):
        d = self.__dict__.copy()
        return d

    def __setstate__(self, d):
        self.__dict__.update(d)

    def _remove_duplicates(self, check_path=False):
        """
        Remove NOT_READY duplicates of current resource
        """
        self_attrs = {
            k: str(v)
            for k, v in self.attrs.iteritems()
            if k not in ctr.ServiceAttributes
        }
        query = {
            "task_id": self.task_id,
            "id__ne": self.id,
        }

        if not check_path:
            query.update({
                "state": self.Model.State.NOT_READY,
                "path": self.file_name,
                "type": str(self.type)
            })
        else:
            query["state__in"] = [self.Model.State.READY, self.Model.State.NOT_READY]

        for res in self.Model.objects(**query).order_by("id"):
            res_path = os.path.normpath(self.file_name).split(os.sep)
            obj_path = os.path.normpath(res.path).split(os.sep)

            if res.state == self.Model.State.NOT_READY and res.path == self.file_name and res.type == str(self.type):
                if self_attrs == res.attributes_dict(exclude=ctr.ServiceAttributes):
                    # reuse duplicate resource when creating new resource (id=0)
                    if self.id == 0:
                        manager.resource_manager.logger.warning(
                            "Resource #%s is already created", res.id
                        )
                        return res
                    else:
                        manager.resource_manager.logger.warning(
                            "Resource #%s was deleted as duplicate of #%s", res.id, self.id
                        )
                        res.delete()
            elif check_path and self.file_name and all(l == r for l, r in zip(res_path, obj_path)):
                raise common.errors.TaskError(
                    'Path `{}` of the resource {} has intersection with the path `{}` of resource {} #{}'.format(
                        self.file_name, self.type, res.path, res.type, res.id
                    )
                )

    def _check_for_dups(self):
        """
        Check for duplicates of resource in task

        :return: object if resource with same type, file name and attributes was found;
                 None if resource with same type and file name was not found
        :raises: SandboxException if resource with same type and file name was
                 found but has different attributes
        """
        self_attrs = {
            k: str(v)
            for k, v in self.attrs.iteritems()
            if k not in ctr.ServiceAttributes
        }
        query = {
            "task_id": self.task_id,
            "id__ne": self.id,
            "state__in": [self.Model.State.READY, self.Model.State.NOT_READY],
            "path": self.file_name,
            "type": str(self.type),
        }
        for res in self.Model.objects(**query):
            if self_attrs == res.attributes_dict(exclude=ctr.ServiceAttributes):
                return res

            raise common.errors.TaskError(
                "Resource already exists (#{}) with different attributes:\n{!r} vs {!r}".format(
                    res.id, self_attrs, res.attributes_dict(exclude=ctr.ServiceAttributes)
                )
            )

    def _update(self, obj=None, insert=False):
        if not obj:
            obj = self.Model.objects.with_id(mapping.ObjectId(self.id))
            if not obj:
                return None

        # Remove extra resources created due to RC caused by DB lags (SANDBOX-2984)
        duplicate = self._remove_duplicates(check_path=(insert or self.state == self.Model.State.READY))
        if duplicate is not None:
            return duplicate

        obj.type = str(self.type)
        obj.name = self.name
        obj.path = self.file_name
        obj.owner = self.owner
        obj.task_id = mapping.ObjectId(self.task_id)
        obj.arch = self.arch
        obj.time.created = dt.datetime.utcfromtimestamp(self.timestamp)
        obj.time.accessed = dt.datetime.utcfromtimestamp(self.last_usage_time or self.timestamp)
        obj.state = self.state
        obj.size = self.size
        obj.md5 = self.file_md5
        obj.skynet_id = self.skynet_id
        obj.multifile = self.multifile
        obj.executable = self.executable
        attrs = self.attrs or {}
        obj.attributes = [self.Model.Attribute(key=k, value=v) for k, v in attrs.iteritems()]
        controller.Resource.set_mds(obj, self.mds)

        # Validation for the TTL attribute, it should be either 'inf' or a number less than 10000
        ttl = attrs.get(ctr.ServiceAttributes.TTL)
        if ttl:
            ttl = float(ttl)
            assert math.isinf(ttl) or 1 <= ttl < 10000, "ttl should be either 'inf' or a number less than 10000"

        # Remove expiration time if resource is RELEASED or ttl is set to infinity
        if attrs.get(ctr.ServiceAttributes.RELEASED) or (ttl and math.isinf(ttl)):
            obj.time.expires = None
        elif not obj.time.expires:
            # If expiration date was not provided for the resource yet, set `expires` to 1 day.
            # The service thread, which checks expired resources, will actualize it later.
            obj.time.expires = dt.datetime.now() + dt.timedelta(days=1)
        elif self.attrs is not None:
            # If attributes were changed, set `expires` to now. Service process clean_resources will actualize it later
            obj.time.expires = dt.datetime.now()

        obj.time.updated = dt.datetime.utcfromtimestamp(getattr(self, "last_updated_time", None) or self.timestamp)

        obj.save(**(
            {
                "force_insert": True,
                "write_concern": common.config.Registry().server.mongodb.write_concern
            } if insert else
            {}
        ))
        return obj

    def _create(self):
        obj = self._check_for_dups()
        if obj is None:
            obj = self._update(self.Model(time=self.Model.Time()), insert=True)
        return obj

    @classmethod
    def _restore(cls, obj):
        if not obj:
            return None
        res = cls(
            resource_id=obj.id,
            resource_type=obj.type,
            name=obj.name,
            file_name=obj.path,
            owner=obj.owner,
            task_id=obj.task_id,
            arch=obj.arch,
            timestamp=calendar.timegm(obj.time.created.timetuple()),
            state=obj.state,
            size=obj.size,
            file_md5=obj.md5,
            skynet_id=obj.skynet_id,
            multifile=obj.multifile,
            executable=obj.executable,
        )
        res.last_usage_time = calendar.timegm(obj.time.accessed.timetuple())
        res.last_updated_time = (
            calendar.timegm(obj.time.updated.timetuple())
            if getattr(obj.time, "updated", None) else res.last_usage_time
        )
        if obj.attributes:
            res.attrs = {a.key: a.value for a in obj.attributes}
        if obj.hosts_states:
            res.hosts = [h.host for h in obj.hosts_states if h.state == cls.Model.HostState.State.OK]
        if obj.mds:
            res.mds = {
                "key": obj.mds.key,
                "namespace": obj.mds.namespace,
            }
        if obj.force_cleanup:
            res.force_cleanup = obj.force_cleanup
        return res

    @classmethod
    def _stringify(cls, value):
        """ Recursively convert all unicode strings to str, keep other values the same """
        if isinstance(value, unicode):
            return value.encode("utf-8")
        elif isinstance(value, dict):
            return {key: cls._stringify(val) for key, val in value.iteritems()}
        elif isinstance(value, list):
            return [cls._stringify(val) for val in value]
        return value

    @classmethod
    def _restore_from_json(cls, data):
        data = cls._stringify(data)
        res = cls(
            resource_id=data["id"],
            resource_type=data["type"],
            name=data["description"],
            file_name=data["file_name"],
            owner=data["owner"],
            task_id=data["task"]["id"],
            arch=data["arch"],
            timestamp=calendar.timegm(common.api.DateTime().decode(data["time"]["created"]).timetuple()),
            state=data["state"],
            size=data["size"] >> 10,
            file_md5=data["md5"],
            skynet_id=data["skynet_id"],
            mds=data.get("mds"),
            multifile=data.get("multifile"),
            executable=data.get("executable"),
        )
        res.last_usage_time = calendar.timegm(common.api.DateTime().decode(data["time"]["accessed"]).timetuple())
        res.last_updated_time = calendar.timegm(
            common.api.DateTime().decode(data["time"].get("updated", data["time"]["accessed"])).timetuple()
        )
        res.attrs = data["attributes"]
        if "sources" in data:
            res.hosts = data["sources"]
        res.__meta__ = data
        return res

    @property
    def sources(self):
        """
        Returns a cacheable list of resource sources' identifiers.
        """
        ret = getattr(self, "_sources_cache", None)
        if ret is not None:
            return ret
        if self.hosts is None:
            self.hosts = controller.Resource.get_hosts(self.id)
        nstg = lambda _: ctc.Tag.STORAGE not in _.tags
        clients = filter(None, manager.client_manager.load_list(self.hosts))
        groups = [list(g) for _, g in itertools.groupby(sorted(clients, key=nstg), nstg)]
        map(random.shuffle, groups)
        ret = self._sources_cache = list(itertools.chain.from_iterable(groups))
        return ret

    def is_dummy(self):
        return self.dummy

    def abs_path(self, *args):
        return self.task_path(self.file_name, *args)

    def local_task_path(self, *arg):
        return os.path.join(*(ctt.relpath(self.task_id) + list(arg)))

    def task_path(self, *arg):
        settings = common.config.Registry()
        return os.path.join(settings.client.tasks.data_dir, self.local_task_path(), *arg)

    def local_path(self):
        return self.local_task_path(self.file_name)

    def basename(self):
        return os.path.basename(self.file_name)

    def get_host(self):
        host = next(iter(self.sources), None)
        return host.hostname if host else ""

    def remote_path(self):
        settings = common.config.Registry()
        return os.path.join(settings.client.tasks.data_dir, self.local_path())

    def remote_dirname(self):
        return os.path.dirname(self.remote_path())

    def rsync_url(self, host=None):
        """
        Get rsync link for the resource

        :param host: get link for the host, by default self.get_host()
        :return: rsync link if the resource exists at least on one host
        :rtype: str
        """
        http_prefix = self.http_url(host)
        if not http_prefix:
            return ""
        host = host or next(iter(self.sources), None)
        r = urlparse.urlparse(http_prefix)
        if ctc.Tag.NEW_LAYOUT in getattr(host, "tags", {}):
            r = r._replace(
                scheme='rsync',
                netloc=r.netloc.split(':')[0],
                path='/sandbox-resources'
            )
            return '/'.join((r.geturl(),) + ctr.relpath(self.id) + (os.path.basename(self.file_name),))

        r = r._replace(
            scheme='rsync',
            netloc=r.netloc.split(':')[0],
            path='/sandbox-tasks'
        )
        return '/'.join((r.geturl(), self.local_path()))

    def get_rsync_links(self, all_hosts=False):
        """
        Returns list of rsync URLs to resource's data sources.
        In case of no servers available, empty list will be returned.

        :param all_hosts:   Flags to return links to hosts marked as offline.
        """
        return map(self.rsync_url, (_ for _ in self.sources if all_hosts or _.is_alive()))

    def http_url(self, host=None):
        """
        Get http-link for the resource

        :param host: get link for the host, by default self.get_host()
        :return: HTTP link if the resource exists at least on one host
        """
        # add http prefix and tasks dir
        host = manager.client_manager.load(host) if isinstance(host, basestring) else host
        host = host or next(iter(self.sources), None)
        http_prefix = (host or {}).get('system', {}).get('fileserver', '')

        quoted_filename = urllib.quote(self.file_name)
        path = (
            "/".join(("resource", str(self.id), os.path.basename(quoted_filename)))
            if host and ctc.Tag.NEW_LAYOUT in host.tags else
            self.local_task_path(quoted_filename)
        )
        return urlparse.urljoin(http_prefix, path) if http_prefix else ""

    def sandbox_view_url(self):
        return urlparse.urljoin(common.utils.server_url(), 'resource/{}'.format(self.id))

    @property
    def proxy_url(self):
        """ Sandbox resource proxy url. """
        settings = common.config.Registry()
        proxy_settings = settings.client.fileserver.proxy
        if proxy_settings.host:
            return '{}://{}/{}'.format(proxy_settings.scheme.http, proxy_settings.host, self.id)
        else:
            return self.http_url(settings.this.id)

    def get_http_links(self, all_hosts=False):
        """
        Get HTTP links for the resource

        :param all_hosts: if True returns also links for dead hosts
        :return: list of HTTP link
        """
        return map(self.http_url, (_ for _ in self.sources if all_hosts or _.is_alive()))

    remote_http = http_url  # deprecated

    def local_dirname(self):
        return os.path.dirname(self.local_path())

    def abs_dirname(self):
        return os.path.dirname(self.abs_path())

    def md5sum(self):
        if not self.type.calc_md5:
            return ''
        if self.type.name != 'SEARCH_DATABASE':
            md5all = True
        else:
            md5all = False
        file_md5 = common.utils.md5sum(self.abs_path(), md5all=md5all)
        return file_md5

    def is_ready(self):
        return self.state == self.Model.State.READY

    def is_not_ready(self):
        return self.state == self.Model.State.NOT_READY

    def is_deleted(self):
        return self.state == self.Model.State.DELETED

    def user_has_permission(self, user):
        if controller.user_has_permission(user, [self.owner]):
            return True
        try:
            task = controller.Task.get(self.task_id)
        except controller.Task.NotExists:
            return False
        return controller.user_has_permission(user, (task.author, task.owner))

    def title(self):
        name = self.name
        if self.state == self.Model.State.DELETED:
            name = "[Removed] %s" % name

        return u'({0}, {1}@{2} {3})'.format(self.formatted_time(), self.id, self.task_id, name)

    def formatted_time(self):
        return dt.datetime.fromtimestamp(self.timestamp).strftime('%d-%m-%Y %H:%M')

    @classmethod
    def register(cls, resource_id, check_attrs=None):
        """ Register resource in AgentR and return True if it's in NOT_READY state. Returns False otherwise. """
        data = sdk2.Task.server.resource[resource_id][:]
        resource = cls._restore_from_json(data)
        if resource.state == ctr.State.NOT_READY:
            if check_attrs is not None:
                local_attrs = {str(k).strip(): str(v).strip() for k, v in check_attrs.iteritems()}
                if local_attrs != data["attributes"]:
                    sdk2.Task.server.resource[resource_id].update(
                        {"attributes": local_attrs}
                    )
                data["attributes"] = local_attrs

            res_cls = sdk2.Resource[resource.type.name]

            # Add source for resources subclassed from TASK_LOGS (such as TASK_CUSTOM_LOGS).
            # It makes them available during the task execution.
            add_source = issubclass(res_cls, sdk2.service_resources.TaskLogs)

            sdk2.Task.current.agentr.resource_register_meta(
                resource.file_name, data, share=res_cls.share,
                service=issubclass(res_cls, sdk2.ServiceResource),
                add_source=add_source,
            )
            return True
        return False

    def update_share_fields(self, res_data):
        new_res = self._restore_from_json(res_data)
        self.state = new_res.state
        self.file_md5 = new_res.file_md5
        self.size = new_res.size
        self.skynet_id = new_res.skynet_id
        self.last_usage_time = new_res.last_usage_time

    @property
    def sdk1_agentr_enabled(self):
        cur_task = sdk2.legacy.current_task
        return cur_task and cur_task.sdk1_agentr_enabled

    def mark_ready(self):
        """ Marks resource as 'READY'. Checks existence of the resource on the host. """
        if self.sdk1_agentr_enabled:
            self.register(self.id, check_attrs=self.attrs)
            res_data = sdk2.Task.current.agentr.resource_complete(self.id)
            self.update_share_fields(res_data)
            return

        resource_path = self.abs_path()
        if not os.path.exists(resource_path):
            raise common.errors.TaskError(
                "cannot mark resource:{} as completed: path {} not found".format(self.id, resource_path)
            )
        relpath = os.path.relpath(resource_path, self.task_path())
        if relpath.split(os.path.sep, 1)[0] in (os.path.curdir, os.path.pardir):
            raise common.errors.TaskError(
                "cannot mark resource:{} as completed: "
                "path {!r} is not a subdirectory of task working directory {!r}".format(
                    self.id,
                    resource_path,
                    self.task_path()
                )
            )
        # if task created with architecture 'any' or architecture not defined,
        # but resource type depends on architecture, then set architecture of
        # the execution host
        if self.arch == ctm.OSFamily.ANY and not self.type.any_arch:
            task = self.get_owner_task()
            self.arch = (
                task.arch
                if task.arch != ctm.OSFamily.ANY else
                controller.Client.get(task.host, create=True).platform
            )

        self.add_host()

        self.state = self.Model.State.READY
        self.file_md5 = self.md5sum()
        self.last_usage_time = time.time()
        self.last_updated_time = time.time()
        try:
            self.size = common.fs.check_resource_data(resource_path)
        except ValueError as ex:
            raise common.errors.TaskError('Resource {} #{} data invalid: {}'.format(self.type, self.id, ex))

        logger.debug('Resource "{0}" was locally marked as "{1}" [md5 "{2}", time "{3}", size "{4}"]'.format(
            str(self), self.state, self.file_md5, self.timestamp, self.size))

        try:
            self.share()
        except Exception:
            settings = common.config.Registry()
            logger.exception("Sharing of resource %s on %s failed" % (self.id, settings.this.id))
            raise

        logger.debug('Update manager.resource_manager after resource state change')
        manager.resource_manager.update(self)

    def mark_broken(self, state=None):
        if self.sdk1_agentr_enabled:
            self.register(self.id, check_attrs=self.attrs)
            res_data = sdk2.Task.current.agentr.resource_complete(self.id, broken=True)
            self.update_share_fields(res_data)
            return

        self.state = state if state else self.Model.State.BROKEN
        self.last_usage_time = time.time()
        self.last_updated_time = time.time()
        if os.path.exists(self.abs_path()):
            self.size = common.fs.get_dir_size(self.abs_path())
        manager.resource_manager.update(self)
        if os.path.exists(self.abs_path()):
            self.add_host()

    def remove_resource_files(self):
        """
        Physically remove files of the resource.
        Must be called on client side only !
        """
        resource_base_dir = self.abs_dirname()
        try:
            path = self.abs_path()

            if os.path.exists(path):
                common.fs.chmod_for_path(self.task_path(), "a+w")
                tmppath = path + ".remove"
                os.rename(path, tmppath)
                if os.path.isdir(tmppath):
                    shutil.rmtree(tmppath)
                else:
                    os.remove(tmppath)

        except Exception as e:
            logger.exception('Unable to remove resource %s data: %s' % (self, e))
        finally:
            # after removal of the resource needed to remove empty directory
            if (
                os.path.exists(resource_base_dir) and
                not os.listdir(resource_base_dir) and
                not os.path.samefile(resource_base_dir, self.task_path())
            ):
                common.fs.chmod_for_path(self.task_path(), "a+w")
                os.removedirs(resource_base_dir)
            common.fs.chmod_for_path(self.task_path(), "a-w")
            self.remove_host()

    def add_host(self):
        settings = common.config.Registry()
        if self.sdk1_agentr_enabled:
            sdk2.Task.server.resource[self.id].source()
        else:
            return manager.resource_manager.add_host(self.id, settings.this.id)

    def touch(self):
        if self.sdk1_agentr_enabled:
            sdk2.Task.server.resource[self.id].update({})
        else:
            return manager.resource_manager.touch(self.id)

    # TODO: useless method, should be removed
    def remove_host(self):
        settings = common.config.Registry()
        return manager.resource_manager.remove_host(self.id, settings.this.id)

    def get_hosts(self, all=False):
        return manager.resource_manager.get_hosts(self.id, all=all)

    def get_storage_hosts(self, all_hosts=False):
        """ List of storage hosts on which copy of the resource exists """
        settings = common.config.Registry()
        hosts = self.get_hosts(all=all_hosts)
        s = set(hosts).intersection(set(settings.server.storage_hosts))
        return list(s)

    def get_worker_hosts(self, all_hosts=False):
        """ List of worker hosts on which copy of the resource exists """
        settings = common.config.Registry()
        hosts = self.get_hosts(all=all_hosts)
        s = set(hosts).difference(set(settings.server.storage_hosts))
        return list(s)

    def share(self):
        """
        in skynet v7.0.x
        resource_id depends on:
        1) names and paths in the resource
        2) chmod of files and directories (i.e. rwxrwxr-x must be equal)
        3) files content
        """
        rpath = self.remote_path()
        # add rights on read for all
        common.fs.chmod_for_path(rpath, "a+rX")
        settings = common.config.Registry()
        if not self.type.share:
            logger.debug('Skip resource %s sharing at "%s"', self.type, self.abs_path())
            return
        if self.state != self.Model.State.READY:
            raise common.errors.TaskError("Trying to share broken resource {}".format(self))
        host = settings.this.id
        logger.debug('Begin share resource "%s" on host "%s" with path "%s"', str(self), host, rpath)

        with sandbox.agentr.utils.TranslateSkyboneErrors(logger, "Unable to share resource {}".format(self)):
            resid = common.share.skynet_share(os.path.dirname(rpath), os.path.basename(rpath))

        logger.debug('Resource "%s" successfully shared with skynet_id "%s"', str(self), resid)
        if self.skynet_id and self.skynet_id != resid:
            import api.copier
            files = api.copier.Copier().handle(resid).list().wait(timeout=10).files()
            logger.debug('Files structure taken: ' + pprint.pformat(files))
            raise common.errors.IncorrectStatus(
                "copier.share: new ID '{}' is not equal to original '{}'".format(resid, self.skynet_id)
            )
        if not self.skynet_id:
            self.skynet_id = resid
            logger.debug('Update manager.resource_manager after sharing')
            manager.resource_manager.update(self)

    def to_dict(self):
        d = {
            k: getattr(self, k)
            for k in [
                'id', 'owner', 'state', 'task_id', 'name', 'arch', 'size', 'timestamp', 'last_usage_time',
                'last_updated_time', 'file_name', 'file_md5', 'skynet_id', 'attrs', 'host', 'dummy',
            ]
        }
        d.update({
            'type_name': self.type.name,
            'url': self.http_url(),
            'proxy_url': self.proxy_url,
            'path': self.remote_path(),
            'sandbox_view_url': self.sandbox_view_url(),
            'rsync': self.rsync_url(),
        })
        #
        if self.state == self.Model.State.READY:
            d['is_ok'] = 1
            d['complete'] = 1
        elif self.state == self.Model.State.NOT_READY:
            d['is_ok'] = 0
            d['complete'] = 0
        else:
            d['is_ok'] = 0
            d['complete'] = 1
        # Don't break xmlrpc in perl
        # because nil tag is not in xmlrpc spec
        for k, v in d.items():
            if v is None:
                d[k] = ''
        return d

    def from_dict(self, params):
        ignore_keys = ('is_ok', 'complete', 'path', 'sandbox_view_url', 'url', 'proxy_url')
        copy_keys = set(params.iterkeys()) - set(ignore_keys)
        self.__dict__.update((k, params[k]) for k in copy_keys)

    def __repr__(self):
        return "<Resource {}/{} (task_id:{}, state:{}, local_path:{})>".format(
            self.type.name, self.id, self.task_id, self.state, self.local_path()
        )

    def get_owner_task(self):
        return manager.task_manager.load(self.task_id)

    def restore(self, host=None):
        """ Restore resource in state 'DELETED' on specified host or on all hosts """
        if self.state != self.Model.State.DELETED:
            raise common.errors.TaskError("Can restore only DELETED resource not {}".format(self.state))

        all_hosts = self.get_hosts(all=True)
        # cannot restore the resource on the host where it did not exist
        if host and host not in all_hosts:
            raise common.errors.TaskError("Can't restore resource on unknown host: {}".format(host))

        # cannot restore the resource if all its copies already deleted,
        # e.i. resource_hosts is empty for the resource
        if not all_hosts:
            raise common.errors.TaskError("Can't restore resource with empty hosts")

        if host:
            hosts = [host]
        else:
            hosts = all_hosts

        restore_tasks = []
        for host in hosts:
            task_id = controller.Task.create_service_task(
                "RESTORE_RESOURCE",
                description="Restoring resource #{} on host {}".format(self.id, host),
                parameters=dict(resource_id=self.id),
                requirements=dict(host=host),
                priority=ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.HIGH)
            )
            restore_tasks.append(task_id)
        return restore_tasks
