import os
import sys
import copy
import time
import json
import shlex
import random
import cPickle
import logging
import tarfile
import calendar
import urlparse
import textwrap
import platform
import datetime as dt
import traceback
import itertools as it
import subprocess

import six
import requests

from sandbox import common
from sandbox.common import patterns as common_patterns
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.common.types import notification as ctn

from sandbox import sdk2
from sandbox.sdk2 import legacy as sdk_legacy

from sandbox.yasandbox import manager
from sandbox.yasandbox.database import mapping

from sandbox.yasandbox import controller
from sandbox.yasandbox.controller import user as user_controller
from sandbox.yasandbox.controller import trigger as trigger_controller
from sandbox.yasandbox.controller import task_status_event as tse_controller

from sandbox.yasandbox.proxy import base


from . import resource as resource_proxy


logger = logging.getLogger(__name__)

# Commonly used data model type shortcut.
Model = mapping.Task


def isValidTaskType(task_type):
    """
    Checks if given task type is valid, whether its code exists currently

    :param task_type: type name
    :rtype: bool
    """
    from sandbox import projects
    return str(task_type) in projects.TYPES


def listTaskTypes():
    """
    Get list of all task types existent currently

    :return: sorted list of types
    :rtype: list
    """
    try:
        from sandbox import projects
    except ImportError:
        return []
    task_types = getattr(projects, "TYPES", ctm.NotExists)
    if task_types is ctm.NotExists:
        common.projects_handler.load_project_types()
    return sorted(projects.TYPES)


def getTaskClass(task_type):
    """
    Get task class by name. If task type does not exist, returns base class 'Task'

    :param task_type: task type name
    :return: task class
    """
    try:
        from sandbox import projects
    except ImportError:
        return Task
    if task_type not in projects.TYPES:
        return Task
    return projects.TYPES[task_type].cls


class FilterCheckResult(dict):
    """ Task filter result """
    def __init__(self, filter_name, task_parameters, client_values, allowed):
        super(FilterCheckResult, self).__init__({
            'filter_name': filter_name,
            'task_parameters': task_parameters,
            'client_values': client_values,
            'allowed': allowed,
        })

    def __nonzero__(self):
        return bool(self['allowed'])

    def __str__(self):
        if self['allowed']:
            allowed = 'allowed'
        else:
            allowed = 'banned'
        return 'Filter {0} results: {1}, task parameters: {2}, client values: {3}'.format(
            self['filter_name'], allowed, self['task_parameters'], self['client_values']
        )

    def __repr__(self):
        return str(self)


class Task(base.Task):
    """ Base sandbox task class. """

    # Sandbox own service context fields.
    SERVICE_CONTEXT_KEYS = [
        '__GSID',
        'kill_timeout',
        'do_not_restart',
        'fail_on_any_error',
        'tasks_archive_resource',
        'notify_via',
        'notify_if_failed',
        'notify_if_finished',
    ]

    # Default context values.
    # noinspection PyPep8Naming
    @common_patterns.singleton_classproperty
    def DEFAULT_CONTEXT(self):
        return {
            'do_not_restart': False,
            'kill_timeout': common.config.Registry().common.task.execution.timeout * 60 * 60,
            'notify_via': ctn.Transport.EMAIL
        }

    # Object attributes, which are registered for serialization and pass through XMLRPC interface.
    SERIALIZE_FIELDS = {
        'priority', 'owner', 'id', 'timestamp_finish', 'descr', 'type', 'parent_id', 'hidden',
        'status', 'timestamp', 'updated', 'timestamp_start', 'timestamp_start', 'arch', 'info', 'ctx',
        'host', 'model', 'cores', 'required_host', 'execution_space', 'required_ram', 'important',
        'scheduler', 'se_tag', 'tags', 'suspend_on_status', 'notifications', 'score'
    }

    def __get_defaults_from_task_class(self):
        """
        Safely calls 'get_default_parameters' ignoring all exceptions.
        Needed for updating of task context by default values.
        If 'get_default_parameters' return dictionary, update the task contex by it.
        :return: True, if the values are successfully obtained, False otherwise.
                 If exception occurred, save it in the field 'info' of the task.
        """

        try:
            self.ctx.update(self.DEFAULT_CONTEXT)
            result = self.get_default_parameters()
            if result and isinstance(result, dict):
                self.ctx.update(result)
        except Exception as error:
            self.set_info("Exception in init_context: %s." % error)
            self.set_info(" ".join(traceback.format_exception(*sys.exc_info())))
            return False
        return True

    def _update_task_gsid(self, context):
        prev_gsid = context.pop("GSID") if "GSID" in context else context.get("__GSID", "")
        if not prev_gsid and self.parent_id:
            parent = controller.TaskWrapper(mapping.Task.objects.with_id(self.parent_id))
            prev_gsid = parent.ctx.get("__GSID", "")

        context["__GSID"] = controller.Task.extend_gsid(prev_gsid, self.id, self.type, self.author)
        return context

    def init_context(self):
        """
        Fill the task context by default values

        :return: True if successful
        """
        # Notice:
        #   this crappy piece of code is used to workaround the following problem:
        #   inside a method `get_default_context_values()` is overridden by `sandboxsdk.task.SandboxTask`
        #   which calls `self.InitCtx` which should return a dictionary, which is supposed to provide default
        #   values for the initializing task's context, the task's context updated directly by the method and
        #   `None` returned. So this code in some sort ready to merge changes, which returned by a method
        #   to a dictionary, which is modified by the same method.

        prev_ctx = copy.deepcopy(self.ctx)
        self._update_task_gsid(prev_ctx)
        self.ctx = {
            p.name: p.default_value
            for p in self.input_parameters if not p.dummy
        }
        if self.__get_defaults_from_task_class():
            self.ctx.update(prev_ctx)
            return True
        else:
            self.ctx = prev_ctx
            return False

    def init_as_copy(self, task, request=None):
        self.ctx.update({
            p.name: task.ctx[p.name]
            for p in task.input_parameters if not p.do_not_copy and p.name in task.ctx
        })

        self.model = task.model
        self.host = task.host
        self.arch = task.arch
        self.required_host = task.required_host
        self.required_ram = task.required_ram
        self.descr = task.descr
        self.client_tags = task.client_tags
        if isinstance(task.priority, ctt.Priority):
            self.priority = task.priority
        else:
            self.priority.__setstate__(task.priority)
        self.execution_space = task.execution_space
        notifications = mapping.Task.objects.scalar("notifications").with_id(task.id)
        for _ in notifications:
            _.recipients = map(
                lambda _: (
                    self.author
                    if task.author == _ else
                    _
                ),
                _.recipients
            )
        mapping.Task.objects(
            id=self.id,
        ).update_one(
            set__notifications=notifications
        )

        self.ctx["copy_of"] = task.id
        prev_ctx = copy.deepcopy(task.ctx)
        for k in self.SERVICE_CONTEXT_KEYS:
            if k in prev_ctx:
                self.ctx[k] = prev_ctx[k]
        self.ctx.update(self._update_task_gsid({}))
        controller.Task.audit(mapping.Audit(task_id=task.id, content="Copied to #{}".format(self.id)))
        controller.Task.audit(mapping.Audit(task_id=self.id, content="Copied from #{}".format(task.id)))

    def __getstate__(self):
        self.check_ctx(self.ctx)
        d = self.__dict__.copy()
        del d['_subproc_list']
        del d['taskLogger']
        return d

    def __setstate__(self, d):
        self.__init__()
        self.__dict__.update(d)

    def _format_sync_log(self, resource, **args):
        sync_log = common.log.get_client_log('sync')
        d = {
            'task_id': self.id,
            'task_type': self.type,
            'resource_id': resource.id,
            'resource_type': resource.type.name,
            'dst_host': self.host,
        }
        d.update(args)
        sync_log.debug(
            "[{task_id}:{task_type}]"
            "[{resource_id}:{resource_type}]"
            "[{src_host}:{dst_host}]"
            "[{protocol}][{result}]".format(**d)
        )

    @property
    def platform_filter(self):
        """ Gets value of filter by platform """
        filter_value = self.arch
        if not filter_value or filter_value == ctm.OSFamily.ANY:
            return None
        else:
            return filter_value

    @platform_filter.setter
    def platform_filter(self, value):
        """ Sets value of filter by platform """
        self.arch = str(value)

    @property
    def cpu_model_filter(self):
        """ Gets value of filter by CPU model """
        return self.model

    @cpu_model_filter.setter
    def cpu_model_filter(self, value):
        """ Sets value of filter by CPU model """
        self.model = str(value)

    @property
    def client_hostname_filter(self):
        """ Gets value of filter by host name """
        return self.required_host

    @client_hostname_filter.setter
    def client_hostname_filter(self, value):
        """ Sets value of filter by host name """
        self.required_host = str(value)

    def _local_get(self, resource):
        settings = common.config.Registry()
        if settings.this.id in resource.get_hosts(all=True):
            if not os.path.exists(resource.abs_path()):
                logger.error(
                    "Can't find resource copy %s at %s" % (
                        resource.abs_path(), settings.this.id))
                return False
            self._format_sync_log(resource,
                                  src_host=self.host,
                                  protocol="none",
                                  result="OK")
            return True
        else:
            return False

    def get_release_info(self):
        """
        Gets info on task release

        :return: None if task has no resources to release, otherwise
          a dictionary with information about a possible release, for example:

        .. code-block:: python

            {
                'release_resources': [ # info of resource for release
                         {
                        'id': 2452352,
                        'type': 'SOME_TYPE',
                        'description': 2452352,
                        'arch': 'freebsd',
                        'releasers': ['user1', 'user2', 'user3', ],
                        'http_url': 'http://...',
                        'skynet_id': 'rbtorrent:fab1fddcb20da9332874db43c4021d0b1528de9f',
                        'file_md5': '0feea7c4f86d57819e0deecffdc87cd0'
                    }
                ]
            }
        """

        result = {'release_resources': [], }
        can_release = None
        for resource in self.list_resources():
            if resource.is_ready() and resource.type.releasable:
                if resource.type.releasers is None:
                    resource_type_releasers = []
                else:
                    resource_type_releasers = resource.type.releasers
                result['release_resources'].append(
                    {
                        'id': str(resource.id),
                        'type': str(resource.type),
                        'description': resource.name,
                        'arch': resource.arch,
                        'releasers': resource_type_releasers,
                        'http_url': resource.proxy_url,
                        'skynet_id': resource.skynet_id,
                        'file_md5': resource.file_md5
                    }
                )
                if can_release is None:
                    can_release = resource_type_releasers
                else:
                    can_release = list(set(can_release) & set(resource_type_releasers))
        result['can_release'] = can_release
        if not result['release_resources']:
            return None
        return result

    def skynet_get(self, resource):
        if not resource.skynet_id:
            return False
        logger.debug(
            'Resource {0} has skynet id ({1}), fetching...'.format(
                resource, resource.skynet_id))
        self._format_sync_log(resource,
                              src_host="unknown",
                              protocol="skynet",
                              result="START")
        try:
            common.share.skynet_get(
                resource.skynet_id,
                resource.abs_dirname(),
                common.share.calculate_timeout(resource.size << 10),
                size=resource.size << 10,
                fallback_to_bb=True,
            )
        except Exception:
            logger.exception("Failed to get resource #%s with skynet '%s'", resource.id, resource.skynet_id)
            self._format_sync_log(resource,
                                  src_host="unknown",
                                  protocol="skynet",
                                  result="FAIL")
            # Make locally created files writable for future rsync fallback attempts.
            common.fs.chmod_for_path(resource.abs_dirname(), 'a+w', recursively=True)
            return False
        logger.debug('Fetched with skynet.')
        self._format_sync_log(resource,
                              src_host="unknown",
                              protocol="skynet",
                              result="OK")
        return True

    def rsync_get(self, resource, host):
        rsync_src = resource.rsync_url(host)
        rsync_dst = '{0}/'.format(resource.abs_dirname())
        self._format_sync_log(resource, src_host=host, protocol="rsync", result="START")
        timeout = common.share.calculate_timeout(resource.size << 10)
        try:
            self.remote_copy(rsync_src, rsync_dst, 'rsync', timeout=timeout, resource_id=resource.id)
        except Exception:
            logger.exception("Failed to get resource #%s with rsync '%s'", resource.id, rsync_src)
            self._format_sync_log(resource, src_host=host, protocol="rsync", result="FAIL")
            return False
        logger.debug("Fetched with rsync.")
        self._format_sync_log(resource, src_host=host, protocol="rsync", result="OK")
        return True

    def _sync_resource(self, resource, http_first=False):
        """
        Synchronizes given resource's data to the current host (if no local copy available already).

        :param resource:    :class:`yasandbox.proxy.resource.Resource` instance
        :param http_first:  Try to download resource's data via HTTP firstly
        :return:            Path to resource's data root
        :rtype: str
        """
        import api.heartbeat.client
        settings = common.config.Registry()
        logger.debug("Sync resource #%s owned by task #%s.", resource.id, resource.task_id)

        if not resource.is_ready():
            raise common.errors.TaskError("Resource #{} is not ready. Cannot sync it.".format(resource.id))

        if self._local_get(resource):
            logger.debug("Resource #%s already exists on local host.", resource.id)
            resource.touch()
            return resource.abs_path()

        journal = common.fs.FSJournal()
        root = (
            journal.mkroot(
                resource.task_id,
                state=journal.Directory.State.RONLY,
                maker=common.fs.create_task_dir,
                args=(resource.task_id,),
            )
            if resource.task_id != self.id else
            journal.mkroot(
                resource.task_id,
                state=journal.Directory.State.KEEP,
                maker=self.abs_path,
            )
        )

        rdir = root.adddir(resource.abs_path(), journal.Directory.State.KEEP)
        try:
            attempts, copied = 3, False
            http_source = next(iter(resource.sources), None) if http_first else None
            if http_source:
                rdir.mkdir()
                url = resource.http_url(http_source) + "?stream=tar"
                logging.info("Trying following HTTP data source for resource #%s: %r", resource.id, url)
                try:
                    def members(tar):
                        for tarinfo in tar:
                            logging.debug("Resource #%s progress: extracting %r", resource.id, tarinfo.name)
                            yield tarinfo

                    r = requests.get(url, stream=True, timeout=60)
                    with tarfile.open(mode="r|", fileobj=r.raw) as tar:
                        tar.extractall(os.path.dirname(resource.abs_path()), members=members(tar))
                    logger.info("Successfully fetched resource #%s via HTTP. Sharing it.", resource.id)
                    resource.share()
                    copied = True
                except Exception as ex:
                    logger.error("Unable to fetch resource #%s data via HTTP: %s", resource.id, ex)

            if not copied:
                copied = self.skynet_get(resource)
            # download resource with rsync
            if not copied:
                rdir.mkdir()
                hosts = resource.get_storage_hosts() + resource.get_worker_hosts()
                hosts = (hosts if len(hosts) >= attempts else hosts * attempts)[:attempts]
                logging.info("Trying following data sources for resource #%s: %r", resource.id, hosts)
                for h in hosts:
                    copied = self.rsync_get(resource, h)
                    if copied:
                        resource.share()
                        break
                if copied and resource.skynet_id:
                    api.heartbeat.client.scheduleReport(
                        "sandbox_sync_stats",
                        {
                            "skybone_fb": False,
                            "skybone_bb": False,
                            "rsync_bb": True,
                            "size": resource.size << 10,
                        },
                        incremental=True
                    )

            if copied:
                resource.add_host()
                if resource.task_id != self.id:
                    rdir.state = rdir.State.RONLY
            else:
                if resource.skynet_id:
                    api.heartbeat.client.scheduleReport(
                        "sandbox_sync_stats",
                        {
                            "skybone_fb": False,
                            "skybone_bb": False,
                            "rsync_bb": False,
                            "size": resource.size << 10,
                        },
                        incremental=True
                    )
                raise common.errors.TemporaryError('Unable to sync resource:{0}'.format(resource.id))
        except Exception:
            logger.exception("Cannot sync resource #%s on '%s'.", resource.id, settings.this.id)
            rdir.state = journal.Directory.State.REMOVE
            raise
        finally:
            journal.commit(resource.task_id)
        return resource.abs_path()

    # TODO: Delete after abandoning the old GUI and XMLRPC API.
    def create_notifications_from_context(self):
        """ Save notifications in the task database object from its context """
        notification_ctx = {k: self.ctx.pop(k, None) for k in ("notify_via", "notify_if_finished", "notify_if_failed")}
        notifications = self.new_notifications(self.author, self.owner, notification_ctx)
        mapping.Task.objects(
            id=self.id,
        ).update_one(
            set__notifications=notifications
        )
        return notification_ctx

    # TODO: Delete after abandoning the old GUI and XMLRPC API.
    def drop_notifications_from_context(self):
        """ Drop fields related to notifications from task context """
        for field in ("notify_via", "notify_if_finished", "notify_if_failed"):
            self.ctx.pop(field, None)

    # TODO: Delete after abandoning the old GUI and XMLRPC API.
    @staticmethod
    def new_notifications(author, owner, ctx):
        """ Convert old notification info format to database documents """
        notifications = []
        notify_via = ctx.get("notify_via")
        if notify_via:
            for key, status in (
                ("finished", ctt.Status.SUCCESS),
                ("failed", ctt.Status.FAILURE)
            ):
                recipients = [author]
                nif = ctx.get("notify_if_" + key)
                if isinstance(nif, six.string_types):
                    nif = nif.strip()
                    if nif:
                        recipients.extend(map(lambda s: s.strip(), nif.split(',')))
                    else:
                        recipients = None
                elif nif is not None and not nif:
                    recipients = None

                notifications.append(manager.notification_manager.notification(
                    notify_via,
                    status,
                    recipients
                ))
            notifications.append(manager.notification_manager.notification(
                notify_via,
                [ctt.Status.EXCEPTION, ctt.Status.NO_RES, ctt.Status.TIMEOUT],
                author, owner
            ))
        return filter(None, notifications)

    def is_subtask(self):
        return self.parent_id is not None

    def list_subtasks(self, load=False, completed_only=False, hidden=True):
        """
        Gets list to subtasks

        :param load: if True, returns list of loaded by manager.task_manager.load objects, otherwise list of ids
        :param completed_only: if True, count only completed tasks, otherwise tasks in any status
        :param hidden: if True, take into account hidden tasks also
        """
        subtasks = manager.task_manager.list_subtasks(
            parent_id=self.id,
            completed_only=completed_only,
            hidden=hidden
        )
        return manager.task_manager.fast_load_list(subtasks) if load else subtasks

    def _release_info(self, user, comments):
        """
        Construct plain text release form

        :param user: releaser login
        :param comments: comments to release
        :return: info about release (author, comments, resources etc)
        :rtype: str
        """
        if not comments:
            comments = u' '
        info_string = u'Author: {}\nSandbox task: {}task/{}\n'.format(
            user,
            common.utils.server_url(),
            self.id
        )
        resource_string = 'Resources:\n'
        for resource in self.list_resources():
            if resource.type.releasable:
                b = u' * {} - id:{} [{}]'.format(
                    resource.type, resource.id, resource.arch or ctm.OSFamily.ANY
                )
                binary_info = u'{}:\n     File name:         {}, {}KB (md5: {})'.format(
                    b,
                    os.path.basename(resource.file_name),
                    resource.size,
                    resource.file_md5
                )
                padding = u'   '
                sandbox_info = u'{}  Resource page:     {}'.format(padding, resource.sandbox_view_url())
                http_info = u'{}  HTTP download URL: {}'.format(padding, resource.proxy_url)
                skynet_info = u'{}  Skynet ID:         {}'.format(padding, resource.skynet_id or 'N/A')

                resource_string = u'{}{}\n\n'.format(
                    resource_string,
                    '\n'.join([binary_info, sandbox_info, http_info, skynet_info])
                )
        sign_string = u'--\nsandbox \n https://sandbox.yandex-team.ru/ \n'
        return u"\n".join([info_string, comments, resource_string, sign_string])

    def send_release_info_to_email(self, additional_parameters):
        from sandbox.sandboxsdk import channel

        author = additional_parameters['releaser']
        release_status = additional_parameters['release_status']
        message_subject = additional_parameters.get('release_subject')
        message_body = additional_parameters.get('release_comments')
        release_to = additional_parameters['email_notifications']['to']
        release_cc = additional_parameters['email_notifications']['cc']
        if not release_to:
            release_to, release_cc = release_cc, release_to
            if not release_to:
                logging.info("No recipients specified for the release.")
                return

        mail_subj = u'[{0}] {1}'.format(release_status, message_subject) if release_status else message_subject
        mail_body = self._release_info(author, message_body)
        try:
            channel.channel.sandbox.server.send_email(
                release_to, release_cc, mail_subj, mail_body,
                'text/plain', 'utf-8', None, False, 'sandbox',
                self.id, ctn.View.RELEASE_REPORT
            )
        except Exception:
            logger.exception('')

    def mark_released_resources(self, status, ttl="inf"):
        resources = [_ for _ in self.list_resources() if _.type.releasable]
        for resource in resources:
            resource.attrs[ctr.ServiceAttributes.TTL] = resource.attrs.get(ctr.ServiceAttributes.TTL_ON_RELEASE) or ttl
            resource.attrs[ctr.ServiceAttributes.RELEASED] = status
            resource.attrs.setdefault(ctr.ServiceAttributes.BACKUP_TASK, True)
            manager.resource_manager.update(resource)
        return resources

    def list_resources(self, resource_type='', file_mask=''):
        """ List all resources owned by a Task instance. """
        if self.sdk1_agentr_enabled:
            from sandbox.sdk2.internal import common as internal_common
            qargs = {"task_id": self.id}
            if resource_type:
                qargs["type"] = resource_type
            query = internal_common.Query(sdk2.Task.server.resource, lambda x: x, **qargs)
            result = []
            for meta in query.limit(0):
                resource = resource_proxy.Resource._restore_from_json(meta)
                if file_mask in resource.file_name:
                    result.append(resource)
            return result
        else:
            result = manager.resource_manager.list_task_resources(self.id, resource_type)
            return [r for r in result if file_mask in r.file_name] if file_mask else result

    def invalidate_resources(self, with_hosts=False):
        """
        Mark all `NOT_READY` resources as broken.

        :param bool with_hosts: Invalidate only resources with hosts
        """
        for resource in self.list_resources():
            if resource.state == ctr.State.NOT_READY:
                if with_hosts and not resource.get_hosts(all=True):
                    # There are no hosts with this resource, don't invalidate it
                    continue
                resource.state = ctr.State.BROKEN
                resource.last_usage_time = time.time()
                resource.last_updated_time = time.time()
                manager.resource_manager.update(resource)

    def _read_resource(self, resource_id, sync=True):
        """
        Reads any resource by its id. If sync is set to True, resource's data
        will be copied to appropriate directory. If you try to sync any resource owned
        by a Task instance, an exception will be raised
        """
        logger.debug("trying to read resource %s (sync: %s) for task %s" % (resource_id, sync, self))
        res = manager.resource_manager.load(resource_id)
        if not res:
            raise common.errors.TaskError("resource %s is not loaded" % resource_id)
        logger.debug("loaded resource %s" % res)
        if sync:
            self._sync_resource(res)
        return res

    def _create_resource(
        self, resource_desc, resource_filename, resource_type, complete=0, arch=None, attrs=None, owner=None
    ):
        if resource_type not in sdk2.Resource:
            raise ValueError("Unknown resource type {!r}".format(resource_type))
        cls = sdk2.Resource[resource_type]

        if os.path.isabs(resource_filename):
            relpath = os.path.relpath(resource_filename, self.abs_path())
            if not relpath.startswith(os.path.pardir):
                resource_filename = relpath

        if self.sdk1_agentr_enabled:
            res_data = self.agentr.resource_register(
                resource_filename, str(resource_type), resource_desc,
                arch=arch or ctm.OSFamily.ANY,
                attrs=attrs, share=cls.share, service=issubclass(cls, sdk2.ServiceResource),
                resource_meta=cls.__getstate__()
            )
            resource = resource_proxy.Resource._restore_from_json(res_data)

            if complete:
                self.agentr.resource_complete(resource.id)
                resource = resource_proxy.Resource._restore_from_json(sdk2.Task.server.resource[resource.id][:])

            if self.important:
                resource.attrs[ctr.ServiceAttributes.TTL] = "inf"
                sdk2.Resource[resource.id].ttl = "inf"
        else:
            resource = resource_proxy.Resource._restore(
                controller.Resource.create(
                    resource_desc, resource_filename, "", str(resource_type), self.id, cls.__getstate__(), self,
                    state=mapping.Resource.State.NOT_READY,
                    arch=arch or ctm.OSFamily.ANY,
                    attrs=attrs
                )
            )

            if complete:
                resource.mark_ready()

            # if task mark as important then do not delete resource
            if self.important:
                resource.attrs[ctr.ServiceAttributes.TTL] = "inf"
                manager.resource_manager.update(resource)

        return resource

    def _register_dep_resource(self, resource_id):
        logging.debug("Registering resource(s) %r dependency for task #%s", resource_id, self.id)
        for resource_id in common.utils.chain(resource_id):
            rs = manager.resource_manager.load(resource_id)
            if rs is None:
                raise common.errors.TaskError("incorrect resource id: %s" % resource_id)
            if self.id == rs.task_id:
                raise common.errors.TaskError("Can't register resource %s to the same task %s" % (rs, self))
            manager.task_manager.register_dep_resource(self.id, resource_id)

    def _get_log_url(self, log_name):
        try:
            current_log = manager.resource_manager.list_task_resources(
                self.id,
                resource_type=sdk2.service_resources.TaskLogs,
                limit=1
            )[0]
            if current_log.state != ctr.State.READY:
                return
            return '/'.join((current_log.proxy_url, log_name))
        except:
            return

    def get_common_log_view(self):
        url = self._get_log_url(ctt.LogName.COMMON)
        if not url:
            return
        return {'url': url, "name": ctt.LogName.COMMON}

    def get_debug_log_view(self):
        url = self._get_log_url(ctt.LogName.DEBUG)
        if not url:
            return
        return {'url': url, "name": ctt.LogName.DEBUG}

    def initLogger(self):
        """ Set root logger """
        if self.taskLogger is None and self.id:
            settings = common.config.Registry()
            self.taskLogger = logging.getLogger()
            self.taskLogger.propagate = False
            log_dir = self.log_path()
            if os.path.exists(log_dir):
                common.fs.chmod_for_path(log_dir, 'a+w', recursively=True)
            else:
                if os.path.exists(self.abs_path()):
                    common.fs.chmod_for_path(self.abs_path(), 'a+w', recursively=True)
                os.makedirs(log_dir, 0o755)
            handler_common = logging.FileHandler(self.log_path(ctt.LogName.COMMON))
            handler_common.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
            handler_common.addFilter(common.log.TimeDeltaMeasurer())
            handler_common.setLevel(logging.INFO)
            self.taskLogger.addHandler(handler_common)
            self.taskLogger.setLevel(getattr(logging, settings.client.executor.log.level))

    def set_status(
        self, new_status, event=None, request=None, lock_host="", force=False, expected_status=None,
        set_last_action_user=False
    ):
        """
        Set status to new_status

        :param new_status:      New task status
        :param event:           Event that cause status changing
        :param request:         SandboxRequest object
        :param lock_host:       New lock value
        :param force:           Check the new status is applicable
        :param expected_status: Current task status, use to prevent conflict
        """
        if self.status == new_status:
            return

        if expected_status and self.status != expected_status:
            raise common.errors.UpdateConflict(
                "Expected task status: {}, but current is {}".format(expected_status, self.status)
            )

        if not self.Status.can_switch(self.status, new_status) and not force:
            raise common.errors.IncorrectStatus(
                "Cannot switch task #{} status from '{}' to '{}'".format(self.id, self.status, new_status)
            )
        logger.info("Switch task #%s status from %r to %r (lock %r)", self.id, self.status, new_status, lock_host)
        now = dt.datetime.utcnow()
        kws = dict(
            set__execution__status=new_status,
            set__lock_host=lock_host,
            set__time__updated=now,
        )
        if lock_host:
            kws["set__lock_time"] = now

        if new_status in self.Status.Group.FAIL_ON_ANY_ERROR and self.ctx.get("fail_on_any_error"):
            event = event + ". " if event else ""
            event += "Switched to {} instead of {}".format(self.Status.FAILURE, new_status)
            new_status = self.Status.FAILURE
            kws["set__execution__status"] = new_status

        if new_status in it.chain(
            (self.Status.TEMPORARY, self.Status.SUCCESS, self.Status.FAILURE),
            self.Status.Group.WAIT, self.Status.Group.BREAK,
        ):
            kws["set__execution__time__finished"] = now

        if set_last_action_user:
            if not request or user_controller.user_has_right_to_act_on_behalf_of(request.user, self.author):
                if self.last_action_user is None:
                    kws.update(set__last_action_user=self.author)
            else:
                kws.update(set__last_action_user=request.user.login)

        task = mapping.Task.objects.with_id(self.id)
        task_status_events_ids = tse_controller.TaskStatusEvent.create(task, new_status)
        kws.update(set__status_events=task_status_events_ids)

        if mapping.Task.objects(id=self.id, execution__status=self.status).update_one(**kws):
            if self._mapping.requirements.semaphores and (
                new_status in ctt.Status.Group.expand(self._mapping.requirements.semaphores.release or ())
            ):
                exc = None
                # noinspection PyBroadException
                try:
                    semaphores_released = controller.TaskQueue.qclient.release_semaphores(
                        self.id, self.status, new_status
                    )
                except Exception as exc:
                    semaphores_released = None
                if not semaphores_released:
                    err_msg = "Cannot release semaphore(s) for task #{} ({} -> {})".format(
                        self.id, self.status, new_status
                    )
                    if semaphores_released is None:
                        err_msg = "{}: {}".format(err_msg, exc)
                    logger.error(err_msg)
        else:
            old_status = self.status
            self.reload()
            raise common.errors.UpdateConflict(
                "Cannot switch task #{} status from '{}' to '{}': conflict on update. Current task status: {}".format(
                    self.id, old_status, new_status, self.status
                )
            )

        controller.Task.audit(mapping.Audit(task_id=self.id, status=new_status, content=event))
        self.status = new_status
        self.updated = time.time()
        self.lock_host = lock_host

        if new_status in (self.Status.SUSPENDING, self.Status.SUSPENDED):
            old_state = self.SessionState.ACTIVE
            new_state = self.SessionState.SUSPENDED
        else:
            old_state = self.SessionState.SUSPENDED
            new_state = self.SessionState.ACTIVE

        # Switch from ACTIVE to SUSPENDED or vice versa
        mapping.OAuthCache.objects(task_id=self.id, state=str(old_state)).update(set__state=str(new_state))

        if new_status == self.Status.WAIT_TASK:
            mapping.TaskStatusTrigger.objects(source=self.id).update(set__activated=True)

    def set_info(self, info, do_escape=True):
        """
        Adds text into field 'info'

        :param info: test to add
        :param do_escape: if True, escape text before add
        :return: added text
        """
        logger.info('Task #%s info added: %s', self.id, info)
        info = common.utils.force_unicode_safe(info)

        if not self.info:
            self.info = u''
        if do_escape:
            info = common.utils.escape(info)
        self.info += u'<div class="hr">{0}</div>{1}\n'.format(common.utils.dt2str(dt.datetime.now()), info)
        return self.info

    def user_has_permission(self, user):
        return user_controller.user_has_permission(user, (self.author, self.owner))

    def delete(self, event=None, request=None):
        """
        Delete task

        :param event: string with description of event that cause deleting
        :param request: SandboxRequest object
        :return: return code: 1 if task was not deleted, 0 otherwise
        :rtype: int
        """
        if request and not self.user_has_permission(request.user):
            raise common.errors.AuthorizationError(
                "User '{}' is not permitted to delete task #{} (author: '{}', owner: '{}')".format(
                    request.user.login, self.id, self.author, self.owner
                )
            )

        if self.status == self.Status.DELETED:
            return True

        if self.status == self.Status.DRAFT:
            self.set_status(new_status=self.Status.DELETED, event=event, request=request)
            return True

        if not self.Status.can_switch(self.status, self.Status.DELETED):
            return False

        if self.status == self.Status.RELEASED and not request:
            raise common.errors.AuthorizationError(
                "Released task #{} can only be deleted within authorized session scope.".format(self.id)
            )

        session = mapping.OAuthCache.objects(task_id=self.id).first()
        if session:
            user_controller.OAuthCache.abort(session, reason=self.Status.DELETED)

        # Drop all wait triggers, don't need them in DELETED
        trigger_controller.delete_wait_triggers(self.id)

        controller.TaskWrapper.before_task_delete(self.id)
        self.set_status(new_status=self.Status.DELETED, event=event, request=request, force=True)

        self.lock_host = ""
        self.hidden = 1
        manager.task_manager.update(self)
        return True

    def stop(self, event=None, request=None):
        """
        Stop task execution

        :param event: string with description of event that cause stop
        :param request: SandboxRequest object
        :return: True if task was successfully stopped, False otherwise
        """
        if self.status in (self.Status.STOPPING, self.Status.STOPPED):
            return True
        if not self.Status.can_switch(self.status, self.Status.STOPPING):
            return False

        if request and not self.user_has_permission(request.user):
            raise common.errors.AuthorizationError(
                "User '{}' is not permitted to stop task #{} (author: '{}', owner: '{}')".format(
                    request.user.login, self.id, self.author, self.owner
                )
            )

        session = mapping.OAuthCache.objects(task_id=self.id).first()
        if session:
            user_controller.OAuthCache.abort(session, reason=self.Status.STOPPING)

        # Drop all wait triggers, task can't be triggered in STOPPED state.
        trigger_controller.delete_wait_triggers(self.id)

        for subtask in controller.TaskWrapper.fast_load_list(self.list_subtasks()):
            subtask.stop(event)

        self.set_status(new_status=self.Status.STOPPING, event=event, request=request, set_last_action_user=True)
        self.save()

        if session:
            self.invalidate_resources(with_hosts=True)

        return True

    def suspend(self, event=None, request=None):
        """
        Suspend task execution

        :param event: string with description of event that cause suspend
        :param request: SandboxRequest object
        """
        if request and not self.user_has_permission(request.user):
            raise common.errors.AuthorizationError(
                "User {!r} is not permitted to suspend task #{} (author: {!r}, owner: {!r})".format(
                    request.user.login, self.id, self.author, self.owner
                )
            )
        self.set_status(
            new_status=self.Status.SUSPENDING,
            event="Suspend self" if request.session else event,
            request=request,
        )
        if request.session:
            logger.info("Ensure task #%s author will receive appropriate notification.", self.id)
            notifications = mapping.Task.objects.scalar("notifications").with_id(self.id)
            if not any(_ for _ in notifications if self.Status.SUSPENDED in _.statuses):
                trigger_controller.TaskStatusNotifierTrigger.append(
                    self.id,
                    manager.notification_manager.notification(
                        ctn.Transport.EMAIL,
                        self.Status.SUSPENDED,
                        self.owner if request.user.robot else self.author
                    )
                )
        return True

    def resume(self, event=None, request=None):
        """
        Resume execution of the suspended task

        :param event: string with description of event that cause resume
        :param request: SandboxRequest object
        """
        if request and not self.user_has_permission(request.user):
            raise common.errors.AuthorizationError(
                "User {!r} is not permitted to resume task #{} (author: {!r}, owner: {!r})".format(
                    request.user.login, self.id, self.author, self.owner
                )
            )
        if self.status != self.Status.SUSPENDED:
            raise common.errors.IncorrectStatus("Cannot resume not suspended task #{}".format(self.id))
        self.set_status(
            new_status=self.Status.RESUMING,
            event=event,
            request=request,
        )
        return True

    def _mark_resources(self, mark_as_ready=None):
        """
        Marks task's resources on task termination. Service resources will always be marked as "READY" if possible.
        :param mark_as_ready:   Mark non-services resources as "READY" if `True`, "BROKEN" if `False`
                                and skip if `None`
        """
        if self.sdk1_agentr_enabled:
            self.agentr.finished(drop_session=False, mark_as_ready=mark_as_ready)
        else:
            logger.debug("Marking task %s resources as %s", self, mark_as_ready)
            skip_non_service = mark_as_ready is None
            failed_resources = []

            for resource in self.list_resources():
                if resource.state != mapping.Resource.State.NOT_READY:
                    # Leave READY/BROKEN resources alone regardless of planned action
                    continue

                try:
                    if mark_as_ready or resource.type.service:
                        resource.mark_ready()
                        continue
                except Exception as ex:
                    (logger.warning if resource.type.service else logger.error)(
                        "Cannot mark resource #%s as ready: %s", resource.id, ex
                    )
                    if not (resource.type.service or skip_non_service):
                        failed_resources.append(str(ex))

                if skip_non_service:
                    continue

                resource.mark_broken()

            if failed_resources:
                raise common.errors.TaskError("; ".join(failed_resources))

    def on_failure(self):
        """ Called when switching task to status FAILURE """
        self._mark_resources(False)

    def on_success(self):
        """ Called when switching task to status SUCCESS """
        self._mark_resources(True)

    def on_break(self):
        """ Called when switching to statuses group BREAK """
        self._mark_resources(False)

    def on_wait(self):
        """ Called when switching to statuses group WAIT """
        self._mark_resources(None)

    @property
    def sdk1_agentr_enabled(self):
        """
        Returns True if registration of SDK1 resources in agentr is enabled
        """
        return isinstance(manager.resource_manager, manager.ManagerDispatchWrapper)

    def _check_subprocess_timeout(self, p, timeout, on_timeout=None, exceptionType=Exception):
        start_time = time.time()
        while p.poll() is None:
            if (time.time() - start_time) > timeout:
                try:
                    if on_timeout:
                        on_timeout(p)
                finally:
                    try:
                        p.terminate()
                        if p.poll() is None:
                            time.sleep(5)
                            p.kill()
                    except OSError:
                        pass
                raise exceptionType('Process was interrupted by timeout: %s' % timeout)
            time.sleep(1)

    def _subprocess_ex(
        self,
        cmd,
        wait=False,
        wait_timeout=0,
        check=True,
        stdin=None,
        extra_env=None,
        exceptionType=common.errors.SubprocessError,
        on_timeout=None,
        stdout=None,
        stderr=None,
        set_pgrp=False,
        shell=False
    ):
        """ Runs subprocess """

        if not extra_env:
            extra_env = {}

        if type(cmd) is unicode:
            cmd = cmd.encode('utf-8')

        if type(cmd) is str:
            cmd = shlex.split(cmd)

        if wait_timeout:
            logger.debug('execute subprocess command with timeout %s: %s' % (wait_timeout, ' '.join(cmd)))
        else:
            logger.debug('execute subprocess command: %s' % ' '.join(cmd))

        subp_args = {
            'stdin': stdin,
            'stdout': stdout,
            'stderr': stderr,
        }
        if extra_env:
            env = {}
            env.update(os.environ)
            for k in extra_env:
                env[k] = extra_env[k]
            subp_args['env'] = env

        if set_pgrp:
            def _preexec():
                import signal

                os.setpgrp()
                signal.signal(signal.SIGPIPE, signal.SIG_DFL)

            subp_args['preexec_fn'] = _preexec

        subp_args['close_fds'] = True

        if shell:
            cmd2 = ' '.join(cmd)
        else:
            cmd2 = cmd
        p = subprocess.Popen(cmd2, shell=shell, **subp_args)
        p.rpid = sdk2.Task.current.agentr.register_process(p.pid)

        # XXX: monkey patch subprocess
        p.saved_cmd = ' '.join(cmd)

        self._subproc_list.append(p)

        if wait_timeout:
            self._check_subprocess_timeout(p, wait_timeout, on_timeout, exceptionType)
        elif wait:
            p.wait()

        if (wait or wait_timeout) and check:
            if p.returncode:
                raise exceptionType('process "%s" died with exit code %s' % (p.saved_cmd, p.returncode))
        return p

    def _get_log_file_name(self, fname, rewrite_log):
        """ Creates file name for log """
        if not os.path.exists(fname) or rewrite_log:
            return fname

        i = 1
        while 1:
            tmp_fname = '%s.%s' % (fname, i)
            if not os.path.exists(tmp_fname):
                return tmp_fname
            i += 1

    def _subprocess(
        self,
        cmd,
        wait=False,
        wait_timeout=0,
        check=True,
        out_file_name='',
        log_prefix='',
        err_log_prefix='',
        rewrite_log=False,
        extra_env=None,
        exceptionType=common.errors.SubprocessError,
        on_timeout=None,
        set_pgrp=False,
        inp_file_name=None,
        shell=False,
        keep_filehandlers=False
    ):
        """
        Opens log files and runs subprocess

        :param on_timeout: callback function with one argument - process object,
            called if process not finished after 'wait_timeout' seconds
        """
        if not extra_env:
            extra_env = {}

        # files to open for subprocess
        log_files = {
            'stdout': None,
            'stderr': None,
        }

        # create stdout log
        if out_file_name:
            f = out_file_name
            log_files['stdout'] = self._get_log_file_name(
                f, rewrite_log)
        elif log_prefix:
            f = '%s.out.txt' % log_prefix
            log_files['stdout'] = self._get_log_file_name(
                self.log_path(f), rewrite_log)

        # create stderr log
        if err_log_prefix:
            f = '%s.err.txt' % err_log_prefix
            log_files['stderr'] = self._get_log_file_name(
                self.log_path(f), rewrite_log)
        elif log_prefix:
            f = '%s.err.txt' % log_prefix
            log_files['stderr'] = self._get_log_file_name(
                self.log_path(f), rewrite_log)

        # open all logs
        for attr, fname in log_files.iteritems():
            if not fname:
                continue
            try:
                log_files[attr] = open(fname, 'w')
            except IOError as e:
                logger.error('Error while opening "%s" for %s: %s' % (fname, attr, e))
                log_files[attr] = None

        stdin = file(inp_file_name, "rb") if inp_file_name else None

        p = self._subprocess_ex(
            cmd=cmd,
            wait=wait,
            wait_timeout=wait_timeout,
            check=check,
            extra_env=extra_env,
            exceptionType=exceptionType,
            on_timeout=on_timeout,
            set_pgrp=set_pgrp,
            stdin=stdin,
            shell=shell,
            **log_files
        )
        for attr in ('stdout', 'stderr') if keep_filehandlers else []:
            fh = log_files.get(attr)
            if fh and not getattr(p, attr):
                setattr(p, attr, fh)
        return p

    def cleanup(self):
        self.agentr.cleanup_work_directory()

    def remote_path(self):  # run only on front server
        if not self.host:
            return ''
        remote_dir = mapping.Client.objects.with_id(self.host).info.get("system", {}).get("tasks_dir", "")
        return os.path.join(remote_dir, self.local_path())

    @property
    def execution_dir_url(self):
        if not self.host:
            return None
        settings = common.config.Registry().client.fileserver
        return (
            "{}://{}/task/{}".format(settings.proxy.scheme.http, settings.proxy.host, self.id)
            if settings.proxy.host else
            self.remote_http("")
        )

    def remote_http(self, path):
        if not self.host:
            return ''
        http_prefix = mapping.Client.objects.with_id(self.host).info.get('system', {}).get('fileserver', '')
        if not http_prefix:
            return ''
        r = urlparse.urlparse(http_prefix)
        if 'yandex.' not in r.netloc:
            if ':' in r.netloc:
                h, p = r.netloc.split(':')
                r = r._replace(netloc='%s.yandex.ru:%s' % (h, p))
            else:
                r = r._replace(netloc='%s.yandex.ru' % r.netloc)
            http_prefix = r.geturl()
        full_url = urlparse.urljoin(http_prefix, self.local_path(path))
        return full_url

    def http_url(self):
        return common.utils.get_task_link(self.id)

    def to_dict(self, safe_xmlrpc=False):
        d = {}
        for attr in self.SERIALIZE_FIELDS:
            val = getattr(self, attr, None)
            if attr[0] != '_' and not callable(val):
                d[attr] = copy.deepcopy(val)
        d['url'] = self.http_url()
        d['type_name'] = self.type
        d['priority'] = d['priority'].__getstate__()
        if d['parent_id']:
            d['parent_id'] = str(d['parent_id'])
        if d.get('info'):
            d['info'] = d['info'].encode('utf-8', 'replace')
        if safe_xmlrpc:
            ctx = d['ctx']
            Task.check_ctx(ctx)
            d['ctx'] = json.dumps(ctx)
            if d.get("info"):
                d['info'] = common.proxy.convert_unicode_to_safe_xml(d['info'])
        d['client_tags'] = str(self.client_tags)
        return d

    def from_dict(self, params):
        ignore_keys = {'type_name', 'owner', 'author', 'ctx'}
        if not params.get('descr'):
            params['descr'] = ''
        if 'ctx' in params:
            self.ctx.update({k: v for k, v in params['ctx'].iteritems() if not k.startswith('__')})
            for task_ctx_param_name, task_param_name in (
                ("__hosts_chooser_os", "arch"),
                ("__hosts_chooser_cpu_model", "model"),
                ("__hosts_chooser_hosts", "required_host"),
            ):
                value = self.ctx.pop(task_ctx_param_name, None)
                if value:
                    setattr(self, task_param_name, value)
        client_tags = params.pop("client_tags", "")
        self.client_tags = ctc.Tag.Query(client_tags) if client_tags else self.__class__.client_tags
        self.__dict__.update({k: v for k, v in params.iteritems() if k not in ignore_keys})

    def __container_resource(self, default):
        for param in self.input_parameters:
            if issubclass(param, sdk_legacy.Container):
                container = self.ctx.get(param.name)
                return int(container) if container else None
        return default

    def mapping(self, release=False):
        mp = self._mapping
        if not mp:
            mp = self._mapping = Model() if not self.id else Model.objects.with_id(self.id)

        if self.id:
            mp.id = self.id
        mp.type = self.type
        if self.parent_id:
            mp.parent_id = self.parent_id
        if self.scheduler:
            mp.scheduler = self.scheduler
        mp.suspend_on_status = self.suspend_on_status
        mp.score = self.score
        mp.description = self.descr
        if mp.context or self.ctx:
            self.check_ctx(self.ctx)  # raises on error
            mp.context = str(cPickle.dumps(self.ctx))
        mp.hidden = bool(self.hidden)
        mp.se_tag = self.se_tag
        mp.tags = [tag.upper() for tag in self.tags if ctt.TaskTag.test(tag)]
        mp.owner = self.owner
        mp.author = self.author
        mp.priority = int(self.priority)
        mp.flagged = bool(self.important)
        mp.tcpdump_args = self.tcpdump_args
        mp.max_restarts = self.max_restarts
        mp.enable_yav = self.__class__.enable_yav
        if not mp.time:
            mp.time = Model.Time()
            mp.requirements = Model.Requirements()
            mp.execution = Model.Execution()
            mp.execution.time = Model.Execution.Time()

        mp.requirements.platform = self.arch
        mp.requirements.cpu_model = self.model
        mp.requirements.host = self.required_host
        mp.requirements.disk_space = self.execution_space or self.__class__.execution_space
        mp.requirements.ram = self.required_ram
        mp.requirements.ramdrive = (
            Model.Requirements.RamDrive(type=self.ramdrive.type, size=self.ramdrive.size)
            if self.ramdrive else
            None
        )
        mp.requirements.cores = self.cores
        mp.requirements.privileged = self.privileged
        mp.requirements.dns = self.dns
        mp.requirements.client_tags = str(self.client_tags)
        mp.requirements.caches = []

        # update container in parameters if not set [SANDBOX-7051]
        if mp.requirements.container_resource:
            for param in self.input_parameters:
                if issubclass(param, sdk_legacy.Container):
                    container = self.ctx.get(param.name)
                    if not container:
                        self.ctx[param.name] = mp.requirements.container_resource
                    break

        # update container in requirements [SANDBOX-7051]
        mp.requirements.container_resource = self.__container_resource(mp.requirements.container_resource)
        mp.fail_on_any_error = bool(self.ctx.get("fail_on_any_error", False))
        mp.dump_disk_usage = bool(self.ctx.get("dump_disk_usage", True))

        if self.tasks_archive_resource:
            mp.requirements.tasks_resource = mapping.Task.Requirements.TasksResource(
                id=int(self.tasks_archive_resource), taskbox_enabled=False, age=0
            )
        else:
            mp.requirements.tasks_resource = None

        mp.kill_timeout = self.ctx.get("kill_timeout")

        if isinstance(mp, mapping.Task):
            if not mp.execution.disk_usage:
                mp.execution.disk_usage = Model.Execution.DiskUsage()
            if mp.execution.auto_restart is None:
                mp.execution.auto_restart = Model.Execution.AutoRestart()
            if self.info:
                mp.execution.description = self.info
            if self.timestamp_start:
                mp.execution.time.started = dt.datetime.utcfromtimestamp(self.timestamp_start)
            if self.timestamp_finish:
                old_value = mp.execution.time.finished
                new_value = dt.datetime.utcfromtimestamp(self.timestamp_finish)
                if not old_value or new_value > old_value:
                    mp.execution.time.finished = new_value
            if self.release_params:
                mp.execution.release_params = self.release_params
            if self.host is not None:
                mp.execution.host = self.host
            if release:
                mp.lock_host = None  # Magic, to force field changed flag set.
                mp.lock_host = self.lock_host = ""
            elif mp.lock_host is None:
                mp.lock_host = self.lock_host
            else:
                try:
                    mp._changed_fields.remove("lock")
                except (ValueError, AttributeError):
                    pass

            if mp.execution.status is None:
                mp.execution.status = self.status
            else:
                try:
                    mp.execution._changed_fields.remove("st")
                except ValueError:
                    pass

        return mp

    def save(self, release=False, save_condition=None):
        op, kws = "Updated", {"save_condition": save_condition}
        mp = self.mapping(release)
        if mp.is_new:
            op = "Created"
            kws = {
                "force_insert": True,
                "write_concern": common.config.Registry().server.mongodb.write_concern
            }
        mp.save(**kws)
        self.id = mp.id
        logger.info(
            "%s task #%s of type '%s', status: '%s', host: '%s'/'%s', req: %s",
            op, mp.id, mp.type, mp.execution.status, mp.execution.host, mp.lock_host, mp.requirements.to_json()
        )
        return self

    def reload(self):
        # Do now forget to protect against unit tests' fake task objects
        if not self._mapping:
            return None
        self._mapping.reload()
        self.restore(self._mapping, self=self)
        return self

    @staticmethod
    def restore(mp, cls=None, self=None):
        if not mp:
            return None
        if not self:
            cls = getTaskClass(mp.type) if cls is None else cls
            self = cls(mp.id)

        self._mapping = mp

        if isinstance(mp, mapping.Task):
            self.parent_id = mp.parent_id
            self.last_action_user = mp.last_action_user
            self.lock_host = mp.lock_host
            self.scheduler = mp.scheduler
            self.suspend_on_status = mapping.to_list(mp.suspend_on_status)
            self.score = mp.score
            self.client_tags = (
                ctc.Tag.Query(mp.requirements.client_tags)
                if mp.requirements.client_tags else
                self.__class__.client_tags
            )
            self.notifications = [
                {
                    "statuses": mapping.to_list(notification.statuses),
                    "transport": notification.transport,
                    "recipients": mapping.to_list(notification.recipients),
                    "check_status": notification.check_status,
                    "juggler_tags": mapping.to_list(notification.juggler_tags)
                } for notification in mp.notifications
            ] if mp.notifications else []
            if mp.time:
                self.timestamp = calendar.timegm(mp.time.created.timetuple())
                self.updated = calendar.timegm(mp.time.updated.timetuple())
            if mp.execution:
                if mp.execution.last_execution_start:
                    self.timestamp_start = calendar.timegm(mp.execution.last_execution_start.timetuple())
                if mp.execution.last_execution_finish:
                    self.timestamp_finish = calendar.timegm(mp.execution.last_execution_finish.timetuple())
                self.info = mp.execution.description
                self.host = mp.execution.host
                self.status = mp.execution.status
                self.release_params = mp.execution.release_params

        self.owner = mp.owner
        self.descr = mp.description
        self.type = mp.type
        self.author = mp.author
        self.hidden = mp.hidden
        self.se_tag = mp.se_tag
        self.tags = mapping.to_list(mp.tags) or []
        if mp.max_restarts is not None:
            self.max_restarts = mp.max_restarts
        self.priority.__setstate__(mp.priority)
        self.important = mp.flagged
        self.tcpdump_args = mp.tcpdump_args
        if mp.context:
            try:
                self.ctx.update(cPickle.loads(mp.context))
            except ImportError as ex:
                self.ctx.update({'ctx_load_error': str(ex)})

        self.model = mp.requirements.cpu_model
        self.arch = mp.requirements.platform
        self.required_host = mp.requirements.host
        self.execution_space = mp.requirements.disk_space
        self.required_ram = mp.requirements.ram
        if mp.requirements.dns:
            self.dns = mp.requirements.dns
        self.ramdrive = (
            ctm.RamDrive(mp.requirements.ramdrive.type, mp.requirements.ramdrive.size, None)
            if mp.requirements.ramdrive else
            None
        )
        self.cores = mp.requirements.cores

        return self

    def remote_copy(
        self,
        src,
        dst,
        protocol=None,
        unstable=None,
        exclude=None,
        timeout=None,
        create_resource=False,
        resource_type=None,
        resource_descr=None,
        resource_arch=ctm.OSFamily.ANY,
        resource_id=None,
        write_to_task_logs=True,
    ):
        if unstable is None:
            unstable = []
        if exclude is None:
            exclude = []
        if timeout is None:
            timeout = self._download_timeout
        if protocol is None:
            protocol = 'skynet'
            if ':' in src:
                temp = src.split(':')[0]
                if temp == 'rsync':
                    protocol = 'rsync'
                elif temp == 'svn+ssh':
                    protocol = 'svn'
                elif temp == 'arcadia':
                    protocol = 'arcadia'
                elif temp in ['http', 'https']:
                    protocol = 'http'

        #############
        if protocol in ('rcp', 'scp'):
            self._subprocess(
                '{} -r {} {}'.format(protocol, src, dst),
                wait_timeout=timeout,
                log_prefix='remote_copy' if write_to_task_logs else '',
            )

        elif protocol == 'http':
            cmd = {
                'linux': 'wget -T 60 -O {} {}',
                'darwin': 'curl -m 60 -o {} {}',
                'freebsd': 'fetch -T 60 -o {} {}',
            }[platform.uname()[0].lower()]
            self._subprocess(
                cmd.format(dst, src),
                wait=True,
                log_prefix='remote_copy' if write_to_task_logs else '',
            )

        elif protocol == 'rsync':
            try:
                # This actually a hack to load skynet.copier's egg.
                # In case it will be broken, the code will "silently" fallback to non-fasbonized mode.
                import api.copier
                api.copier.Copier()
                try:
                    import ya.skynet.services.copier.client.utils as copier_tools
                except ImportError:
                    import ya.skynet.services.skybone.client.utils as copier_tools
                urls = [src, copier_tools.fastbonizeURL(src)]
            except Exception as ex:
                logging.error("Unable to load skynet.copier's utility module: %s", str(ex))
                urls = [src] * 2

            if urls[0] == urls[1]:
                urls.pop()
            while urls:
                url = urls.pop()
                log_suffix = str(resource_id) if resource_id else '{:#x}'.format(random.randrange(0xFFFFFFFF))
                try:
                    cmd = [
                        'rsync',
                        '-vvv',             # set pretty big verbosity level
                        '--checksum',       # skip based on checksum, not mod-time & size
                        '--out-format=[%t] %i %n%L',  # set custom output format
                        '--progress',       # show progress during transfer
                        '--recursive',      # recurse into directories
                        '--partial',        # keep partially transferred files
                        '--links',          # copy symlinks as symlinks
                        '--perms',          # preserve permissions
                        '--group',          # preserve group
                        '--chmod=+w',       # affect file and/or directory permissions
                        '--inplace',        # update destination files in-place
                        '--bwlimit=75000',  # limit I/O bandwidth; KBytes per second
                        '--timeout={}'.format(timeout),  # set I/O timeout in seconds
                    ]
                    if common.config.Registry().this.system.family not in ctm.OSFamily.Group.OSX:
                        cmd.append('--contimeout=60')  # set daemon connection timeout in seconds
                    if exclude:
                        for exc in exclude:
                            cmd.append("--exclude")
                            cmd.append(exc)
                    cmd += [url, dst]

                    p = self._subprocess(
                        cmd,
                        wait_timeout=timeout,
                        keep_filehandlers=True,
                        log_prefix='remote_copy.{}'.format(log_suffix) if write_to_task_logs else '',
                    )
                    if p and p.stdout:
                        with open(p.stdout.name, 'r') as fh:
                            logging.info(
                                "Rsync #%s summary:\n%s",
                                log_suffix,
                                '\n'.join(list(common.fs.tail(fh, 4))[:2])
                            )
                        p.stdout.close()
                        p.stderr.close()
                        os.remove(p.stdout.name)
                    break
                except common.errors.SubprocessError:
                    if not urls:
                        raise
                    logging.exception("Error fetching data via '%s'", url)

        elif protocol == 'svn':
            self._subprocess(
                'svn export {} {}'.format(src, dst),
                wait=True,
                log_prefix='remote_copy' if write_to_task_logs else '',
            )

        elif protocol == 'arcadia':
            from sandboxsdk import svn
            svn.Arcadia.export(src, dst)

        elif protocol == 'skynet':
            # check if we get torrent id
            if src.startswith('rbtorrent:'):
                common.share.skynet_get(src, dst, timeout)
            else:
                common.share.skynet_copy(
                    src.partition(":")[0], src.partition(":")[2], dst, unstable=unstable, exclude=exclude
                )
        else:
            raise common.errors.TaskError('unknown remote file protocol {}'.format(protocol))

        ############
        if create_resource:
            # force path to be relative
            if dst.startswith(self.abs_path()):
                dst = os.path.relpath(dst, self.abs_path())
            resource = self._create_resource(
                resource_descr or self.descr,
                dst,
                resource_type,
                arch=resource_arch)
            resource.mark_ready()
            return resource

    def set_priority(self, priority):
        """
            Increase priority of child tasks for the same value
        """
        for task in self.list_subtasks(load=True):
            task.set_priority(priority)
        self.priority = priority
        mapping.Task.objects.filter(id=self.id).update(set__priority=int(priority))

    def restart(self, event=None, request=None):
        """
        Restart task execution i.e. put it to queue

        :param event: string with description of event that cause restarting
        :param request: SandboxRequest object
        :return: True, if task has been restarted; False otherwise
        """
        # reset the flag that the task was killed by timeout
        return manager.task_manager.restart_task(self, event, request=request)

    def _get_last_resource(self, resource_type, attr=None):
        res_list = manager.resource_manager.list_task_resources(
            resource_type=resource_type, state=mapping.Resource.State.READY, arch=self.arch, attr=attr, limit=1)
        if not res_list:
            raise common.errors.TaskError('Can\'t find "{0}" resource.'.format(resource_type))
        resource = res_list[0]
        self._sync_resource(resource)
        return resource

    def get_last_enqueued_timestamp(self):
        events = manager.task_manager.get_last_history_event(self.id, 'enqueued')
        if not events:
            return self.timestamp
        else:
            return calendar.timegm(events[0]['time'].timetuple())

    def check_release_permissions(self, user):
        """
        Checks rights of user to release of task, may be overridden

        :param user: user object
        :return: True, if user has permission and ability to release of task, False otherwise
        """
        return True

    def get_short_task_result(self):
        """
        Gets short task result (for example for task list)

        :return: short result as string; None, if result for task is not defined
        :rtype: str or None
        """
        return self.ctx.get('task_short_result')

    def get_results(self):
        """
        Gets full task result (fiels 'task_results' in task context),
        if context has no 'task_results' field, returns short result

        :return: full result as string; None, if result for task is not defined
        :rtype: str or None
        """
        if 'task_results' in self.ctx:
            return self.ctx['task_results']
        else:
            return self.get_short_task_result()

    def _view_short_result(self):
        return self._result_for_table_list(self.get_short_task_result)

    def _view_long_result(self):
        return self._result_for_table_list(self.get_results)

    def _result_for_table_list(self, result_fetcher):
        """
        Protected method for getting string representation of task result

        :return: short result as string
        """
        try:
            result = result_fetcher()
            if not result:
                return ''
        except:
            result = 'Error'
        return str(result)

    @staticmethod
    def check_ctx(ctx, check_unpickle=False):
        try:
            json.dumps(ctx)
        except Exception as ex:
            raise common.errors.TaskContextError("Context serialization error: {}".format(ex))

        for key, value in ctx.iteritems():
            if value is ctm.NotExists:
                ctx[key] = None

        if check_unpickle:
            ctx_checker = textwrap.dedent(
                """
                import pickle

                ctx = input()
                pickle.loads(ctx)
                """
            )
            env = os.environ.copy()
            env["PYTHONPATH"] = "/skynet"
            process = subprocess.Popen(
                [sys.executable, "-c", ctx_checker],
                env=env,
                stderr=subprocess.PIPE,
                stdin=subprocess.PIPE,
            )

            pout, perr = process.communicate(input=repr(cPickle.dumps(ctx)))
            retcode = process.poll()
            if retcode:
                raise common.errors.TaskContextError("Context serialization error: {}".format(str(perr)))

    @property
    def disk_usage(self):
        """
        Get task disk usage.

        :return: document with last and max disk usage values
        :rtype: Model.Execution.DiskUsage or None
        """
        return self._mapping.execution.disk_usage if self._mapping is not None else None

    @common_patterns.classproperty
    def description(cls):
        html = getattr(cls, '__htmldoc__', None)
        if html is None:
            doc = cls.__doc__ or sys.modules[cls.__module__].__doc__
            html = common.format.rst2html(textwrap.dedent(doc).strip()) if doc else ""
        # docutils converter spent about 3 seconds to convert all tasks' descriptions into HTML, so cache them globally.
        cls.description = html
        return html

    def semaphores(self, semaphores):
        semaphores = (
            ctt.Semaphores(**semaphores)
            if isinstance(semaphores, dict) else
            ctt.Semaphores(*semaphores)
        )
        self._mapping.requirements.semaphores = Model.Requirements.Semaphores(**semaphores.to_dict())
