import os
import sys
import time
import logging
import importlib
import threading

from sandbox import common
import sandbox.common.types.misc as ctm


logger = logging.getLogger(__name__)


class Brigadier(common.process.Master):
    """
    The class knows and holds all the service threads, which can be executed by Sandbox.
    Singleton. Does NOT support multi-threaded initialization and forking.
    """

    @property
    def finish_on_error(self):
        return True

    def __init__(self):
        # Worker mode - if not `None`, contains `Worker` class instance.
        self.worker = None
        super(Brigadier, self).__init__(logging.getLogger('jobs.brigadier'))

    def _slaves_builder(self):
        from sandbox.services import modules
        settings = common.config.Registry()

        # in (pre)production, all new-style services are run as Samogon servants,
        # so this is only for them to work at local/test environments
        if settings.common.installation in ctm.Installation.Group.LOCAL:
            for service in modules.service_registry:
                if settings.server.services.get(service, {}).get("enabled", False):
                    yield ServiceS(service)

        yield Worker(os.getpid(), self.queue, settings)

    def start(self):
        self.init()
        super(Brigadier, self).start()

    def wakeup(self, name, reason):
        if not self.processes:
            self.logger.warn(
                "Unable to wake up '%s' service thread with reason '%s' - worker process is not running.",
                name, reason
            )
            return
        try:
            if not self.worker:
                self.logger.debug("Waking up '%s' remote service thread. Reason: %s.", name, reason)
                self.process((name, reason))
            else:
                self.worker.process((name, reason))
        except Exception as ex:
            self.logger.warning("Error waking up a service thread: %s", ex)

    def join(self, maxwait=None):
        from common import config
        super(Brigadier, self).join(config.Registry().server.autorestart.timeout * 4 / 5)


class ServiceS(common.process.Slave):
    def __init__(self, name):
        self.name = name
        pidfile = os.path.join(common.config.Registry().client.dirs.run, "service_{}.pid".format(self.name))
        super(ServiceS, self).__init__(logger, pidfile, None)

    def process(self, data):
        # Service servants cannot process any IPC messages.
        pass

    def main(self):
        import sandbox
        env = os.environ.copy()
        env["PYTHONPATH"] = env.get("PYTHONPATH", "") + ":" + os.path.dirname(os.path.dirname(sandbox.__file__))
        cmd = [sys.executable, "-m", "sandbox.services", "run", "--name", self.name]
        self.logger.info("Service '%s' started with PID #%s. execv(%r)", self.name, self.mypid, cmd)
        os.execve(sys.executable, cmd, env)


class Worker(common.process.Slave):
    def __init__(self, ppid, queue, settings):
        self.ppid = ppid
        self.stopping = False
        self.settings = settings
        # Registry for created threads.
        self.threads = {}
        self.rwlock = None

        super(Worker, self).__init__(None, os.path.join(settings.client.dirs.run, 'service.py.pid'), queue)

    def on_start(self):
        from common.log import setup_log

        import kernel.util.console

        # First of all, switch logging. To do this, clean up any loggers first.
        root = logging.getLogger()
        map(root.removeHandler, root.handlers[:])
        map(root.removeFilter, root.filters[:])

        setup_log(
            os.path.join(self.settings.server.services.log.root, self.settings.server.services.log.name),
            self.settings.server.log.level
        )

        # Now, switch our own state
        self.logger = logging.getLogger('service')
        kernel.util.console.setProcTitle('[sandbox] Services Brigadier')
        # Switch brigadier to local worker mode
        Brigadier().ppid = self.ppid

        # Be like a parent )
        self.logger.info('------GO-GO-GO------')
        logger.info('Process #%d (service threads): initializing managers and controllers.', self.pid)
        import yasandbox.manager
        import yasandbox.controller
        import yasandbox.database.mapping
        logger.info('Establishing database connection.')
        yasandbox.database.mapping.ensure_connection()
        logger.info('Initializing manager objects.')
        yasandbox.manager.initialize_locally()
        logger.info('Initializing controllers')
        yasandbox.controller.initialize()
        self.logger.info('Loading tasks code.')
        sys.path.insert(0, self.settings.client.tasks.code_dir)
        import common.projects_handler
        common.projects_handler.load_project_types()

        # initialize ZK before use
        import common.zk
        common.zk.Zookeeper(self.logger).start()

        self.rwlock = common.threading.FairRWLock()
        self.logger.info("Starting service threads. Parent's PID: %d", self.ppid)
        self.__start_threads()

        common.statistics.Signaler(common.statistics.ServerSignalHandler(), component=ctm.Component.SERVICE)

        th = threading.Thread(target=self.kamikadze)
        th.daemon = True
        th.start()

    def kamikadze(self):
        self.logger.info("Kamikadze thread started. Monitoring parent PID is #%d", self.ppid)
        while os.getppid() == self.ppid:
            time.sleep(10)
        self.logger.warn("Parent process died. Killing self.")
        os._exit(42)

    def on_stop(self):
        self.stopping = True
        self.logger.info('Stopping service process.')
        with self.rwlock.writer:
            import common.zk
            common.zk.Zookeeper().stop()

        self.logger.info('Signaling service threads.')
        [_.wakeup() for _ in self.threads.itervalues()]
        self.logger.info('Waiting for service threads.')
        [_.join() for _ in self.threads.itervalues()]
        common.statistics.Signaler().wait()

    def process(self, data):
        name, reason = data
        th = self.threads.get(name)
        if th:
            self.logger.info("Waking up '%s' service thread. Reason: %s.", name, reason)
            th.wakeup()
        else:
            self.logger.warn("'%s' service thread not found. Wakeup reason: %s.", name, reason)

    def __start_thread(self, cls, *args, **kwagrs):
        kwagrs['stopping'] = lambda: self.stopping
        kwagrs['logger'] = self.logger
        kwagrs['rwlock'] = self.rwlock
        th = self.threads[cls.__name__] = cls(*args, **kwagrs)
        th.start()

    def __start_threads(self):
        """
        Starts all the known service threads.
        """
        relimport = lambda x: importlib.import_module(x, __name__)
        ########################################
        # with ZK or Test-mode
        ########################################

        # Main core thread
        if self.settings.server.services.core.enabled:
            self.__start_thread(relimport('.core').Core)

        ########################################
        # with ZK Production-mode only
        ########################################

        # Update sandbox resources
        if self.settings.server.services.packages_updater.enabled:
            self.__start_thread(
                relimport('.update_sandbox_resources').UpdateSandboxResources
            )

        ########################################
        # without ZK Production-mode only
        ########################################

        # Update projects code
        if self.settings.server.services.packages_updater.enabled:
            self.__start_thread(
                relimport('.update_server_code').UpdateServerCode,
                ppid=self.ppid,
            )

        # Update statistics data for golovan charts
        if self.settings.server.services.statistics_updater.enabled:
            self.__start_thread(relimport('.statistics_updater').StatisticsUpdater)
