import json
from collections import defaultdict

from sandbox import common
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr

import sandbox.yasandbox.manager
import sandbox.yasandbox.services.zk
from sandbox.yasandbox.database import mapping
from sandbox.yasandbox import controller

from sandbox.sandboxsdk import environments as env


class UpdateSandboxResources(sandbox.yasandbox.services.zk.ThreadWithZK):
    """ System thread for updating of client resources with specified interval"""
    NOTIFICATION_TIMEOUT = 5

    def __init__(self, *args, **kwargs):
        self.settings = common.config.Registry()
        run_interval = self.settings.server.services.packages_updater.update_interval.meta
        self.sandbox_release_status = self.settings.server.services.packages_updater.release_status
        super(UpdateSandboxResources, self).__init__(*args, **kwargs)
        self.run_interval = run_interval
        self.logger = common.log.get_core_log("update_sandbox_resources")

    @classmethod
    def _conv_lists(cls, res):
        if res is not None:
            return {k: (list(v) if k in ("rsync", "http") else v) for k, v in res.iteritems()}

    @staticmethod
    def make_res_dict(res):
        if not res:
            return
        if isinstance(res, list):
            res = res[0]
        commit_revision = res.attrs.get("commit_revision")
        res_dict = dict(
            id=res.id,
            type_name=res.type.name,
            size=res.size,
            file_name=str(res.file_name),
            skynet_id=str(res.skynet_id),
            revision=str(commit_revision) if commit_revision else res.attrs.get("version"),
            default=res.attrs.get("default"),
        )
        res_proxy = sandbox.yasandbox.manager.resource_manager.load(res.id)
        if res_proxy:
            res_dict.update(
                rsync=map(str, res_proxy.get_rsync_links()),
                http=map(str, res_proxy.get_http_links())
            )
        return res_dict

    @staticmethod
    def sandbox_package_by_type(release_status, rtype):
        return sandbox.yasandbox.manager.resource_manager.list_task_resources(
            resource_type="SANDBOX_ARCHIVE",
            owner=common.config.Registry().common.service_group,
            state=ctr.State.READY,
            all_attrs={"released": release_status, "type": rtype},
            limit=1,
            order_by="-task_id"
        )

    @classmethod
    def _get_resource(cls, resource_name):
        return controller.Resource._get_updated_sandbox_resource(resource_name)

    @common.utils.classproperty
    def tasks_image(cls):
        return cls._get_resource("tasks_image")

    @common.utils.classproperty
    def tasks_resource(cls):
        return cls._get_resource("tasks_res")

    @common.utils.classproperty
    def ui_resource(cls):
        return cls._get_resource("ui_res")

    @common.utils.classproperty
    def docs_resource(cls):
        return cls._get_resource("docs_res")

    @classmethod
    def venv_resource(cls, client_platform):
        return controller.Resource.venv_resource(client_platform)

    @common.utils.classproperty
    def lxc_resources(cls):
        return controller.Resource.lxc_resources

    @common.utils.classproperty
    def porto_layers_resources(cls):
        return controller.Resource.porto_layers_resources

    def update_env_resources(self):
        # possible options:
        # released - only looks for released resources
        # any - looks for any resources
        # binary_compatible - allow binary compatible platforms
        # any_version - environment has no version by default, use last released resource of any version

        common_envs = [
            (env.SvnEnvironment.resource_type, ["any"]),
            (env.SandboxHgEnvironment.resource_type, ["any"]),
            (env.GDBEnvironment.resource_type, ["released", "any_version"]),
            (env.GCCEnvironment.resource_type, ["released", "any", "binary_compatible"]),
            (env.NodeJS.resource_type, ["released", "any", "any_version"]),
        ]

        # resource_type -> platform -> version -> resource_id
        result = {}

        def add_to_result(resource_type, platform, version, resource_id):
            if not result[resource_type][platform][version]:
                self.logger.debug(
                    "Adding resource #%s with type: %s, platform: %s, version: %s",
                    resource_id, resource_type, platform, version,
                )
                result[resource_type][platform][version] = resource_id

        model = mapping.Resource
        rm = sandbox.yasandbox.manager.resource_manager
        for resource_type, opts in common_envs:
            result[resource_type] = defaultdict(lambda: defaultdict(int))

            found = []
            if "released" in opts:
                found.extend(
                    model.objects(**rm.list_query(
                        resource_type=resource_type,
                        state=ctr.State.READY,
                        all_attrs={"released": ctt.ReleaseStatus.STABLE}
                    )).order_by("-id")
                )
            if "any" in opts:
                found.extend(
                    model.objects(**rm.list_query(
                        resource_type=resource_type,
                        state=ctr.State.READY,
                    )).order_by("-id")
                )
            for resource in found:
                attrs = resource.attributes_dict()
                platforms = attrs.get("platform")
                version = attrs.get("version")
                if platforms and version:
                    for pl in platforms.split(","):
                        platform = common.platform.get_platform_alias(pl)
                        add_to_result(resource_type, platform, version, resource.id)
                        if "any_version" in opts:
                            add_to_result(resource_type, platform, "", resource.id)
                        if "binary_compatible" in opts:
                            for platform_alias in common.platform.PLATFORM_ALIASES:
                                if common.platform.is_binary_compatible(platform, platform_alias):
                                    add_to_result(resource_type, platform_alias, version, resource.id)

        # mongo does not allow "." in dictionary keys, so convert to json
        self.model.context["cache"] = json.dumps(result)

    @classmethod
    def get_env_resource(cls, resource_type, platform, version):
        platform = common.platform.get_platform_alias(platform)
        obj = mapping.Service.objects.with_id(cls.__name__)
        if not obj:
            return None
        cache = obj.context.get("cache")
        if not cache:
            return None
        return json.loads(cache).get(resource_type, {}).get(platform, {}).get(version)

    def _proc(self):
        res = {}
        log_msg = []
        for rt in ("ui",):
            v = res[rt + "_res"] = self.sandbox_package_by_type(self.sandbox_release_status, rt)
            log_msg.append("{}: {}".format(rt, v[0].id if v else None))

        venv_platforms, pts = None, set()
        pt2rt = (
            ("venv_res", "SANDBOX_DEPS"),
            ("lxc_res", "LXC_CONTAINER"),
            ("porto_layers_res", "PORTO_LAYER"),
        )
        for pt, rt in pt2rt:
            for platform_alias in common.platform.PLATFORM_ALIASES:
                last_res = sandbox.yasandbox.manager.resource_manager.list_task_resources(
                    resource_type=rt,
                    owner=common.config.Registry().common.service_group,
                    state=ctr.State.READY,
                    all_attrs={
                        "released": self.sandbox_release_status,
                        "platform": platform_alias
                    },
                    limit=1,
                    order_by="-task_id"
                )
                if not last_res:
                    continue
                if venv_platforms is not None and platform_alias not in venv_platforms:
                    self.logger.warning(
                        "There's no venv for platform %r (LXC resource %s).",
                        platform_alias, last_res
                    )
                    continue
                pts.add(platform_alias)
                resources = res.setdefault(pt, [])
                resources.append((platform_alias, self.make_res_dict(last_res)))
                log_msg.append("{} for {}: {}".format(pt, platform_alias, last_res[0].id))
            venv_platforms = pts if venv_platforms is None else venv_platforms

        base_query = dict(
            owner=common.config.Registry().common.service_group,
            state=ctr.State.READY,
            limit=1,
            order_by="-task_id"
        )
        query = base_query.copy()
        query.update(dict(resource_type="SANDBOX_TASKS_ARCHIVE", attr_name="auto_deploy"))
        last_tasks_res = sandbox.yasandbox.manager.resource_manager.list_task_resources(**query)
        res["tasks_res"] = last_tasks_res
        log_msg.append("tasks_res: {}".format(last_tasks_res[0].id if last_tasks_res else None))

        query = base_query.copy()
        query.update(dict(resource_type="SANDBOX_TASKS_IMAGE", attr_name="auto_deploy"))
        last_tasks_res = sandbox.yasandbox.manager.resource_manager.list_task_resources(**query)
        res["tasks_image"] = last_tasks_res
        log_msg.append("tasks_image: {}".format(last_tasks_res[0].id if last_tasks_res else None))

        query = base_query.copy()
        query.update(dict(resource_type="SANDBOX_DOCS"))
        last_docs_res = sandbox.yasandbox.manager.resource_manager.list_task_resources(**query)
        res["docs_res"] = last_docs_res
        log_msg.append("docs: {}".format(last_docs_res[0].id if last_docs_res else None))

        if log_msg:
            self.logger.info(", ".join(log_msg))
        # SANDBOX-3867
        if self.model.context.get("resources", {}).get("lxc_res") and not res.get("lxc_res"):
            self.logger.warning(
                "Empty result returned from DB for LXC_CONTAINER resources, possibly due to switching PRIMARY."
                " Will try again later."
            )
            return
        self.model.context["resources"] = {
            k: (self.make_res_dict(v) if v and not isinstance(v[0], tuple) else v)
            for k, v in res.iteritems()
        }
        self.update_env_resources()
        self.model.save()
