import os
import time
import random
import signal

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.yasandbox import controller
from sandbox.yasandbox.services import zk
from sandbox.yasandbox.services import update_sandbox_resources

from sandbox import serviceq


class UpdateServerCode(zk.ThreadWithZK):
    """ System thread for server code updating with specified interval """
    NOTIFICATION_TIMEOUT = 5

    def __init__(self, *args, **kwargs):
        self.ppid = kwargs.pop("ppid")
        super(UpdateServerCode, self).__init__(*args, **kwargs)

        self.run_interval = 30
        self.start_time = time.time()
        self.ttl = random.randint(20, 30) * 60
        self.logger = common.log.get_core_log("update_server_code")
        self.qclient = serviceq.client.Client()

    @property
    def first_run_delay(self):
        # Always sleep typical interval on start to avoid possible server crash during startup process.
        return self.run_interval

    @staticmethod
    def read_only():
        # Do not disable thread even the Sandbox is in read-only mode.
        return False

    def _run_with_zk(self):
        # Disable any Zookeeper locking.
        return self._run()

    def _proc(self):
        try:
            if update_packages(self.qclient, self.logger):
                self.logger.info("Asking web-server with PID #%s to reload by update server code", self.ppid)
                os.kill(self.ppid, signal.SIGUSR1)
            elif (
                time.time() - self.start_time > self.ttl and
                common.config.Registry().common.installation in ctm.Installation.Group.NONLOCAL
            ):
                self.logger.info("Asking web-server with PID #%s to reload by ttl %s", self.ppid, self.ttl)
                os.kill(self.ppid, signal.SIGUSR1)
        except common.package.PackageOutdatedError:
            self.logger.error("Update failed, current package is outdated")
            controller.Notification.report_service_error(
                comment="Update failed, current package is outdated. Shutting down the service",
                thread_name=self.__class__.__name__
            )
            os.kill(self.ppid, signal.SIGTERM)


def get_package_resource(package_name):
    res = None
    usr = update_sandbox_resources.UpdateSandboxResources

    if package_name == "tasks":
        res = usr.tasks_resource
    elif package_name == "docs":
        res = usr.docs_resource
    elif package_name == "ui":
        res = usr.ui_resource

    if res is None:
        raise Exception("Unknown package: {}".format(package_name))

    return res


def update_packages(qclient=None, logger=None):
    if qclient is None:
        qclient = serviceq.client.Client()

    updater = common.package.PackageUpdater(logger)
    # static resources, no need to restart after update
    updater.update_package("docs", get_package_resource("docs"))
    # update code, restart in case of successful update
    nodes = [node.split(":", 1)[0] for node in qclient.contenders()]
    return updater.update_tasks_with_node_id_check(get_package_resource("tasks"), nodes)
