#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import json
import html

import msal
from vault_client.instances import Production as VaultClient
from vault_client.errors import ClientError as VaultClientError
from flask import Flask, session, request, render_template, redirect, Markup

TENANT = os.environ.get('TENANT', 'yandexteam.onmicrosoft.com')
APP_ID = os.environ.get('APP_ID', 'bb6b77ff-7b38-4d38-a2fe-7fcb3ebee61e')   # (sib-authorize-app) - registered Application (client) ID used for OAuth application
REDIRECT_URI = os.environ['REDIRECT_URI']                                   # 'http://127.0.0.1:8080/getToken'
FLASK_SECRET_KEY = os.environ['FLASK_SECRET_KEY']
ROBOT_PREFIX = os.environ.get('ROBOT_PREFIX', 'robot-sp-')

MY_ROBOT = os.environ['MY_ROBOT']
MY_ROBOT_UID = os.environ['MY_ROBOT_UID']
MY_ROOT_YAV_TOKEN = os.environ['MY_ROOT_YAV_TOKEN']

TEST_ROBOT = 'robot-msds@yandex-team.ru'


# I use client-side sessions, no server storage. Currently I don't store in sessions any sensitive data.
flaskapp = Flask(__name__)
flaskapp.secret_key = FLASK_SECRET_KEY

@flaskapp.route("/ping")
def ping():
    return "OK"

@flaskapp.route("/", methods=['GET', 'POST'])
def index():
    scopes=['MyFiles.Read', 'MyFiles.Write', 'Files.ReadWrite.All']
    robot_name = request.args.get('robot-name')
    secret_id = request.args.get('secret-id')
    if secret_id == None or robot_name == None:
        return render_template('index.html',
            scopes=scopes,
            my_robot=MY_ROBOT,
            previous_secret_id=session.get('secret-id', ''),
            previous_robot_name=session.get('robot-name', ''),
            version=msal.__version__
        )
    else:
        session['secret-id'] = secret_id
        session['robot-name'] = robot_name
        
        if robot_name.startswith(ROBOT_PREFIX) == False and robot_name != TEST_ROBOT:
            return render_template('error.html', message=Markup(f"Your robot's name <b>must</b> start with prefix <code>{ ROBOT_PREFIX }*</code>"))
        try:
            client = VaultClient(rsa_auth=None, authorization='OAuth {}'.format(MY_ROOT_YAV_TOKEN), decode_files=True)
            if client.can_user_read_secret(secret_id, MY_ROBOT_UID) == True:
                return render_template('error.html', message=Markup(f"Robot <code>{ MY_ROBOT }</code> <b>must not</b> have <b>read</b> privileges on your <code>{ html.escape(secret_id) }</code> secret"))
        except VaultClientError as err:
            return render_template('error.html', message=Markup(f"Error checking vault's <code>{ html.escape(session['secret-id']) }</code> secret: <code>{ html.escape(err.kwargs['message']) }</code>"))

        clientapp = _build_msal_app()
        # session['flow'] = clientapp.initiate_auth_code_flow(scopes=scopes, redirect_uri=REDIRECT_URI, prompt='select_account', login_hint=TEST_ROBOT)
        session['flow'] = clientapp.initiate_auth_code_flow(scopes=scopes, redirect_uri=REDIRECT_URI, prompt='select_account', login_hint=robot_name)
        return redirect(session['flow']['auth_uri'])

@flaskapp.route("/getToken")
def getToken():
    refresh_token_lifetime = 90 # in days
    
    try:
        cache = msal.SerializableTokenCache()
        clientapp = _build_msal_app(cache)
        result = clientapp.acquire_token_by_auth_code_flow(session['flow'], request.args)
        if 'error' in result:
            raise Exception(json.dumps(result, indent=4))
        
        if clientapp.get_accounts()[0]['username'].startswith(ROBOT_PREFIX) == False and clientapp.get_accounts()[0]['username'] != TEST_ROBOT:
            return render_template('error.html', message=Markup(f"Stop tampering with me !!! <br> Your robot's name <b>must</b> start with prefix <code>{ ROBOT_PREFIX }*</code>"))
        
        client = VaultClient(rsa_auth=None, authorization='OAuth {}'.format(MY_ROOT_YAV_TOKEN), decode_files=True)
        try:
            client.create_secret_version(session["secret-id"], value={"msal-cache": cache.serialize()}, ttl=60 * 60 * 24 * refresh_token_lifetime, comment="AzureAD refresh token is valid only for 90 days.")
        except VaultClientError as err:
            if err.kwargs['code'] == 'access_error':
                return render_template('error.html', message=Markup(f"Error storing obtained secrets into vault <code>{ html.escape(session['secret-id']) }</code> secret: <code>{ html.escape(err.kwargs['message']) }</code>"))
            else:
                raise err
    except Exception as err:
        return render_template('error.html', message=Markup(f"<div class='text-left' style='white-space: pre-wrap'><code> { html.escape(str(err)) } </code></div>"))
    
    return render_template('success.html',
        robot_name=result['id_token_claims']['preferred_username'],
        scope=result['scope'],
        secret_id=session.get('secret-id', ''),
        version=msal.__version__
    )

def _build_msal_app(cache=None):
    return msal.PublicClientApplication(APP_ID, token_cache=cache, authority=msal.authority.AuthorityBuilder(msal.authority.AZURE_PUBLIC, TENANT))

if __name__ == "__main__":
    flaskapp.run(host="127.0.0.1", port=8080, debug=True)


# some sharepoint-related privileges:
#   https://microsoft.sharepoint-df.com/MyFiles.Read         - Read user files - Allows the app to read the current user's files
#   https://microsoft.sharepoint-df.com/MyFiles.Write        - Read and write user files - Allows the app to read, create, update, and delete the current user's files
# PS> Find-MgGraphPermission file
#   https://graph.microsoft.com/Files.Read                   - Read user files - Allows the app to read the signed-in user's files
#   https://graph.microsoft.com/Files.Read.All               - Read all files that user can access - Allows the app to read all files the signed-in user can access
#   https://graph.microsoft.com/Files.Read.Selected          - Read files that the user selects (preview) - (Preview) Allows the app to read files that the user selects. The app has access for several hours after the user selects a file
#   https://graph.microsoft.com/Files.ReadWrite              - Have full access to user files - Allows the app to read, create, update and delete the signed-in user's files
#   https://graph.microsoft.com/Files.ReadWrite.All          - Have full access to all files user can access - Allows the app to read, create, update and delete all files the signed-in user can access
#   https://graph.microsoft.com/Files.ReadWrite.AppFolder    - Have full access to the application's folder (preview) - (Preview) Allows the app to read, create, update and delete files in the application's folder
#   https://graph.microsoft.com/Files.ReadWrite.Selected     - Read and write files that the user selects (preview) - (Preview) Allows the app to read and write files that the user selects. The app has access for several hours after the user selects a file
# PS> Find-MgGraphPermission site
#   https://graph.microsoft.com/Sites.FullControl.All    - Have full control of all site collections - Allows the application to have full control of all site collections on behalf of the signed-in user
#   https://graph.microsoft.com/Sites.Manage.All         - Create, edit, and delete items and lists in all site collections - Allows the application to create or delete document libraries and lists in all site collections on behalf of the signed-in user
#   https://graph.microsoft.com/Sites.Read.All           - Read items in all site collections - Allows the application to read documents and list items in all site collections on behalf of the signed-in user
#   https://graph.microsoft.com/Sites.ReadWrite.All      - Edit or delete items in all site collections - Allows the application to edit or delete documents and list items in all site collections on behalf of the signed-in user


# python msal lib:
#   https://msal-python.readthedocs.io/en/latest/#scenarios
#   https://github.com/AzureAD/microsoft-authentication-library-for-python/wiki/Scenarios

#   initiate_auth_code_flow + acquire_token_by_auth_code_flow
#   get_authorization_request_url + acquire_token_by_authorization_code
#   acquire_token_interactive
#   acquire_token_silent OR acquire_token_silent_with_error
#   adal + acquire_token_by_refresh_token
# special modes:
#   initiate_device_flow + acquire_token_by_device_flow
#   acquire_token_by_username_password
#   OBO: acquire_token_on_behalf_of
# get token for Confidential Application itself:
#   acquire_token_for_client

