#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# $env:SSL_CERT_FILE = "$env:USERPROFILE\ca-bundle.crt"
# $env:REQUESTS_CA_BUNDLE = "$env:USERPROFILE\ca-bundle.crt"

import os
import re
from datetime import datetime, timedelta

import msal
from vault_client.instances import Production as VaultClient

from office365.runtime.auth.token_response import TokenResponse
from office365.runtime.auth.authentication_context import AuthenticationContext
from office365.sharepoint.client_context import ClientContext

TENANT = os.environ.get('TENANT', 'yandexteam.onmicrosoft.com')
APP_ID = os.environ.get('APP_ID', 'bb6b77ff-7b38-4d38-a2fe-7fcb3ebee61e')   # registered Application (client) ID used for OAuth application

SECRET_ID_MSAL_CACHE = 'sec-01g0xrw1hr1zdbt749wgmkqb5b'     # require read & write access for robot
MY_ROOT_YAV_TOKEN = 'AQAD-...'

###

def update_vault_cache(msalapp, vault, days_before_expire=7):
    """
    Update MSAL cache stored in vault if refresh token will expire soon (e.g. 7 days).
    Default Azure AD refresh token lifetime equals to 90 days (unless it is single-page application).
    """
    refresh_token_lifetime = 90 # in days
    last_modification_time = msalapp.token_cache.find(msal.TokenCache.CredentialType.REFRESH_TOKEN)[0]['last_modification_time']
    if datetime.fromtimestamp(int(last_modification_time)) + timedelta(days=refresh_token_lifetime - days_before_expire) < datetime.now():
        msalapp.acquire_token_silent_with_error(scopes=[".default"], account=msalapp.get_accounts()[0])
        if msalapp.token_cache.has_state_changed:
            vault.create_secret_version(SECRET_ID_MSAL_CACHE, value={"msal-cache": msalapp.token_cache.serialize()}, ttl=60 * 60 * 24 * refresh_token_lifetime, comment=f"AzureAD refresh token is valid only for {refresh_token_lifetime} days.")

def get_msal_app():
    """
    Setup MSAL application with 
    """

    vault = VaultClient(rsa_auth=None, authorization='OAuth {}'.format(MY_ROOT_YAV_TOKEN), decode_files=True)
    msal_cache = vault.get_version(SECRET_ID_MSAL_CACHE)['value']['msal-cache']

    cache = msal.SerializableTokenCache()
    cache.deserialize(msal_cache)
    
    msalapp = msal.PublicClientApplication(APP_ID, token_cache=cache, authority=msal.authority.AuthorityBuilder(msal.authority.AZURE_PUBLIC, TENANT))
    update_vault_cache(msalapp, vault)
    return msalapp

def acquire_token_gen(msalapp, audience):
    """
    Return 'access token' generator for sharepoint's library.
    """
    audience = re.findall(r'(https?://.*?)/', audience)[0]

    def acquire_token():
        token = msalapp.acquire_token_silent_with_error(scopes=[f'{audience}/.default'], account=msalapp.get_accounts()[0])
        return TokenResponse(accessToken=token['access_token'], tokenType=token['token_type'])
    return acquire_token

### example 1

def download_file_ex1(msalapp):

    base_url = "https://yandexteam.sharepoint.com/sites/avasite-o365grp"
    auth_context = AuthenticationContext(base_url)
    auth_context.register_provider(acquire_token_gen(msalapp, base_url))
    ctx = ClientContext(base_url, auth_context=auth_context)

    with open("test.docx", "wb") as local_file:
        file = ctx.web.get_file_by_server_relative_path("/sites/avasite-o365grp/Shared Documents/test.docx").download(local_file).execute_query()
    print('[*] file from yandexteam.sharepoint.com successfully downloaded')

### example 2

def download_file_ex2(msalapp):

    base_url = "https://yandexteam-my.sharepoint.com/personal/avasite_yandex-team_ru"
    auth_context = AuthenticationContext(base_url)
    auth_context.register_provider(acquire_token_gen(msalapp, base_url))
    ctx = ClientContext(base_url, auth_context=auth_context)

    with open("test-my.docx", "wb") as local_file:
        file = ctx.web.get_file_by_server_relative_path("/personal/avasite_yandex-team_ru/Documents/test.docx").download(local_file).execute_query()
    print('[*] file from yandexteam-my.sharepoint.com successfully downloaded')

###

if __name__ == '__main__':
    msalapp = get_msal_app()
    print('[*] msal application initialized')

    download_file_ex1(msalapp)
    download_file_ex2(msalapp)
