# -*- coding: utf-8 -*-
import os
import time
import logging

import ydb
from concurrent.futures import TimeoutError
from http import requests_retry_session


logger = logging.getLogger('staff-users')


UploadQuery = '''
PRAGMA TablePathPrefix("{}");

DECLARE $staffUsers AS "List<Struct<
    uid: Uint64,
    login: String,
    updated_at: Uint64>>";

REPLACE INTO staff_users
SELECT
    uid, login, updated_at
FROM AS_TABLE($staffUsers);
'''


class StaffUser(object):
    __slots__ = ('uid', 'login', 'updated_at')

    def __init__(self, uid, login):
        self.uid = int(uid)
        self.login = login.encode('latin-1')
        self.updated_at = int(time.time())


class UserIterator(object):
    def __init__(self, oauth_token):
        self.session = requests_retry_session()
        self.session.headers.update({'Authorization': 'OAuth ' + oauth_token})
        self.url = 'https://staff-api.yandex-team.ru/v3/persons?_fields=uid,login'

    def next(self):
        if not self.url:
            raise StopIteration()

        resp = self.session.get(self.url).json()
        self.url = resp.get('links', {}).get('next', None)
        users = []
        for res in resp.get('result', []):
            login = res.get('login', None)
            uid = res.get('uid', None)
            if not login or not uid:
                continue
            users.append(StaffUser(uid, login))
        return users

    # Python 3 compatibility
    def __next__(self):
        return self.next()

    def __iter__(self):
        return self


def create_tables(session, path):
    session.create_table(
        os.path.join(path, 'staff_users'),
        ydb.TableDescription()
        .with_column(ydb.Column('uid', ydb.OptionalType(ydb.DataType.Uint64)))
        .with_column(ydb.Column('login', ydb.OptionalType(ydb.DataType.String)))
        .with_column(ydb.Column('updated_at', ydb.OptionalType(ydb.DataType.Uint64)))
        .with_primary_key('uid')
    )


def upload(session, path, users):
    query = UploadQuery.format(path)
    prepared_query = session.prepare(query)
    session.transaction(ydb.SerializableReadWrite()).execute(
        prepared_query, {
            '$staffUsers': users,
        },
        commit_tx=True
    )


def run(endpoint, database, path, auth_token):
    connection_params = ydb.ConnectionParams(endpoint, database=database, auth_token=auth_token)
    try:
        driver = ydb.Driver(connection_params)
        driver.wait(timeout=5)
    except TimeoutError:
        raise RuntimeError('Connect failed to YDB')
    session = driver.table_client.session().create()

    create_tables(session, path)
    logger.info('start update')
    batch = []
    updated = 0
    for users in UserIterator(auth_token):
        updated += len(users)
        batch.extend(users)
        if len(batch) > 100:
            upload(session, database, batch)
            del batch[:]

    if batch:
        upload(session, database, batch)

    logger.info('updated %d users' % updated)
    logger.info('done')
