#include "oauth.h"

#include <security/ant-secret/internal/string_utils/common.h>

#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/digest/old_crc/crc.h>

namespace NBlackBox {
    const size_t kV2TokenLen = 39; // base64url(shard, uid, clientid, random)
    const size_t kV3TokenLen = 58; // y(env)_base64url(shard, uid, clientid, token_id, random, crc32)
    const ui8 kMaxShard = 19;
    const ui32 kMaxClientId = 1e7;

    bool IsEmbeddedV2Token(const TStringBuf token) {
        if (token.size() != kV2TokenLen) {
            return false;
        }

        if (NStringUtils::IsMasked(token, 20)) {
            return false;
        }

        TString decoded = Base64DecodeUneven(token);
        if (decoded.size() < 28) {
            return false;
        }

        ui8 shard = decoded[0];
        if (shard == 0 || shard > kMaxShard) {
            return false;
        }

        ui64 uid = NStringUtils::FromBytesMsb<ui64>(decoded, 1);
        if (uid == 0) {
            return false;
        }

        ui32 clientId = NStringUtils::FromBytesMsb<ui32>(decoded, 9);
        if (clientId == 0 || clientId > kMaxClientId) {
            return false;
        }

        return true;
    }

    bool IsEmbeddedV3Token(const TStringBuf token) {
        if (token.size() != kV3TokenLen) {
            return false;
        }

        if (NStringUtils::IsMasked(token, 20)) {
            return false;
        }

        if (token[0] != 'y' || token[2] != '_') {
            return false;
        }

        TString decoded = Base64DecodeUneven(TStringBuf(token).Skip(3));
        if (decoded.empty()) {
            return false;
        }

        ui8 shard = decoded[0];
        if (shard == 0 || shard > kMaxShard) {
            return false;
        }

        ui64 uid = NStringUtils::FromBytesMsb<ui64>(decoded, 1);
        if (uid == 0) {
            return false;
        }


        ui32 clientId = NStringUtils::FromBytesMsb<ui32>(decoded, 9);
        if (clientId == 0 || clientId > kMaxClientId) {
            return false;
        }

        ui64 tokenId = NStringUtils::FromBytesMsb<ui64>(decoded, 13);
        if (tokenId == 0) {
            return false;
        }


        ui32 expectedChecksum = NStringUtils::FromBytesMsb<ui32>(decoded, decoded.size() - 4);
        // It's important to use crc32, not crc32c
        ui32 actualChecksum = crc32(decoded.data(), decoded.size() - 4);
        return actualChecksum == expectedChecksum;
    }
}
