#pragma once

#include <security/libs/cpp/hyperscan/hyperscan.h>
#include <util/generic/hash.h>
#include <util/system/mutex.h>

namespace NSSInternal {
    struct TRegexStorageItem {
        NHyperscan::TDatabase db;
        NHyperscan::TScratch scratch;

        TRegexStorageItem(const TVector<const char*>& patterns,
                          const TVector<unsigned int>& flags,
                          const TVector<unsigned int>& ids,
                          NHyperscan::EPlatform platform) {
            db = NHyperscan::CompileMultiFor(patterns, flags, ids, platform);
            scratch = NHyperscan::MakeScratch(db);
        }

        explicit TRegexStorageItem(const TStringBuf serialized) {
            db = NHyperscan::Deserialize(serialized);
            scratch = NHyperscan::MakeScratch(db);
        }
    };

    struct TRegexHolder {
        NHyperscan::TDatabasePtr db = nullptr;
        NHyperscan::TScratch scratch;

        TRegexHolder() = default;

        explicit TRegexHolder(const TRegexStorageItem& item)
            : db(item.db.Get())
            , scratch(NHyperscan::CloneScratch(item.scratch))
        {
        }
    };

    using TRegexStorageCache = THashMap<TString, TRegexStorageItem>;

    class TRegexStorage: public TNonCopyable {
    public:
        static TRegexStorage& Instance();

        TRegexHolder GetOrCompile(const TString& name,
                                  const TVector<const char*>& patterns,
                                  const TVector<unsigned int>& flags,
                                  const TVector<unsigned int>& ids);

        void SetTargetPlatform(NHyperscan::EPlatform target) {
            platform = target;
        }

        void Flush() {
            cache.clear();
        }

        const TString DumpDb();

    protected:
        TRegexStorage();

        bool initDb();

    protected:
        TRegexStorageCache cache;
        TMutex lock;
        NHyperscan::EPlatform platform;
    };

}
