#include <security/ant-secret/secret-search/public/cpp/collector/collector.h>
#include <security/ant-secret/snooper/cpp/collector/collector.h>

#include <security/libs/cpp/log/log.h>
#include <security/ant-secret/internal/regex_storage/storage.h>

#include <library/cpp/getopt/last_getopt.h>

namespace {
    NHyperscan::EPlatform parsePlatform(const TString& platform) {
        if (platform == "avx2") {
            return NHyperscan::EPlatform::AVX2;
        } else if (platform == "avx512") {
            return NHyperscan::EPlatform::AVX512;
        } else if (platform == "any") {
            return NHyperscan::EPlatform::ANY;
        }

        ythrow yexception() << "failed to parse platform, unsupported: " << platform;
    }

    TString platformToString(NHyperscan::EPlatform platform) {
        switch (platform) {
            case NHyperscan::EPlatform::AVX512:
                return "avx512";
            case NHyperscan::EPlatform::AVX2:
                return "avx2";
            case NHyperscan::EPlatform::ANY:
                return "any";
            default:
                ythrow yexception() << "failed to serialize platform, unsupported: " << static_cast<int>(platform);
        }
    }

}

int main(int argc, char** argv) {
    using namespace NLastGetopt;
    TOpts opts = TOpts::Default();

    opts.SetTitle("Hyperscan regex dumper");

    TString platform;
    opts.AddLongOption("platform", "target platform (avx2, avx512, any)")
        .DefaultValue("any")
        .StoreResult(&platform);

    TString source;
    opts.AddLongOption("source", "regex source (secret-search, snooper)")
        .DefaultValue("secret-search")
        .StoreResult(&source);

    TOptsParseResult args(&opts, argc, argv);

    NSecurityHelpers::TLogger::Instance(NSecurityHelpers::TLOG_ERR);

    auto currentPlatform = NHyperscan::CurrentPlatform();
    auto targetPlatform = parsePlatform(platform);
    if (currentPlatform < targetPlatform) {
        NSecurityHelpers::LogErr("trying to compile unsupported platform",
                                 "target", platformToString(targetPlatform),
                                 "current", platformToString(currentPlatform)
        );
        return 0;
    }

    auto& regex = NSSInternal::TRegexStorage::Instance();
    regex.Flush();
    regex.SetTargetPlatform(targetPlatform);

    if (source == "secret-search") {
        NSecretSearch::NColletor::CollectRegexes();
    } else if (source == "snooper") {
        NSnooper::NColletor::CollectRegexes();
    } else {
        NSecurityHelpers::LogErr("unknown source", "source", source);
        return 1;
    }

    Cout << regex.DumpDb();
    return 0;
}
