#pragma once

#include "result.h"

#include <library/cpp/cache/thread_safe_cache.h>
#include <util/generic/strbuf.h>
#include <util/generic/maybe.h>

namespace NValidation {
    class TValidatorCache; // forward declaration

    class IValidator {
    public:
        virtual ~IValidator() = default;

        virtual TMaybe<TResult>
        Call(const TStringBuf type, const TStringBuf token) = 0;

        virtual std::pair<bool, bool>
        CallSsl(const TString& serial, const TVector<TString>& chain, bool isClient, bool isServer) = 0;
    };

    class TValidator: public IValidator {
    public:
        TValidator();

        explicit TValidator(const TString& baseUrl);

        TValidator(const TString& baseUrl, TString authToken);

        TValidator(const TString& url, TString  hostname, TString  authToken);

        TMaybe<TResult>
        Call(const TStringBuf type, const TStringBuf token) override;

        TResult*
        CallBypassCache(const TStringBuf type, const TStringBuf token);

        std::pair<bool, bool>
        CallSsl(const TString& serial, const TVector<TString>& chain, bool isClient, bool isServer) override;

        void SkipKnown(bool skip);

    protected:
        bool skipKnown = false;
        const TString baseUrl;
        const TString hostname;
        const TString sslUrl;
        const TString authToken;
        THolder<TValidatorCache> cache;
    };

    using TValidatorCacheImpl = TThreadSafeCache<TString, TResult, const TStringBuf, const TStringBuf>;

    class TValidatorCache: public TValidatorCacheImpl::ICallbacks {
    public:
        explicit TValidatorCache(TValidator& validator, size_t cacheSize = 1000);

        TMaybe<TResult>
        Get(const TStringBuf type, const TStringBuf token);

    private:
        //  TThreadSafeCache interface
        TResult* CreateObject(const TStringBuf type, const TStringBuf token) const final;

        TString GetKey(const TStringBuf type, const TStringBuf token) const final;

    private:
        TValidator& validator;
        TValidatorCacheImpl cache;
    };

}
