#include "server.h"

#include <util/string/builder.h>
#include <util/generic/strbuf.h>
#include <util/string/join.h>

namespace NMaskerServer {
    namespace {
        const TStringBuf kPingPath = "/ping";
        const TStringBuf kMaskPath = "/mask";

        const int kMaxConnections = 100;
    }

    bool TMaskReplier::DoReply(const TReplyParams& params) {
        THttpResponse resp;
        try {
            TParsedHttpFull req(params.Input.FirstLine());

            if (req.Path == kPingPath) {
                const TString rsp = "OK";
                resp.SetHttpCode(HTTP_OK);
                resp.SetContent(rsp);
                params.Output << resp;
                return true;
            }

            if (req.Method != "POST" || !params.Input.HasContent()) {
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.SetContent("Only POST w/ content allowed");
                params.Output << resp;
                return true;
            }

            auto content = params.Input.ReadAll();
            if (req.Path == kMaskPath) {
                auto searcher = snooper.Searcher(NSnooper::ESecretType::All);
                auto secrets = searcher->Mask(content, validateSecrets);
                if (secrets.empty()) {
                    resp.SetHttpCode(HTTP_NO_CONTENT);
                    params.Output << resp;
                    return true;
                }

                resp.SetContent(content);
                params.Output << resp;
                return true;
            } else {
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.SetContent("Path was not found, sorry :(");
                params.Output << resp;
                return true;
            }

        } catch (yexception& e) {
            Cerr << (TStringBuilder() << "REQUEST_ERROR: " << e.what() << Endl);
            resp.SetHttpCode(HTTP_BAD_REQUEST);
            params.Output << resp;
            return true;
        }
    }

    void StartServer(const TServerOptions& opts) {
        THttpServer::TOptions httpServerOptions;
        httpServerOptions.MaxConnections = kMaxConnections;
        httpServerOptions.KeepAliveEnabled = true;
        {
            size_t colon = opts.ServerAddr.rfind(':');
            httpServerOptions.Host = opts.ServerAddr.substr(0, colon);
            if (!TryFromString(opts.ServerAddr.substr(colon + 1), httpServerOptions.Port)) {
                httpServerOptions.Port = 10203;
            }
        }

        THttpCallback httpCallback(opts.ValidateSecrets);
        THttpServer httpServer(&httpCallback, httpServerOptions);
        httpServer.Start();
        Cerr << "Server started" << Endl;
        httpServer.Wait();
    }

}
