#include "options.h"

#include <security/ant-secret/secret-search/public/cpp/output/all.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/string/strip.h>
#include <util/string/join.h>
#include <util/system/cpu_id.h>

#include <thread>

namespace NSecretSearch::NBin {
    namespace {
        TString detectHsMode() {
            if (NX86::HaveAVX512F() && NX86::HaveAVX512BW()) {
                return "avx512";
            } else if (NX86::HaveAVX() && NX86::HaveAVX2()) {
                return "avx2";
            } else if (NX86::HaveSSE42() && NX86::HavePOPCNT()) {
                return "corei7";
            } else {
                return "core2";
            }
        }

    }

    TOptions TOptions::Parse(int argc, char** argv) {
        auto options = TOptions();

        using namespace NLastGetopt;
        TOpts opts = TOpts::Default();

        opts.SetTitle("Secrets searcher");

        opts.AddLongOption("validate", "validate tokens/ssh-keys/etc")
            .NoArgument();

        opts.AddLongOption("valid-only", "report only _validated_ secrets")
            .NoArgument();

        opts.AddLongOption("check-struct", "check struct (only json for now")
            .NoArgument();

        opts.AddLongOption("format", "output format (text, console, hector, st, json, jsonlines or proto)")
            .DefaultValue(options.OutFormat)
            .StoreResult(&options.OutFormat);

        opts.AddLongOption("num-threads", "Number of searcher threads, autodetected by default")
            .DefaultValue("auto");

        opts.AddLongOption("verbose", "verbose output")
            .NoArgument();

        opts.AddLongOption("with-source-line", "show source line in output")
            .NoArgument();

        opts.AddLongOption("status-code", "exit status code when something founded")
            .DefaultValue(options.StatusCode)
            .StoreResult(&options.StatusCode);

        opts.AddLongOption("max-size", "maximum file size to search (set 0 to disable limit at all)")
            .DefaultValue(options.MaxFileSize)
            .StoreResult(&options.MaxFileSize);

        opts.AddLongOption("excludes", "comma-separated list of path names to exclude from scan")
            .DefaultValue(JoinSeq(",", options.Excludes));

        // For backward compatibility
        opts.AddLongOption("all")
            .Hidden()
            .NoArgument();

        opts.AddLongOption("version", "show the current version")
            .NoArgument();

        // TODO(buglloc): implement it!
        opts.AddLongOption("not-ignore", "Disable '.secretsignore'")
            .Hidden()
            .NoArgument();

        opts.SetFreeArgsMax(1);
        opts.SetFreeArgTitle(0, "path", "path to search (default: .)");

        TOptsParseResult args(&opts, argc, argv);
        {
            const auto& freeArgs = args.GetFreeArgs();
            if (freeArgs) {
                options.InputPath = freeArgs[0];
            }
        }

        options.WithSourceLine = args.Has("with-source-line");
        options.ShowVersion = args.Has("version");
        options.Verbose = args.Has("verbose");
        options.CheckStructMode = args.Has("check-struct");
        options.ValidOnly = args.Has("valid-only");
        options.Validate = options.ValidOnly || args.Has("validate");

        if (args.Has("excludes")) {
            options.Excludes.clear();
            for (const auto& excl : SplitString(args.Get("excludes"), ",")) {
                options.Excludes.push_back(StripString(excl));
            }
        }

        const auto& argThreads = args.Get("num-threads");
        if (strcasecmp(argThreads, "auto") == 0) {
            auto numThreads = static_cast<size_t>(std::thread::hardware_concurrency());
            if (options.Validate) {
                // if we need to validate our search - double threads count to deal with I/O bound tasks
                numThreads *= 2;
            }
            options.NumThreads = numThreads;
        } else {
            options.NumThreads = FromString<size_t>(argThreads);
        }

        return options;
    }

    void TOptions::Print(IOutputStream& out) {
        out
            << "InputPath = " << InputPath << Endl
            << "Excludes = " << JoinSeq(",", Excludes) << Endl
            << "CheckStructMode = " << (CheckStructMode ? "yes" : "no") << Endl
            << "NumThreads = " << NumThreads << Endl
            << "Validate = " << (Validate ? "yes" : "no") << Endl
            << "ValidOnly = " << (ValidOnly ? "yes" : "no") << Endl
            << "MaxFileSize = " << MaxFileSize << Endl
            << "StatusCode = " << StatusCode << Endl
            << "Verbose = " << (Verbose ? "yes" : "no") << Endl
            << "Hyperscan mode = " << detectHsMode() << Endl;
    }

    TSearchOptions TOptions::SearcherOptions() const {
        return TSearchOptions{
            .MaxFileSize = MaxFileSize,
            .Excludes = Excludes,
            .Validate = Validate,
            .ValidOnly = ValidOnly,
            .WithSourceLine = WithSourceLine,
        };
    }

    NOutput::IWriter* TOptions::OutputWriter() const {
        if (OutFormat == "auto") {
            if (isatty(STDOUT_FILENO)) {
                return new NOutput::TConsole(Cout);
            }
            return new NOutput::TText(Cout);
        } else if (OutFormat == "text") {
            return new NOutput::TText(Cout);
        } else if (OutFormat == "console") {
            return new NOutput::TConsole(Cout);
        } else if (OutFormat == "jsonlines") {
            return new NOutput::TJsonLines(Cout);
        } else if (OutFormat == "json") {
            return new NOutput::TJson(Cout);
        } else if (OutFormat == "hector") {
            return new NOutput::THector(Cout);
        } else if (OutFormat == "startrek" || OutFormat == "st") {
            return new NOutput::TStartrek(Cout);
        } else if (OutFormat == "proto") {
            return new NOutput::TProto(Cout);
        }

        ythrow NLastGetopt::TUsageException() << "unknown format: " << OutFormat;
    }

}
