#include "options.h"

#include <security/ant-secret/secret-search/git-hook/lib/searcher.h>
#include <security/ant-secret/secret-search/git-hook/lib/output.h>

#include <security/libs/cpp/log/log.h>
#include <contrib/libs/openssl/include/openssl/rsa.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/svnversion/svnversion.h>
#include <util/generic/strbuf.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/system/shellcommand.h>
#include <util/system/execpath.h>
#include <util/thread/pool.h>

using namespace NSecretSearchGitHook;

namespace {
    TString kVersion = "0.6." + ToString(GetProgramSvnRevision());
    const int kSecretsFoundExitCode = 1;
    const TString kRepoPath{"."};
    constexpr TStringBuf kNeededRef = "refs/heads/";
    const THashSet<TString> kAcceptableSources{
        {"blob edit"},
    };

    enum EHookKind {
        NONE,
        GITHUB,
        GITLAB,
    };

    struct THookEnv {
        EHookKind Kind;
        TString User;
        TString Source;
        bool PublicRepo;
    };

    THookEnv ParseEnv()  {
        auto githubUser = GetEnv("GITHUB_USER_LOGIN");
        if (!githubUser.empty()) {
            return THookEnv{
                .Kind = EHookKind::GITHUB,
                /*
                * $GITHUB_USER_LOGIN - The user id who created the ref.
                * https://help.github.com/enterprise/2.15/admin/guides/developer-workflow/creating-a-pre-receive-hook-script/#environment-variables
                */
                .User = githubUser,
                 /*
                 * $GITHUB_VIA - Method used to create the ref. In case of git push - empty
                 * Possible values:
                 *   - auto-merge deployment api
                 *   - blob edit
                 *   - branch merge api
                 *   - branches page delete button
                 *   - git refs create api
                 *   - git refs delete api
                 *   - git refs update api
                 *   - merge api
                 *   - pull request branch delete button
                 *   - pull request branch undo button
                 *   - pull request merge api
                 *   - pull request merge button
                 *   - pull request revert button
                 *   - releases delete button
                 *   - stafftools branch restore
                 *   - slumlord (#{sha})
                 * https://help.github.com/enterprise/2.15/admin/guides/developer-workflow/creating-a-pre-receive-hook-script/#environment-variables
                 */
                .Source =  GetEnv("GITHUB_VIA"),
                /*
                * $GITHUB_REPO_PUBLIC - A boolean value that when true represents a public repository, and when false represents a private repository.
                * https://help.github.com/enterprise/2.15/admin/guides/developer-workflow/creating-a-pre-receive-hook-script/#environment-variables
                */
                .PublicRepo = GetEnv("GITHUB_REPO_PUBLIC") != "false",
            };
        }

        auto gitlabUser = GetEnv("GL_USERNAME");
        if (!gitlabUser.empty()) {
            // https://docs.gitlab.com/ee/administration/server_hooks.html#environment-variables-available-to-server-hooks
            return THookEnv{
                    .Kind = EHookKind::GITLAB,
                    /*
                    * $GL_USERNAME - GitLab username of the user that initiated the push.
                    */
                    .User = githubUser,
            };
        }

        return THookEnv{.Kind = EHookKind::NONE};
    }

    void run(TSearcher& searcher, TOutput& writer) {
        try {
            /*
         * Stdin data: ${oldrev} ${newrev} ${refname}
         * More info:
         *   https://git.io/fNLf0
         *   https://git-scm.com/book/ru/v1/%D0%9D%D0%B0%D1%81%D1%82%D1%80%D0%BE%D0%B9%D0%BA%D0%B0-Git-%D0%9F%D1%80%D0%B8%D0%BC%D0%B5%D1%80-%D0%BD%D0%B0%D0%B2%D1%8F%D0%B7%D1%8B%D0%B2%D0%B0%D0%BD%D0%B8%D1%8F-%D0%BF%D0%BE%D0%BB%D0%B8%D1%82%D0%B8%D0%BA%D0%B8-%D1%81-%D0%BF%D0%BE%D0%BC%D0%BE%D1%89%D1%8C%D1%8E-Git
         */
            TString line;
            while (Cin.ReadLine(line)) {
                TVector<TStringBuf> input;
                Split(line, " ", input);
                /*
                * Sometimes github do weird things and send multimple refs at once so we use only first 3 items, e.g.:
                * 20f0257d45bd796a45e6c0cc4ed8a0548a4f7199 793a62c6e1036dec44dfa7b6691ef8b7a394202c refs/heads/rc-ad-preload 0000000000000000000000000000000000000000 7ffd9176896cce45a2bf753f954fa474d4379d29 refs/tags/17.5.353
                *
                * TODO(buglloc): R&D and fix
                */
                if (input.size() < 3) {
                    NSecurityHelpers::LogErr("failed to parse hook input", "line", line);
                    continue;
                }

                auto revA = TString{input[0]};
                auto revB = TString{input[1]};
                auto refname = input[2];
                NSecurityHelpers::LogDebug("handle revision", "revA", revA, "revB", revB, "refname", refname);

                if (!refname.StartsWith(kNeededRef)) {
                    // Skip trash like tags and so on
                    NSecurityHelpers::LogInfo("skip reference", "refname", refname);
                    continue;
                }

                searcher.CheckDiff(writer, revA, revB);
            }
        } catch (const yexception& e) {
            NSecurityHelpers::LogErr("Failed to execute", "err", e.AsStrBuf());
        }
    }

    int doRealRun(const TProgramOptions& options) {
        // TODO(buglloc): hack :(
        // Remove after update to openssl-1.1.0h
        RSA_get_default_method();

        TOutput writer(Cout);
        writer.Start();
        TSearchOptions searchOpts{
            .RepoPath = kRepoPath,
            .Validate = true,
            .ValidOnly = true,
            .EachRev = options.EachRev,
            .SkipKnownSecrets = options.SkipKnownSecrets,
            .SkipKnownRevs = options.SkipKnownRevs,
        };
        TSearcher searcher(searchOpts, options.NumThreads);

        if (options.Timeout) {
            TThreadPool queue{TThreadPool::TParams().SetBlocking(false).SetCatching(false)};
            queue.Start(1);

            auto future = NThreading::Async([&searcher, &writer] {
                run(searcher, writer);
            },
                                            queue);

            if (!future.Wait(options.Timeout)) {
                // TODO(buglloc): postpone!
                searcher.Stop();
                NSecurityHelpers::LogWarn("SecretSearch working too long. Results may be incomplete.");
            }
        } else {
            run(searcher, writer);
        }

        writer.Finish();
        return !writer.Empty() ? kSecretsFoundExitCode : 0;
    }

    int doBootstrapRun(const TProgramOptions& options) {
        auto env = ParseEnv();
        if (env.Kind == EHookKind::NONE) {
            NSecurityHelpers::LogWarn("Unsupported environment. Do you start hook on Github or Gitlab?");
        }

        if (!options.CheckPrivate && env.PublicRepo) {
            NSecurityHelpers::LogDebug("skip non-public repo");
            return 0;
        }

        if (options.SafeLogins.contains(env.User)) {
            // https://st.yandex-team.ru/FEI-20958
            NSecurityHelpers::LogInfo("ignore ref checking from whitelisted user", "user", env.User);
            return 0;
        }

        if (options.RestrictSource && !env.Source.empty()) {
            if (kAcceptableSources.contains(env.Source)) {
                NSecurityHelpers::LogDebug("skip not acceptable source", "source", env.Source);
                return 0;
            }
        }

        auto args = options.Args(
            true,
            options.Colorized || NSecurityHelpers::TLogger::Instance().Colorized());

        TShellCommandOptions opts;
        opts.SetCloseInput(true);
        opts.SetUseShell(false);
        opts.SetInputStream(&Cin);
        opts.SetOutputStream(&Cout);
        opts.SetErrorStream(&Cerr);

        TShellCommand cmd(GetExecPath(), args, opts);

        bool logUnknownExit = true;
        if (options.Timeout) {
            TThreadPool queue{TThreadPool::TParams().SetBlocking(false).SetCatching(false)};
            queue.Start(1);

            auto future = NThreading::Async([&cmd] {
                cmd.Run().Wait();
            },
                                            queue);

            if (!future.Wait(options.Timeout)) {
                // TODO(buglloc): postpone!
                logUnknownExit = false;
                cmd.Terminate();
            }
        } else {
            cmd.Run();
        }

        auto exitCode = cmd.GetExitCode().GetOrElse(0);
        switch (exitCode) {
            case 0:
                return 0;
            case 1:
                return 1;
            default:
                if (logUnknownExit) {
                    NSecurityHelpers::LogErr("Unexpected exit code", "exit_code", ToString(exitCode));
                }
                return 0;
        }
    }

}

int main(int argc, char** argv) {
    TInstant start = TInstant::Now();

    auto options = TProgramOptions::Parse(argc, argv);
    if (options.ShowVersion) {
        Cout << kVersion << Endl;
        return 0;
    }

    if (options.Verbose) {
        Cerr << "Started " << (options.RealRun ? "real hook" : "bootstrap") << " with options:" << Endl;
        options.Print(Cerr);

        Cerr << "Build info: " << Endl;
        Cerr << GetProgramSvnVersion() << Endl;

        NSecurityHelpers::TLogger::Instance(NSecurityHelpers::TLOG_DEBUG);
    }

    if (options.Colorized) {
        NSecurityHelpers::TLogger::Instance().ForceColorized();
    }

    int ret = 0;
    try {
        if (options.RealRun) {
            ret = doRealRun(options);
        } else {
            ret = doBootstrapRun(options);
        }
    } catch (const yexception& e) {
        NSecurityHelpers::LogErr("Failed to execute", "err", e.AsStrBuf());
    }

    NSecurityHelpers::LogDebug("secret search done",
                               "type", (options.RealRun ? "real hook" : "bootstrap"),
                               "elapsed_ms", ToString((TInstant::Now() - start).MilliSeconds()));
    return ret;
}
