#include "options.h"

#include <library/cpp/getopt/last_getopt.h>
#include <util/string/strip.h>
#include <util/string/vector.h>
#include <util/string/join.h>
#include <util/system/cpu_id.h>

#include <thread>

namespace NSecretSearchGitHook {
    namespace {
        const int kRealrunThreshold = 150;

    }

    const TProgramOptions TProgramOptions::Parse(int argc, char** argv) {
        auto options = TProgramOptions();

        using namespace NLastGetopt;
        TOpts opts = TOpts::Default();

        opts.SetTitle("Git pre-receive hook for secrets search");

        opts.AddLongOption("num-threads", "Number of searcher threads (use \"auto\" to autodect)")
            .DefaultValue(options.NumThreads);

        opts.AddLongOption("verbose", "Verbose output")
            .DefaultValue(options.Verbose)
            .StoreResult(&options.Verbose);

        opts.AddLongOption("each-rev", "Checks each revision separately")
            .DefaultValue(options.EachRev)
            .StoreResult(&options.EachRev);

        opts.AddLongOption("check-private", "Checks private repos too")
            .DefaultValue(options.CheckPrivate)
            .StoreResult(&options.CheckPrivate);

        opts.AddLongOption("restrict-source",
                           "Restrict method used to create the ref (by default checks only normal pushes if possible)")
            .DefaultValue(options.RestrictSource)
            .StoreResult(&options.RestrictSource);

        opts.AddLongOption("timeout", "Timeout to check (ms)")
            .DefaultValue(900);

        opts.AddLongOption("colorized", "Force colorized output")
            .Hidden()
            .DefaultValue(options.Colorized)
            .StoreResult(&options.Colorized);

        opts.AddLongOption("skip-known-secrets", "Skip known secrets")
            .DefaultValue(options.SkipKnownSecrets)
            .StoreResult(&options.SkipKnownSecrets);

        opts.AddLongOption("skip-known-revs", "Skip known revisions (referenced to any HEAD)")
            .DefaultValue(options.SkipKnownRevs)
            .StoreResult(&options.SkipKnownRevs);

        opts.AddLongOption("safe-logins", "comma-separated list of \"safe\" (ignored) logins")
                .DefaultValue(JoinSeq(",", options.SafeLogins));

        opts.AddLongOption("version", "show the current version")
            .NoArgument();

        opts.AddLongOption("real-run", "")
            .Hidden()
            .NoArgument();

        TOptsParseResult args(&opts, argc, argv);

        options.RealRun = args.Has("real-run");
        options.ShowVersion = args.Has("version");

        const auto& argTimeout = args.Get("timeout");
        options.Timeout = TDuration::MilliSeconds(FromString<size_t>(argTimeout));

        if (args.Has("safe-logins")) {
            options.SafeLogins.clear();
            for (const auto& excl : SplitString(args.Get("safe-logins"), ",")) {
                options.SafeLogins.insert(StripString(excl));
            }
        }

        const auto& argThreads = args.Get("num-threads");
        if (strcasecmp(argThreads, "auto") == 0) {
            options.NumThreads = static_cast<size_t>(std::thread::hardware_concurrency());
        } else {
            options.NumThreads = FromString<size_t>(argThreads);
        }
        return options;
    }

    void TProgramOptions::Print(IOutputStream& out) {
        out
            << "NumThreads = " << NumThreads << Endl
            << "Verbose = " << (Verbose ? "yes" : "no") << Endl
            << "EachRev = " << (EachRev ? "yes" : "no") << Endl
            << "CheckPrivate = " << (CheckPrivate ? "yes" : "no") << Endl
            << "RestrictSource = " << (RestrictSource ? "yes" : "no") << Endl
            << "Colorized = " << (Colorized ? "yes" : "no") << Endl
            << "SkipKnownSecrets = " << (SkipKnownSecrets ? "yes" : "no") << Endl
            << "SkipKnownRevs = " << (SkipKnownRevs ? "yes" : "no") << Endl
            << "Timeout = " << Timeout << Endl
            << "SafeLogins = " << JoinSeq(",", SafeLogins) << Endl
            << Endl;
    }

    const TList<TString> TProgramOptions::Args(bool addRealRun, bool addColorized) const {
        auto result = TList<TString>{
            "--num-threads",
            ToString(NumThreads),
            "--each-rev",
            (EachRev ? "yes" : "no"),
            "--check-private",
            (CheckPrivate ? "yes" : "no"),
            "--colorized",
            (addColorized || Colorized ? "yes" : "no"),
            "--restrict-source",
            (RestrictSource ? "yes" : "no"),
            "--skip-known-secrets",
            (SkipKnownSecrets ? "yes" : "no"),
            "--skip-known-revs",
            (SkipKnownRevs ? "yes" : "no"),
            "--verbose",
            (Verbose ? "yes" : "no"),
        };

        if (Timeout) {
            result.push_back("--timeout=" + ToString(Timeout.MilliSeconds() - kRealrunThreshold));
        }

        if (addRealRun) {
            result.push_back("--real-run");
        }

        return result;
    }

}
