#include "git.h"
#include "git_common.h"

#include <contrib/libs/libgit2/include/git2.h>
#include <security/libs/cpp/log/log.h>
#include <util/generic/singleton.h>
#include <util/string/cast.h>
#include <sys/stat.h>

namespace NSecretSearchGitHook {
    namespace NGit {
        namespace {
            const size_t kMinLineLen = 5;
            const size_t kMaxLineLen = 2048;
            const TString kZeroCommitId{};
            const TString kZeroCommit{"0000000000000000000000000000000000000000"};
            const unsigned int kOpenFlags = GIT_REPOSITORY_OPEN_NO_SEARCH | GIT_REPOSITORY_OPEN_FROM_ENV;
            const size_t kMaxRevs = 100;
            const git_diff_find_options kDiffFindFlags{
                .version = GIT_DIFF_FIND_OPTIONS_VERSION,
                .flags = GIT_DIFF_FIND_RENAMES | GIT_DIFF_FIND_COPIES | GIT_DIFF_FIND_EXACT_MATCH_ONLY};
            const git_diff_options kDiffFlags{
                .version = GIT_DIFF_OPTIONS_VERSION,
                .flags = GIT_DIFF_NORMAL | GIT_DIFF_IGNORE_FILEMODE | GIT_DIFF_IGNORE_SUBMODULES | GIT_DIFF_IGNORE_CASE | GIT_DIFF_DISABLE_PATHSPEC_MATCH | GIT_DIFF_IGNORE_WHITESPACE,
                .ignore_submodules = GIT_SUBMODULE_IGNORE_ALL,
                .pathspec = {nullptr, 0},
                .notify_cb = nullptr,
                .progress_cb = nullptr,
                .payload = nullptr,
                .context_lines = 0,
                .interhunk_lines = 0,
            };

            inline bool shouldSkipDelta(const git_diff_delta* delta) {
                return (
                    // skip not regular files
                    !S_ISREG(delta->new_file.mode)
                    // skip binary files
                    || ((delta->flags & GIT_DIFF_FLAG_BINARY) != 0)
                    // skip unwanted deltas
                    ||
                    (delta->status != GIT_DELTA_ADDED && delta->status != GIT_DELTA_MODIFIED /*&& delta->status != GIT_DELTA_DELETED*/));
            }

            inline bool invokeCb(git_patch* patch, const TString& commitId, const TDiffWalkCb& cb) {
                int error;
                size_t numLines;
                size_t numHunks;
                size_t lineNo;
                const git_diff_hunk* hunk;
                const git_diff_line* line;

                const git_diff_delta* delta = git_patch_get_delta(patch);
                TDiff diff{
                    .CommitId = commitId,
                    .SourceFile = (delta->status != GIT_DELTA_ADDED) ? delta->old_file.path : nullptr,
                    .TargetFile = (delta->status != GIT_DELTA_DELETED) ? delta->new_file.path : nullptr,
                };

                numHunks = git_patch_num_hunks(patch);
                for (size_t hunkIdx = 0; hunkIdx < numHunks; ++hunkIdx) {
                    error = git_patch_get_hunk(&hunk, &numLines, patch, hunkIdx);
                    if (error) {
                        const git_error* e = giterr_last();
                        NSecurityHelpers::LogErr("failed to get hunk in patch",
                                                 "hunk_index", ToString(hunkIdx),
                                                 "err", TString(e->message));
                        continue;
                    };

                    if (numHunks == 1 && hunk->new_start == 1) {
                        diff.Fulfilled = true;
                    }

                    for (size_t lineIdx = 0; lineIdx < numLines; ++lineIdx) {
                        error = git_patch_get_line_in_hunk(&line, patch, hunkIdx, lineIdx);
                        if (error) {
                            const git_error* e = giterr_last();
                            NSecurityHelpers::LogErr("failed to get line in hunk",
                                                     "hunk_index", ToString(hunkIdx),
                                                     "line_index", ToString(lineIdx),
                                                     "err", TString(e->message));
                            break;
                        }

                        if (
                            // skip unwanted lines
                            (line->origin != GIT_DIFF_LINE_ADDITION && line->origin != GIT_DIFF_LINE_DELETION)
                            // skip not acceptable lines
                            || (line->content_len < kMinLineLen) || (line->content_len > kMaxLineLen))
                        {
                            continue;
                        }

                        if (line->new_lineno >= 0) {
                            lineNo = static_cast<size_t>(line->new_lineno);
                        } else {
                            lineNo = 0;
                        }

                        size_t lineLen = line->content_len;
                        if (line->content[lineLen - 1] == '\n') {
                            // strip EOL
                            lineLen -= 1;
                        }

                        switch (line->origin) {
                            case GIT_DIFF_LINE_ADDITION:
                                diff.Added.emplace_back(lineNo, TString(line->content, lineLen));
                                break;
                            case GIT_DIFF_LINE_DELETION:
                                // TODO(buglloc): turn on deleted lines back!
                                //diff.Removed.push_back(diffLine);
                                break;
                            default:
                                // pass
                                break;
                        }
                    }
                }

                if (!diff.Added.empty() || !diff.Removed.empty()) {
                    return cb(std::move(diff));
                }
                return true;
            }

            bool
            processDiff(git_repository* repo, const TString& commitId, git_tree* treeA, git_tree* treeB, const TDiffWalkCb& cb) {
                git_diff* diffPtr = nullptr;
                HandleGitResult(
                    git_diff_tree_to_tree(&diffPtr, repo, treeA, treeB, &kDiffFlags),
                    "failed to diff tries");
                TGitDiff diff(diffPtr);

                HandleGitResult(
                    git_diff_find_similar(diffPtr, &kDiffFindFlags),
                    "failed to reduce copied/moved files");

                git_patch* patchPtr = nullptr;
                size_t numDeltas = git_diff_num_deltas(diffPtr);
                for (size_t idx = 0; idx < numDeltas; ++idx) {
                    const git_diff_delta* delta = git_diff_get_delta(diffPtr, idx);
                    if (shouldSkipDelta(delta)) {
                        continue;
                    }

                    HandleGitResult(
                        git_patch_from_diff(&patchPtr, diffPtr, idx),
                        "failed to create patch from diff");
                    TGitPatch patch(patchPtr);

                    if (!invokeCb(patchPtr, commitId, cb)) {
                        return false;
                    }
                }
                return true;
            }

        }

        void TDiffWalker::DiffWalk(const TString& revA, const TString& revB, const TDiffWalkCb& cb) {
            if (revB == kZeroCommit) {
                // Branch got deleted, nothing to do
                return;
            }

            InitGit();
            git_repository* repoPtr = nullptr;
            HandleGitResult(
                git_repository_open_ext(&repoPtr, repoPath.c_str(), kOpenFlags, nullptr),
                "failed to open git repo");
            TGitRepository repo(repoPtr);

            TString initialRev;
            if (revA == kZeroCommit) {
                initialRev = FindFirstRev(repoPtr, revB);
                if (initialRev.empty()) {
                    NSecurityHelpers::LogDebug(
                        "failed to get oldest revision for new branch",
                        "revB", revB,
                        "action", skipKnown ? "exit" : "ignore");

                    if (skipKnown) {
                        return;
                    }
                }
            } else {
                if (skipKnown) {
                    initialRev = FindFirstUnknownRev(repoPtr, revA, revB);
                    if (initialRev.empty()) {
                        NSecurityHelpers::LogDebug("all revs are known, nothing to do");
                        return;
                    }
                } else {
                    initialRev = revA;
                }
            }

            TGitTree treeA;
            if (skipKnown) {
                // in case of skip unknown rev we have first _unknown_ rev in initialRev. So we must use parent revision
                treeA = GetRevParentTree(repoPtr, initialRev);
            } else if (initialRev == revB || initialRev.empty()) {
                treeA = GetRevParentTree(repoPtr, revB);
            } else {
                treeA = GetRevTree(repoPtr, initialRev);
            }

            auto treeB = GetRevTree(repoPtr, revB);

            processDiff(repoPtr, kZeroCommitId, treeA.Get(), treeB.Get(), cb);
        }

        void TDiffWalker::RevsWalk(const TString& revA, const TString& revB, const TDiffWalkCb& cb) {
            if (revB == kZeroCommit) {
                // Branch got deleted, nothing to do
                return;
            }

            InitGit();
            git_repository* repoPtr = nullptr;
            HandleGitResult(
                git_repository_open_ext(&repoPtr, repoPath.c_str(), kOpenFlags, nullptr),
                "failed to open git repo");
            TGitRepository repo(repoPtr);

            git_revwalk* walkPtr = nullptr;
            HandleGitResult(
                git_revwalk_new(&walkPtr, repoPtr),
                "allocate revwalk");
            git_revwalk_sorting(walkPtr, GIT_SORT_TOPOLOGICAL);
            TGitRevWalk walk(walkPtr);

            if (revA != kZeroCommit) {
                // Not a new branch
                PushWalkRev(repoPtr, walkPtr, revA, true);
            }

            if (revA == kZeroCommit || skipKnown) {
                HandleGitResult(
                    git_revwalk_hide_glob(walkPtr, kFilterRef),
                    "filter heads");
            }

            PushWalkRev(repoPtr, walkPtr, revB);

            TDeque<git_oid> spanList;
            {
                git_oid oid;
                size_t oidCount = 0;
                while (!git_revwalk_next(&oid, walkPtr)) {
                    spanList.push_front(oid);
                    if (++oidCount >= kMaxRevs) {
                        break;
                    }
                }
            }

            if (spanList.empty()) {
                return;
            }

            TGitTree treeA(nullptr);
            {
                auto parentCommit = GetParentCommitId(repoPtr, spanList.front());
                if (!git_oid_iszero(&parentCommit)) {
                    // try to create parent tree only if we have the parent commit!
                    treeA.Reset(GetCommitTree(repoPtr, parentCommit));
                }
            }

            TString commitId(GIT_OID_HEXSZ, '\x00');
            for (auto&& oid : spanList) {
                git_oid_fmt(commitId.begin(), &oid);

                NSecurityHelpers::LogDebug("check revision", "rev", commitId);
                auto treeB = GetCommitTree(repoPtr, oid);
                if (!processDiff(repoPtr, commitId, treeA.Get(), treeB.Get(), cb)) {
                    // stop iteration
                    break;
                }

                treeA.Destroy();
                treeA.Reset(treeB);
            }
        }

    }
}
