#pragma once

#include "ptr.h"

#include <contrib/libs/libgit2/include/git2.h>

#include <util/generic/strbuf.h>
#include <util/generic/singleton.h>
#include <util/generic/yexception.h>
#include <security/libs/cpp/log/log.h>

namespace NSecretSearchGitHook {
    namespace NGit {
        namespace {
            const char* kFilterRef{"refs/heads/*"};

        }

        using TGitRepository = THolder<git_repository, TDeleter<decltype(&git_repository_free), &git_repository_free>>;
        using TGitTree = THolder<git_tree, TDeleter<decltype(&git_tree_free), &git_tree_free>>;
        using TGitCommit = THolder<git_commit, TDeleter<decltype(&git_commit_free), &git_commit_free>>;
        using TGitDiff = THolder<git_diff, TDeleter<decltype(&git_diff_free), &git_diff_free>>;
        using TGitObject = THolder<git_object, TDeleter<decltype(&git_object_free), &git_object_free>>;
        using TGitPatch = THolder<git_patch, TDeleter<decltype(&git_patch_free), &git_patch_free>>;
        using TGitRevWalk = THolder<git_revwalk, TDeleter<decltype(&git_revwalk_free), &git_revwalk_free>>;

        struct TInit {
            TInit() {
                git_libgit2_init();
                git_libgit2_opts(GIT_OPT_ENABLE_STRICT_HASH_VERIFICATION, 0);
            }

            ~TInit() {
                git_libgit2_shutdown();
            }
        };

        void InitGit() {
            Singleton<TInit>();
        }

        void HandleGitResult(int result, TStringBuf message) {
            if (result < 0) {
                const git_error* e = giterr_last();
                ythrow TSystemError() << message << ": " << e->message;
            }
        }

        TGitTree GetRevTree(git_repository* repo, const TString& rev) {
            git_object* objPtr = nullptr;
            HandleGitResult(
                git_revparse_single(&objPtr, repo, rev.c_str()),
                "failed to looking up object");
            TGitObject obj(objPtr);

            git_tree* out = nullptr;
            HandleGitResult(
                git_object_peel((git_object**)&out, objPtr, GIT_OBJ_TREE),
                "failed to resolve object to tree");

            Y_ENSURE(out != nullptr);
            return TGitTree(out);
        }

        TGitTree GetRevParentTree(git_repository* repo, const TString& rev) {
            git_object* objPtr = nullptr;
            HandleGitResult(
                git_revparse_single(&objPtr, repo, rev.c_str()),
                "failed to looking up object");
            TGitObject obj(objPtr);

            git_commit* commitPtr = nullptr;
            HandleGitResult(
                git_commit_lookup(&commitPtr, repo, git_object_id(objPtr)),
                "lookup target commit");
            TGitCommit commit(commitPtr);

            git_commit* parentPtr = nullptr;
            {
                if (git_commit_parent(&parentPtr, commitPtr, 0) != 0) {
                    const git_error* e = giterr_last();
                    // Probably this is first commit in repo, nothing special
                    NSecurityHelpers::LogDebug("failed to get parent commit", "rev", rev, "err", TString(e->message));
                    return nullptr;
                }
            }
            TGitCommit parent(parentPtr);

            git_tree* out = nullptr;
            HandleGitResult(
                git_commit_tree(&out, parentPtr),
                "parent tree");

            Y_ENSURE(out != nullptr);
            return TGitTree(out);
        }

        TGitTree GetCommitTree(git_repository* repo, git_oid commitId) {
            git_commit* commitPtr = nullptr;
            HandleGitResult(
                git_commit_lookup(&commitPtr, repo, &commitId),
                "lookup target commit");
            TGitCommit commit(commitPtr);

            git_tree* out = nullptr;
            HandleGitResult(
                git_commit_tree(&out, commitPtr),
                "parent tree");

            Y_ENSURE(out != nullptr);
            return TGitTree(out);
        }

        git_oid GetParentCommitId(git_repository* repo, git_oid commitId) {
            git_commit* commitPtr = nullptr;
            HandleGitResult(
                git_commit_lookup(&commitPtr, repo, &commitId),
                "lookup target commit");
            TGitCommit commit(commitPtr);

            auto parentId = git_commit_parent_id(commitPtr, 0);
            if (parentId) {
                return *parentId;
            }

            return git_oid{0};
        }

        void PushWalkRev(git_repository* repo, git_revwalk* walk, const TString& rev, bool hide = false) {
            git_object* revObjPtr = nullptr;
            HandleGitResult(
                git_revparse_single(&revObjPtr, repo, rev.c_str()),
                "parse revision");
            TGitObject revObj(revObjPtr);

            if (hide) {
                HandleGitResult(
                    git_revwalk_hide(walk, git_object_id(revObjPtr)),
                    "hide revision");
            } else {
                HandleGitResult(
                    git_revwalk_push(walk, git_object_id(revObjPtr)),
                    "show revision");
            }
        }

        TString FindFirstRev(git_repository* repo, const TString& targetRev) {
            git_revwalk* walkPtr = nullptr;
            HandleGitResult(
                git_revwalk_new(&walkPtr, repo),
                "allocate revwalk");
            git_revwalk_sorting(walkPtr, GIT_SORT_TOPOLOGICAL | GIT_SORT_REVERSE);
            TGitRevWalk walk(walkPtr);

            HandleGitResult(
                git_revwalk_hide_glob(walkPtr, kFilterRef),
                "filter heads");

            PushWalkRev(repo, walkPtr, targetRev);

            git_oid oid;
            char buf[GIT_OID_HEXSZ + 1] = {0};
            while (!git_revwalk_next(&oid, walkPtr)) {
                git_oid_fmt(buf, &oid);
                buf[GIT_OID_HEXSZ] = '\0';
                break;
            }
            return buf;
        }

        TString FindFirstUnknownRev(git_repository* repo, const TString& revA, const TString& revB) {
            git_revwalk* walkPtr = nullptr;
            HandleGitResult(
                git_revwalk_new(&walkPtr, repo),
                "allocate revwalk");
            git_revwalk_sorting(walkPtr, GIT_SORT_TOPOLOGICAL | GIT_SORT_REVERSE);
            TGitRevWalk walk(walkPtr);

            PushWalkRev(repo, walkPtr, revA, true);

            HandleGitResult(
                git_revwalk_hide_glob(walkPtr, kFilterRef),
                "filter heads");

            PushWalkRev(repo, walkPtr, revB);

            git_oid oid;
            char buf[GIT_OID_HEXSZ + 1] = {0};
            while (!git_revwalk_next(&oid, walkPtr)) {
                git_oid_fmt(buf, &oid);
                buf[GIT_OID_HEXSZ] = '\0';
                break;
            }
            return buf;
        }

    }
}
