#include "searcher.h"
#include "diff_source.h"
#include "git.h"

#include <security/ant-secret/secret-search/public/cpp/stream_output.h>

#include <security/libs/cpp/log/log.h>

namespace NSecretSearchGitHook {
    namespace {
        const size_t kQueueSize = 1024;

    }

    TSearcher::TSearcher(const TSearchOptions& opts, size_t numThreads)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , numThreads(numThreads)
    {
        Y_ENSURE(!opts.RepoPath.empty(), "repo path is empty");

        Y_ENSURE(numThreads > 0, "num threads must be gather than 0");

        for (size_t i = 0; i < numThreads; ++i) {
            workers.emplace_front(ctx);
        }
    }

    void TSearcher::Stop() {
        AtomicSet(stopped, 1);
        for (size_t i = 0; i < numThreads; ++i) {
            workers[i].Stop();
        }
    }

    void TSearcher::CheckDiff(TOutput& writer, const TString& revA, const TString& revB) {
        if (AtomicGet(stopped)) {
            return;
        }

        TJobsQueue jobsQueue(kQueueSize);

        auto* threadPool = SystemThreadFactory();
        TDeque<TAutoPtr<IThreadFactory::IThread>> threads;
        for (auto&& worker : workers) {
            worker.ResetState(&jobsQueue, &writer);
            threads.emplace_front(threadPool->Run(&worker));
        }

        auto walkFn = [&jobsQueue, this](TDiff diff) -> bool {
            if (AtomicGet(stopped)) {
                // stop iterate
                return false;
            }

            if (!diff.TargetFile) {
                // skip deleted files
                return true;
            }

            TSourceDiff source = TSourceDiff::FromDiff(diff);
            jobsQueue.Push(std::move(source));
            return true;
        };

        try {
            NGit::TDiffWalker walker(opts.RepoPath);
            walker.SetSkipKnown(opts.SkipKnownRevs);

            if (opts.EachRev) {
                walker.RevsWalk(revA, revB, walkFn);
            } else {
                walker.DiffWalk(revA, revB, walkFn);
            }
        } catch (const yexception& e) {
            NSecurityHelpers::LogErr("Failed to check update",
                                     "revA", revA,
                                     "revB", revB,
                                     "err", e.AsStrBuf());
        }

        // Wait workers
        jobsQueue.Stop();
        for (auto& thread : threads) {
            thread->Join();
        }
    }

}
