#pragma once

#include <security/ant-secret/secret-search/git-hook/lib/ptr.h>

#include <contrib/libs/libgit2/include/git2.h>
#include <util/folder/path.h>
#include <util/stream/file.h>

namespace NSecretSearchGitHook {
    namespace NTests {
        namespace {
            void writeFile(const TFsPath& path, const TString& content) {
                TFileOutput out(path);
                out.Write(content);
                out.Finish();
            }

        }

        using TGitRepository = THolder<git_repository, TDeleter<decltype(&git_repository_free), &git_repository_free>>;
        using TGitTree = THolder<git_tree, TDeleter<decltype(&git_tree_free), &git_tree_free>>;
        using TGitCommit = THolder<git_commit, TDeleter<decltype(&git_commit_free), &git_commit_free>>;
        using TGitDiff = THolder<git_diff, TDeleter<decltype(&git_diff_free), &git_diff_free>>;
        using TGitObject = THolder<git_object, TDeleter<decltype(&git_object_free), &git_object_free>>;
        using TGitPatch = THolder<git_patch, TDeleter<decltype(&git_patch_free), &git_patch_free>>;
        using TGitRevWalk = THolder<git_revwalk, TDeleter<decltype(&git_revwalk_free), &git_revwalk_free>>;
        using TGitIndex = THolder<git_index, TDeleter<decltype(&git_index_free), &git_index_free>>;
        using TGitSignature = THolder<git_signature, TDeleter<decltype(&git_signature_free), &git_signature_free>>;

        struct TInit {
            inline TInit() {
                git_libgit2_init();
            }

            inline ~TInit() {
                git_libgit2_shutdown();
            }
        };

        inline void InitGit() {
            Singleton<TInit>();
        }

        inline void HandleGitResult(int result, TStringBuf message) {
            if (result < 0) {
                const git_error* e = giterr_last();
                ythrow TSystemError() << message << ": " << e->message;
            }
        }

        inline git_oid AddFile(
            git_repository* repo,
            const TString& path,
            const TString& content,
            git_oid parentId = {0},
            const TString& updateRef = TString()) {
            git_index* idx = nullptr;
            HandleGitResult(
                git_repository_index(&idx, repo),
                "get repo index");

            TGitIndex idxHolder(idx);

            writeFile(TFsPath(git_repository_workdir(repo)) / path, content);
            HandleGitResult(
                git_index_add_bypath(idx, path.c_str()),
                "add file");

            git_commit* parent = nullptr;
            if (!git_oid_iszero(&parentId)) {
                HandleGitResult(
                    git_commit_lookup(&parent, repo, &parentId),
                    "commit lookup");
            }
            TGitCommit parentHolder(parent);

            git_signature* author;
            git_signature_new(&author, "test", "test@yt.ru", 123456789, 60);
            TGitSignature authorHolder(author);

            git_oid tree_id;
            git_index_write_tree(&tree_id, idx);
            git_tree* tree;
            HandleGitResult(
                git_tree_lookup(&tree, repo, &tree_id),
                "tree lookup");
            TGitTree treeHolder(tree);

            git_oid commit_id;
            git_commit_create_v(
                &commit_id, /* out id */
                repo,
                updateRef ? updateRef.c_str() : nullptr,
                author,
                author,
                nullptr,
                "commit",
                tree,
                1,
                parent);

            return commit_id;
        }

        inline TGitRepository CreateGitRepo(const TString& path) {
            InitGit();
            git_repository* repoPtr = nullptr;
            HandleGitResult(
                git_repository_init(&repoPtr, path.c_str(), 0),
                "failed to init git repo");
            return TGitRepository{repoPtr};
        }

    }
}
