#include "common.h"

#include <security/ant-secret/secret-search/git-hook/lib/git.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>
#include <util/generic/deque.h>

Y_UNIT_TEST_SUITE(TGitTests) {
    using namespace NSecretSearchGitHook;
    using namespace NGit;

    Y_UNIT_TEST(Initial) {
        TTempDir tempDir;
        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test.txt", "first");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        TString secondCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto secondCommitId = NTests::AddFile(repo.Get(), "test.txt", "second\n", firstCommitId);
        git_oid_fmt(secondCommitIdHex.begin(), &secondCommitId);
        Cout << "second commit id: " << secondCommitIdHex << Endl;

        TString thirdCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto thirdCommitId = NTests::AddFile(repo.Get(), "test.txt", "third", secondCommitId);
        git_oid_fmt(thirdCommitIdHex.begin(), &thirdCommitId);
        Cout << "third commit id: " << thirdCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        bool called = false;
        {
            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            called = false;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, firstCommitIdHex);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }
        }

        {
            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            TDeque<TString> commitHist;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    commitHist.push_back(diff.CommitId);

                    if (diff.CommitId == firstCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, firstCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    } else if (diff.CommitId == secondCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, secondCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "second");
                    } else if (diff.CommitId == thirdCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, thirdCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else {
                        ythrow yexception() << "unknown commit id: " << diff.CommitId << Endl;
                    }

                    return true;
                });

            // check commits order
            UNIT_ASSERT_VALUES_EQUAL(commitHist.size(), 3);
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), firstCommitIdHex);
            commitHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), secondCommitIdHex);
            commitHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), thirdCommitIdHex);
            commitHist.pop_front();
        }
    }

    Y_UNIT_TEST(InitialSingle) {
        TTempDir tempDir;
        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test.txt", "first");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        bool called = false;
        {
            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            called = false;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, firstCommitIdHex);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }
        }

        walker.SetSkipKnown(true);
        {
            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            called = false;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, firstCommitIdHex);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "first");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }
        }
    }

    Y_UNIT_TEST(GeneralFlows) {
        TTempDir tempDir;

        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test.txt", "first", {0}, "HEAD");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        TString secondCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto secondCommitId = NTests::AddFile(repo.Get(), "test.txt", "second", firstCommitId, "HEAD");
        git_oid_fmt(secondCommitIdHex.begin(), &secondCommitId);
        Cout << "second commit id: " << secondCommitIdHex << Endl;

        TString thirdCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto thirdCommitId = NTests::AddFile(repo.Get(), "test.txt", "third", secondCommitId, "HEAD");
        git_oid_fmt(thirdCommitIdHex.begin(), &thirdCommitId);
        Cout << "third commit id: " << thirdCommitIdHex << Endl;

        TString bareCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto bareCommitId = NTests::AddFile(repo.Get(), "test.txt", "bare\n", thirdCommitId);
        git_oid_fmt(bareCommitIdHex.begin(), &bareCommitId);
        Cout << "bare commit id: " << bareCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        bool called = false;
        {
            // delete branch
            walker.DiffWalk(
                firstCommitIdHex,
                "0000000000000000000000000000000000000000",
                [&](TDiff diff) -> bool {
                    Y_UNUSED(diff);
                    ythrow yexception() << "called diff for deleted branch";
                });

            walker.RevsWalk(
                firstCommitIdHex,
                "0000000000000000000000000000000000000000",
                [&](TDiff diff) -> bool {
                    Y_UNUSED(diff);
                    ythrow yexception() << "called diff for deleted branch";
                });
        }

        {
            called = false;
            walker.DiffWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            TDeque<TString> commitHist;
            walker.RevsWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    commitHist.push_back(diff.CommitId);

                    if (diff.CommitId == secondCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, secondCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "second");
                    } else if (diff.CommitId == thirdCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, thirdCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else {
                        ythrow yexception() << "unknown commit id: " << diff.CommitId << Endl;
                    }

                    return true;
                });

            // check commits order
            UNIT_ASSERT_VALUES_EQUAL(commitHist.size(), 2);
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), secondCommitIdHex);
            commitHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), thirdCommitIdHex);
            commitHist.pop_front();
        }

        {
            // imitate new branch
            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                bareCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "bare");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            called = false;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                bareCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, bareCommitIdHex);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "bare");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }
        }
    }

    Y_UNIT_TEST(SkipKnownRevsSimple) {
        TTempDir tempDir;

        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test_first.txt", "first", {0}, "HEAD");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        TString secondCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto secondCommitId = NTests::AddFile(repo.Get(), "test_second.txt", "second", firstCommitId, "HEAD");
        git_oid_fmt(secondCommitIdHex.begin(), &secondCommitId);
        Cout << "second commit id: " << secondCommitIdHex << Endl;

        TString thirdCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto thirdCommitId = NTests::AddFile(repo.Get(), "test_third.txt", "third", secondCommitId);
        git_oid_fmt(thirdCommitIdHex.begin(), &thirdCommitId);
        Cout << "third commit id: " << thirdCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        bool called = false;
        {
            // W/o skipRevs we walk from revA to revB
            walker.SetSkipKnown(false);

            called = false;
            TDeque<TString> fileHist;
            walker.DiffWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    fileHist.push_back(diff.TargetFile);

                    if (diff.TargetFile == "test_second.txt") {
                        UNIT_ASSERT(diff.CommitId.empty());
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "second");
                    } else if (diff.TargetFile == "test_third.txt") {
                        UNIT_ASSERT(diff.CommitId.empty());
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else {
                        ythrow yexception() << "unknown file in diff: " << diff.TargetFile << Endl;
                    }

                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            // check files order
            UNIT_ASSERT_VALUES_EQUAL(fileHist.size(), 2);
            UNIT_ASSERT_STRINGS_EQUAL(fileHist.front(), "test_second.txt");
            fileHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(fileHist.front(), "test_third.txt");

            called = false;
            TDeque<TString> commitHist;
            walker.RevsWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    commitHist.push_back(diff.CommitId);

                    if (diff.CommitId == secondCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, secondCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_second.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "second");
                    } else if (diff.CommitId == thirdCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, thirdCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_third.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else {
                        ythrow yexception() << "unknown commit id: " << diff.CommitId << Endl;
                    }

                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            // check commits order
            UNIT_ASSERT_VALUES_EQUAL(commitHist.size(), 2);
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), secondCommitIdHex);
            commitHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), thirdCommitIdHex);
            commitHist.pop_front();
        }

        {
            // W/ skipRevs we walk from revA to revB, except known commits
            walker.SetSkipKnown(true);

            called = false;
            walker.DiffWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, "");
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_third.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");

                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            called = false;
            walker.RevsWalk(
                firstCommitIdHex,
                thirdCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, thirdCommitIdHex);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_third.txt");
                    UNIT_ASSERT(diff.Fulfilled);
                    UNIT_ASSERT(diff.Removed.empty());
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                    UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }
        }
    }

    Y_UNIT_TEST(SkipKnownRevsMultiple) {
        TTempDir tempDir;

        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test_first.txt", "first", {0}, "HEAD");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        TString secondCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto secondCommitId = NTests::AddFile(repo.Get(), "test_second.txt", "second", firstCommitId, "HEAD");
        git_oid_fmt(secondCommitIdHex.begin(), &secondCommitId);
        Cout << "second commit id: " << secondCommitIdHex << Endl;

        TString thirdCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto thirdCommitId = NTests::AddFile(repo.Get(), "test_third.txt", "third", secondCommitId);
        git_oid_fmt(thirdCommitIdHex.begin(), &thirdCommitId);
        Cout << "third commit id: " << thirdCommitIdHex << Endl;

        TString fourthCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto fourthCommitId = NTests::AddFile(repo.Get(), "test_fourth.txt", "fourth", thirdCommitId);
        git_oid_fmt(fourthCommitIdHex.begin(), &fourthCommitId);
        Cout << "third fourth id: " << fourthCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        walker.SetSkipKnown(true);
        bool called = false;
        {
            called = false;
            TDeque<TString> fileHist;
            walker.DiffWalk(
                firstCommitIdHex,
                fourthCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    fileHist.push_back(diff.TargetFile);

                    if (diff.TargetFile == "test_third.txt") {
                        UNIT_ASSERT(diff.CommitId.empty());
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else if (diff.TargetFile == "test_fourth.txt") {
                        UNIT_ASSERT(diff.CommitId.empty());
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "fourth");
                    } else {
                        ythrow yexception() << "unknown file in diff: " << diff.TargetFile << Endl;
                    }

                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            // check tha we have called two times
            UNIT_ASSERT_VALUES_EQUAL(fileHist.size(), 2);
        }

        {
            called = false;
            TDeque<TString> commitHist;
            walker.RevsWalk(
                firstCommitIdHex,
                fourthCommitIdHex,
                [&](TDiff diff) -> bool {
                    called = true;
                    commitHist.push_back(diff.CommitId);

                    if (diff.CommitId == thirdCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, thirdCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_third.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "third");
                    } else if (diff.CommitId == fourthCommitIdHex) {
                        UNIT_ASSERT_STRINGS_EQUAL(diff.CommitId, fourthCommitIdHex);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "test_fourth.txt");
                        UNIT_ASSERT(diff.Fulfilled);
                        UNIT_ASSERT(diff.Removed.empty());
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.size(), 1);
                        UNIT_ASSERT_VALUES_EQUAL(diff.Added.front().LineNo, 1);
                        UNIT_ASSERT_STRINGS_EQUAL(diff.Added.front().Line, "fourth");
                    } else {
                        ythrow yexception() << "unknown commit id: " << diff.CommitId << Endl;
                    }

                    return true;
                });

            if (!called) {
                ythrow yexception() << "diffCb not called";
            }

            // check commits order
            UNIT_ASSERT_VALUES_EQUAL(commitHist.size(), 2);
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), thirdCommitIdHex);
            commitHist.pop_front();
            UNIT_ASSERT_STRINGS_EQUAL(commitHist.front(), fourthCommitIdHex);
        }
    }

    Y_UNIT_TEST(SkipKnownRevsAllKnown) {
        TTempDir tempDir;

        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto repo = NTests::CreateGitRepo(repoPath);

        TString firstCommitIdHex(GIT_OID_HEXSZ, '\x00');
        auto firstCommitId = NTests::AddFile(repo.Get(), "test_first.txt", "first", {0}, "HEAD");
        git_oid_fmt(firstCommitIdHex.begin(), &firstCommitId);
        Cout << "first commit id: " << firstCommitIdHex << Endl;

        NGit::TDiffWalker walker(repoPath);
        bool called = false;
        {
            walker.SetSkipKnown(true);

            called = false;
            walker.DiffWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    Y_UNUSED(diff);
                    called = true;
                    return true;
                });

            if (called) {
                ythrow yexception() << "diffCb called, but must not";
            }

            called = false;
            walker.RevsWalk(
                "0000000000000000000000000000000000000000",
                firstCommitIdHex,
                [&](TDiff diff) -> bool {
                    Y_UNUSED(diff);
                    called = true;
                    return true;
                });

            if (called) {
                ythrow yexception() << "diffCb called, but must not";
            }
        }
    }
}
