#include "common.h"

#include <security/ant-secret/secret-search/git-hook/lib/searcher.h>

#include <util/generic/deque.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>
#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TSearcherTests) {
    using namespace NSecretSearchGitHook;

    namespace {
        class TOutputHolder: public TOutput {
        public:
            TOutputHolder()
                : TOutput(Cout){};

            TDeque<TDiffResult> Results() {
                return results;
            }

            void Write(const TDiffResult& result) override {
                results.push_back(result);
            };

        protected:
            TDeque<TDiffResult> results;
        };

        struct TRepoInfo {
            TString FirstCommitId{GIT_OID_HEXSZ, '\x00'};
            TString SecretCommitId{GIT_OID_HEXSZ, '\x00'};
            TString LastCommitId{GIT_OID_HEXSZ, '\x00'};
            TDeque<TDiffResult> Results;
        };

        TRepoInfo prepareRepo(const TString& repoPath, bool hideSecret = false, const TString& ref = TString()) {
            TRepoInfo result{};

            auto repo = NTests::CreateGitRepo(repoPath);

            auto firstCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "not secret", {0}, ref);
            git_oid_fmt(result.FirstCommitId.begin(), &firstCommitId);

            auto secondCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "token = '03c7c0ace395d80182db07ae2c30f034'",
                                                  firstCommitId, ref);
            git_oid_fmt(result.SecretCommitId.begin(), &secondCommitId);

            if (!hideSecret) {
                result.LastCommitId = result.SecretCommitId;
                return result;
            }

            auto thirdCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "token = os.env['token']", secondCommitId, ref);
            git_oid_fmt(result.LastCommitId.begin(), &thirdCommitId);
            return result;
        }

        TRepoInfo SimpleCheck(bool eachRev, bool hideSecret) {
            TTempDir tempDir;
            auto repoPath = tempDir.Path() / "git";
            MakePathIfNotExist(repoPath.c_str(), 0777);

            auto result = prepareRepo(repoPath, hideSecret);

            TOutputHolder writer;
            writer.Start();
            TSearchOptions opts{
                .RepoPath = repoPath.GetPath(),
                .Validate = false,
                .ValidOnly = false,
                .EachRev = eachRev,
                .SkipKnownRevs = false,
            };
            TSearcher searcher(opts, 2);
            searcher.CheckDiff(writer, "0000000000000000000000000000000000000000", result.LastCommitId);
            writer.Finish();

            result.Results = writer.Results();
            return result;
        }

        void CheckResults(const TDeque<TDiffResult>& results, const TString& commitId) {
            UNIT_ASSERT_VALUES_EQUAL(results.size(), 1);
            auto result = results.front();

            UNIT_ASSERT_STRINGS_EQUAL(result.CommitId, commitId);
            UNIT_ASSERT_STRINGS_EQUAL(result.Path, "oauth.txt");

            UNIT_ASSERT_VALUES_EQUAL(result.SearcherResults.size(), 1);
            auto secret = result.SearcherResults.front();

            UNIT_ASSERT_EQUAL(secret.LineNo, 1);
            UNIT_ASSERT_EQUAL(secret.Secret, "03c7c0ace395d80182db07ae2c30f034");
            UNIT_ASSERT(!secret.Additional.empty());
            UNIT_ASSERT_EQUAL(secret.Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
            UNIT_ASSERT(!secret.Validated);
            UNIT_ASSERT(!secret.Ignore);
        }

    }

    Y_UNIT_TEST(WholeDiffHideSecrets) {
        auto repo = SimpleCheck(false, true);
        UNIT_ASSERT(repo.Results.empty());
    }

    Y_UNIT_TEST(WholeDiff) {
        auto repo = SimpleCheck(false, false);
        UNIT_ASSERT(!repo.Results.empty());
        CheckResults(repo.Results, "");
    }

    Y_UNIT_TEST(RevListHideSecrets) {
        auto repo = SimpleCheck(true, true);
        UNIT_ASSERT(!repo.Results.empty());
        CheckResults(repo.Results, repo.SecretCommitId);
    }

    Y_UNIT_TEST(RevList) {
        auto repo = SimpleCheck(true, false);
        UNIT_ASSERT(!repo.Results.empty());
        CheckResults(repo.Results, repo.SecretCommitId);
    }

    Y_UNIT_TEST(Output) {
        TTempDir tempDir;
        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        auto result = prepareRepo(repoPath, true);

        TStringStream out;
        TOutput writer(out);
        writer.Start();
        TSearchOptions opts{
            .RepoPath = repoPath.GetPath(),
            .Validate = false,
            .ValidOnly = false,
            .EachRev = true,
            .SkipKnownRevs = false,
        };
        TSearcher searcher(opts, 2);
        searcher.CheckDiff(writer, "0000000000000000000000000000000000000000", result.LastCommitId);
        writer.Finish();
        out.Finish();

        auto output = out.Str();
        // not empty
        UNIT_ASSERT(!writer.Empty());
        // have commit
        UNIT_ASSERT_STRING_CONTAINS(output, result.SecretCommitId);
        // have path ant line no
        UNIT_ASSERT_STRING_CONTAINS(output, "/oauth.txt:1");
        // have token type
        UNIT_ASSERT_STRING_CONTAINS(output, "YandexOAuth");
    }

    Y_UNIT_TEST(SkipKnownRevs) {
        TTempDir tempDir;
        auto repoPath = tempDir.Path() / "git";
        MakePathIfNotExist(repoPath.c_str(), 0777);

        TRepoInfo repoInfo{};

        auto repo = NTests::CreateGitRepo(repoPath);

        // first commit
        auto firstCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "not secret", {0}, "HEAD");
        git_oid_fmt(repoInfo.FirstCommitId.begin(), &firstCommitId);

        // second commit WITH secret but referenced to HEAD
        auto secondCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "token = '03c7c0ace395d80182db07ae2c30f111'",
                                              firstCommitId, "HEAD");

        // last commit WITH secret but not referenced to any refs
        Y_UNUSED(secondCommitId);
        auto thirdCommitId = NTests::AddFile(repo.Get(), "oauth.txt", "token = '03c7c0ace395d80182db07ae2c30f034'",
                                             secondCommitId);
        git_oid_fmt(repoInfo.SecretCommitId.begin(), &thirdCommitId);
        repoInfo.LastCommitId = repoInfo.SecretCommitId;

        TOutputHolder writer;
        writer.Start();
        TSearchOptions opts{
            .RepoPath = repoPath.GetPath(),
            .Validate = false,
            .ValidOnly = false,
            .EachRev = true,
            .SkipKnownRevs = true,
        };
        TSearcher searcher(opts, 2);
        searcher.CheckDiff(writer, repoInfo.FirstCommitId, repoInfo.LastCommitId);
        writer.Finish();

        UNIT_ASSERT(!writer.Results().empty());
        CheckResults(writer.Results(), repoInfo.SecretCommitId);
    }
}
