#include "filewalker.h"
#include "path.h"
#include "secrets_ignore.h"

#include <security/libs/cpp/log/log.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/folder/filelist.h>
#include <util/system/file.h>
#include <util/folder/path.h>

namespace NSSInternal {
    namespace NFileWalker {
        namespace {
            constexpr TStringBuf kSecretIgnoreName = ".secretsignore";

            inline bool isExcludedByUser(const TVector<TPathMatcher>& userExcludes, const TString& path) {
                for (const auto& excl : userExcludes) {
                    if (excl.Match(path)) {
                        return true;
                    }
                }
                return false;
            }

        }

        TWalker::TWalker(const TWalkerOptions& options) {
            if (!options.Excludes.empty()) {
                systemExcluder.Reset(new TDirMatcher(options.Excludes));
            }
        }

        bool TWalker::Walk(const TFsPath& rootPath, const TWorkerCb& cb) const {
            try {
                if (Y_UNLIKELY(!rootPath.Exists())) {
                    ythrow TSystemError() << rootPath << " not exists";
                }

                if (rootPath.IsFile()) {
                    cb(rootPath);
                    return true;
                }

                TVector<TPathMatcher> userExcludes;
                return WalkImpl(rootPath, cb, userExcludes);
            } catch (const yexception& e) {
                NSecurityHelpers::LogErr("Failed to walk path", "err", e.AsStrBuf(), "path", rootPath.GetPath());
                return false;
            }
        }

        bool TWalker::WalkImpl(const TString& rootPath, const TWorkerCb& cb, TVector<TPathMatcher>& userExcludes) const {
            bool popUserExcludes = false;
            {
                TFileList fl;
                const char* name;
                const auto& secretIgnorePath = TString::Join(rootPath, PathSep, kSecretIgnoreName);

                if (NFs::Exists(secretIgnorePath)) {
                    TVector<TString> excludes;
                    TVector<TString> includes;
                    TSecretsIgnore::ParseFile(secretIgnorePath, includes, excludes);
                    if (!includes.empty() || !excludes.empty()) {
                        popUserExcludes = true;
                        userExcludes.emplace_back(rootPath + PathSep, includes, excludes);
                    }
                }

                fl.Fill(rootPath, false);
                while ((name = fl.Next())) {
                    if (name == kSecretIgnoreName) {
                        continue;
                    }

                    auto path = rootPath + PathSep + name;
                    if (isExcludedByUser(userExcludes, path)) {
                        NSecurityHelpers::LogDebug("Skip file", "source", "user", "path", path);
                        continue;
                    }

                    if (!cb(path)) {
                        NSecurityHelpers::LogErr("Stopped file walking", "path", path);
                        return false;
                    }
                }
            }

            {
                TDirsList dl;
                const char* name;

                dl.Fill(rootPath, false);
                while ((name = dl.Next())) {
                    auto path = rootPath + PathSep + name;
                    if (systemExcluder && systemExcluder->Match(name)) {
                        NSecurityHelpers::LogDebug("Skip dir", "source", "system", "path", path);
                        continue;
                    }

                    if (isExcludedByUser(userExcludes, path)) {
                        NSecurityHelpers::LogDebug("Skip dir", "source", "user", "path", path);
                        continue;
                    }

                    if (!WalkImpl(path, cb, userExcludes)) {
                        NSecurityHelpers::LogErr("Stopped file walking", "path", path);
                        return false;
                    }
                }
            }

            if (popUserExcludes) {
                userExcludes.pop_back();
            }

            return true;
        }

    }
}
