#include "path_matcher.h"
#include "path.h"

#include <security/libs/cpp/log/log.h>
#include <library/cpp/regex/pcre/regexp.h>
#include <util/stream/str.h>
#include <util/string/builder.h>

namespace NSSInternal {
    namespace NFileWalker {
        namespace {
            inline TRegExMatch* compileRegexes(const TString& prefix, const TVector<TString>& patterns) {
                TStringBuilder Rex;
                if (patterns.empty()) {
                    return nullptr;
                } else if (patterns.size() > 1) {
                    for (const auto& p : patterns) {
                        if (!Rex.empty())
                            Rex << "|";
                        Rex << TString::Join("(?:", p + ")");
                    }
                } else {
                    Rex << patterns[0];
                }

                return new TRegExMatch(TString::Join("^", prefix, "(?:", Rex, ")", "$"));
            }

            const TString ParsePattern(const TString& pattern) {
                if (pattern.empty()) {
                    return TString();
                }

                TStringBuilder reg;
                TStringInput stream(pattern);
                char ch;
                bool haveNext = stream.ReadChar(ch);
                while (haveNext) {
                    if (ch == '*') {
                        if ((haveNext = stream.ReadChar(ch)) && ch == '*') {
                            // is some flavor of "**"

                            haveNext = stream.ReadChar(ch);
                            if (ch == PathSep) {
                                // treat **/ as ** so eat the "/"
                                haveNext = stream.ReadChar(ch);
                            }

                            if (!haveNext) {
                                // is "**EOF" - to align with .gitignore just accept all
                                reg << ".*";
                            } else {
                                reg << "(?:.*" << PathSepEsc << ")?";
                            }
                        } else {
                            // is "*" so map it to anything but "/"
                            reg << "[^" << PathSepEsc << "]*";
                        }
                        // manually reading of next char :(
                        continue;
                    } else if (ch == '?') {
                        // "?" is any char except "/"
                        reg << "[^" << PathSepEsc << "]";
                    } else if (ch == '.' || ch == '+' || ch == '|' || ch == '$' || ch == '(' || ch == ')') {
                        // escape some regexp special chars
                        reg << "\\" << ch;
                    }
#ifdef _win_
                    else if (ch == '\\') {
                        // On windows map "\" to "\\", meaning an escaped backslash,
                        // and then just continue because filepath.Match on
                        // Windows doesn't allow escaping at all
                        regOut << PathSepEsc << ch;
                    }
#endif
                    else {
                        reg << ch;
                    }
                    haveNext = stream.ReadChar(ch);
                }

                return reg;
            }

        }

        TPathMatcher::TPathMatcher(const TString& prefix, const TVector<TString>& includesPatterns) {
            if (!includesPatterns.empty()) {
                try {
                    includes.Reset(compileRegexes(prefix, ParsePatterns(includesPatterns)));
                } catch (const yexception& e) {
                    NSecurityHelpers::LogErr("failed to compile excludes", "err", e.what());
                }
            }
        }

        TPathMatcher::TPathMatcher(const TString& prefix, const TVector<TString>& includesPatterns,
                                   const TVector<TString>& excludePatterns) {
            if (!includesPatterns.empty()) {
                try {
                    includes.Reset(compileRegexes(prefix, ParsePatterns(includesPatterns)));
                } catch (const yexception& e) {
                    NSecurityHelpers::LogErr("failed to compile excludes", "err", e.what());
                }
            }

            if (!excludePatterns.empty()) {
                try {
                    excludes.Reset(compileRegexes(prefix, ParsePatterns(excludePatterns)));
                } catch (const yexception& e) {
                    NSecurityHelpers::LogErr("failed to compile excludes", "err", e.what());
                }
            }
        }

        bool TPathMatcher::Match(const TString& path) const {
            if (excludes) {
                if (excludes->Match(path.c_str())) {
                    return false;
                }
            }

            if (includes) {
                if (includes->Match(path.c_str())) {
                    return true;
                }
            }

            return false;
        }

        const TVector<TString> TPathMatcher::ParsePatterns(const TVector<TString>& patterns) {
            TVector<TString> result;
            for (const auto& pattern : patterns) {
                auto parsed = ParsePattern(pattern);
                if (!parsed.empty()) {
                    result.push_back(parsed);
                }
            }
            return result;
        }

    }
}
