#include <security/ant-secret/secret-search/internal/filewalker/filewalker.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>
#include <util/system/file.h>
#include <util/stream/file.h>

using namespace NSSInternal::NFileWalker;

Y_UNIT_TEST_SUITE(TFileWalker) {
    Y_UNIT_TEST(Basic) {
        TTempDir tempDir("nonexistingdir");
        const auto& tempDirPath = tempDir.Path();

        const auto file0Name = tempDirPath / "file0";
        TFile(file0Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir");
        const auto file1Name = tempDirPath / "subdir" / "file1";
        TFile(file1Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir1");
        const auto file2Name = tempDirPath / "subdir1" / "file2";
        TFile(file2Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "exclude");
        const auto file3Name = tempDirPath / "exclude" / "file3";
        TFile(file3Name, CreateAlways);

        TWalkerOptions opts = {
            .Excludes = {"exclude"}};
        TWalker walker(opts);
        TVector<const TString> paths;
        walker.Walk(tempDir(), [&paths](const TString& path) -> bool {
            paths.push_back(path);
            return true;
        });

        UNIT_ASSERT_EQUAL(paths.size(), 3);
        UNIT_ASSERT(std::find(paths.begin(), paths.end(), file0Name.GetPath()) != paths.end());
        UNIT_ASSERT(std::find(paths.begin(), paths.end(), file1Name.GetPath()) != paths.end());
        UNIT_ASSERT(std::find(paths.begin(), paths.end(), file2Name.GetPath()) != paths.end());
    }

    Y_UNIT_TEST(Stoppable) {
        TTempDir tempDir("nonexistingdir");
        const auto& tempDirPath = tempDir.Path();

        const auto file0Name = tempDirPath / "file0";
        TFile(file0Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir");
        const auto file1Name = tempDirPath / "subdir" / "file1";
        TFile(file1Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir1");
        const auto file2Name = tempDirPath / "subdir1" / "file2";
        TFile(file2Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "exclude");
        const auto file3Name = tempDirPath / "exclude" / "file3";
        TFile(file3Name, CreateAlways);

        TWalkerOptions opts;
        TWalker walker(opts);
        TVector<const TString> paths;
        walker.Walk(tempDir(), [&paths](const TString& path) -> bool {
            paths.push_back(path);
            // stop iterate
            return false;
        });

        UNIT_ASSERT_EQUAL(paths.size(), 1);
        UNIT_ASSERT(
            std::find(paths.begin(), paths.end(), file0Name.GetPath()) != paths.end() || std::find(paths.begin(), paths.end(), file1Name.GetPath()) != paths.end() || std::find(paths.begin(), paths.end(), file2Name.GetPath()) != paths.end());
    }

    Y_UNIT_TEST(ExcludedSecretsIgnore) {
        TTempDir tempDir("nonexistingdir");
        TFile(tempDir.Path() / ".secretsignore", CreateAlways);

        TWalkerOptions opts;
        TWalker walker(opts);
        TVector<const TString> paths;
        walker.Walk(tempDir.Path(), [&paths](const TString& path) -> bool {
            paths.push_back(path);
            return true;
        });
        UNIT_ASSERT_EQUAL(paths.size(), 0);
    }

    Y_UNIT_TEST(SecretsIgnore) {
        TTempDir tempDir("nonexistingdir");
        const auto& tempDirPath = tempDir.Path();

        {
            TString secretsIgnoreData = R"(
                file0
                subdir/file1
                subdir2
            )";
            TFileOutput secretsIgnore(tempDirPath / ".secretsignore");
            secretsIgnore.Write(secretsIgnoreData);
            secretsIgnore.Flush();
        }

        const auto file0Name = tempDirPath / "file0";
        TFile(file0Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir");
        const auto file1Name = tempDirPath / "subdir" / "file1";
        TFile(file1Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir1");
        const auto file2Name = tempDirPath / "subdir1" / "file2";
        TFile(file2Name, CreateAlways);

        MakeDirIfNotExist(tempDirPath / "subdir2");
        const auto file3Name = tempDirPath / "subdir2" / "file3";
        TFile(file3Name, CreateAlways);

        TWalkerOptions opts;
        TWalker walker(opts);
        TVector<const TString> paths;
        walker.Walk(tempDirPath, [&paths](const TString& path) -> bool {
            paths.push_back(path);
            return true;
        });

        UNIT_ASSERT_EQUAL(paths.size(), 1);
        UNIT_ASSERT(std::find(paths.begin(), paths.end(), file2Name.GetPath()) != paths.end());
    }

    Y_UNIT_TEST(SingleFile) {
        TTempDir tempDir("nonexistingdir");
        const auto& tempDirPath = tempDir.Path();
        const auto filePath = tempDirPath / "file";
        TFile(filePath, CreateAlways);

        TWalkerOptions opts;
        TWalker walker(opts);
        TVector<const TString> paths;
        walker.Walk(filePath.GetPath(), [&paths](const TString& path) -> bool {
            paths.push_back(path);
            return true;
        });

        UNIT_ASSERT_EQUAL(paths.size(), 1);
        UNIT_ASSERT(std::find(paths.begin(), paths.end(), filePath.GetPath()) != paths.end());
    }
}
