#include "search_worker.h"

#include "json_walker.h"
#include <security/libs/cpp/log/log.h>

namespace NSSInternal {
    namespace {
        constexpr size_t kMinLineLen = 5;
        constexpr size_t kMaxLineLen = 4096;

        using TFilteredSearchers = THashMap<TString, NSearchers::ISearcher *>;

        TFilteredSearchers FilterSearchers(const TSearchers &searchers, ISource &source) {
            TFilteredSearchers result;
            for (auto&&[name, search]: searchers) {
                if (source.Path() && !search->IsAcceptablePath(source.Path())) {
                    continue;
                }

                if (search->Binary() && !source.IsBinary()) {
                    continue;
                }

                if (source.IsBinary() && !search->Binary()) {
                    continue;
                }

                result[name] = search.Get();
            }
            return result;
        }

    }

    void TSearchWorkerBase::CollectRegexes() {
        // TODO(buglloc)
    }

    TMaybe<TPathResult> TSearchWorkerBase::CheckPath(const TString &path) {
        try {
            TSourceFile sourceFile(path);

            if (sourceFile.Size() == 0) {
                return Nothing();
            }

            if (ctx.MaxFileSize && sourceFile.Size() > ctx.MaxFileSize) {
                NSecurityHelpers::LogDebug("Skip too big file", "size", ToString(sourceFile.Size()), "path", path);
                return Nothing();
            }

            auto result = CheckSource(sourceFile);
            if (!result.Defined()) {
                return Nothing();
            }

            return TPathResult{
                    .Path = path,
                    .SearcherResults = result.GetRef(),
            };
        } catch (const yexception &e) {
            NSecurityHelpers::LogErr("Failed to check file", "err", e.AsStrBuf(), "path", path);
            return Nothing();
        }
    }

    TMaybe<TSourceResult> TSearchWorkerBase::CheckSource(ISource &source) {
        try {
            TSourceResult results;
            doCheckFile(source, results);
            doCheckLine(source, results);
            if (!results.empty()) {
                return results;
            }
        } catch (const yexception &e) {
            NSecurityHelpers::LogErr("Failed to check content", "err", e.AsStrBuf());
        }
        return Nothing();
    }

    TVector<TStructResult> TSearchWorkerBase::CheckStruct(NJson::TJsonValue &root) {
        TVector<TStructResult> results;
        auto processor = [&results, this](const TString &path, const TString &value) {
            for (auto&&[searcherName, searcher]: lineSearches) {
                if (searcher->Binary()) {
                    continue;
                }

                try {
                    auto result = searcher->CheckLine(value, 1, path);
                    if (!result.empty()) {
                        results.push_back(TStructResult{
                                .Path = "$." + path,
                                .SearcherResults = std::move(result),
                        });
                    }
                } catch (const TSystemError &e) {
                    NSecurityHelpers::LogWarn("Failed to check json",
                                              "searcher", searcherName,
                                              "err", e.AsStrBuf(),
                                              "path", path
                    );
                }
            }
            return true;
        };

        try {
            auto walker = TJsonWalker(processor);
            root.Scan(walker);
        } catch (const TSystemError &e) {
            NSecurityHelpers::LogWarn("Failed to check json", "err", e.AsStrBuf());
        }

        return results;
    }

    void TSearchWorkerBase::doCheckLine(ISource &source, TSourceResult &results) const {
        auto searchers = FilterSearchers(lineSearches, source);
        if (searchers.empty()) {
            return;
        }

        size_t lineNo = 0;
        TStringBuf line;
        while (source.ReadLine(lineNo, line)) {
            if (line.size() < kMinLineLen) {
                continue;
            }

            //TODO(buglloc): fix this bullshit plz
            if (line.size() > kMaxLineLen) {
                line = line.Trunc(kMaxLineLen);
            }

            for (auto&&[searcherName, searcher]: searchers) {
                try {
                    const auto &result = searcher->CheckLine(line, lineNo, source.Path());
                    if (!result.empty()) {
                        results.insert(
                                results.end(),
                                result.begin(),
                                result.end());
                    }
                } catch (const TSystemError &e) {
                    NSecurityHelpers::LogWarn("Failed to check line",
                                              "searcher", searcherName,
                                              "err", e.AsStrBuf(),
                                              "path", source.Path().GetPath(),
                                              "line_no", ToString(lineNo));
                }
            }
        }
    }

    void TSearchWorkerBase::doCheckFile(ISource &source, TSourceResult &results) const {
        if (!source.IsFulfilled()) {
            // Don't check incomplete source
            return;
        }

        auto searchers = FilterSearchers(fileSearches, source);
        if (searchers.empty()) {
            return;
        }

        for (auto&&[searcherName, searcher]: searchers) {
            try {
                const auto &result = searcher->CheckFile(source);
                if (!result.empty()) {
                    results.insert(
                            results.end(),
                            result.begin(),
                            result.end());
                }
            } catch (const TSystemError &e) {
                NSecurityHelpers::LogErr("Failed to check file",
                                         "searcher", searcherName,
                                         "err", e.AsStrBuf(),
                                         "path", source.Path().GetPath());
            }
        }
    }

    void TSearchWorkerThread::DoExecute() {
        while (true) {
            auto maybePath = jobsQueue->Pop();
            if (maybePath.Empty())
                break;

            auto result = CheckPath(maybePath.GetRef());
            if (result.Defined()) {
                resultsQueue->Push(result.GetRef());
            }
        }
    }

}
