#include "keys.h"

#include <security/ant-secret/internal/string_utils/hash.h>
#include <security/ant-secret/secret-search/internal/mime/mime.h>

#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/stream/buffer.h>
#include <util/string/join.h>

namespace NSSInternal {
    namespace NSearchers {
        namespace {
            // To skip Bag Attributes
            constexpr size_t kHeaderLen = 512;
            constexpr TStringBuf kSignature = "PRIVATE KEY-----";
            const TVector<const TString> kBlacklistNames = {"snakeoil"};
            const TVector<const TString> kBlacklistPaths = {"/etc/ssh/ssh_host_"};

        }

        void TKeys::CheckFileTo(NSSInternal::ISource& source, NSSInternal::TSecretList& list) {
            if (source.Size() <= sizeof("-----BEGIN PRIVATE KEY-----\n-----END PRIVATE KEY-----")) {
                // source size less than needed
                return;
            }

            if (!source.Head(kHeaderLen).Contains(kSignature)) {
                // not a private key
                return;
            }

            const auto& content = source.Content();
            const auto& localPath = source.IsLocal() ? source.Path() : nullptr;
            const auto& tlsKey = NTlsKey::TTlsKey::FromPem(content, localPath);
            if (tlsKey && tlsKey->HaveCerts()) {
                // This is SSL certificate!
                if (!tlsKey->IsValid()) {
                    // Skip expired or self-signed certificates
                    return;
                }

                ProcessTlsKey(content, tlsKey.Get(), list);
                return;
            }

            const auto& sshKey = NSshKey::TSshKey::FromPem(content);
            if (sshKey) {
                // This is SSH private key
                ProcessSshKey(content, sshKey.Get(), list);
                return;
            }

            // Time to process "generic" private key
            if (ctx.ValidOnly || !(tlsKey || sshKey)) {
                return;
            }

            TString type;
            if (tlsKey && sshKey) {
                type = "SSL-SSHKey";
            } else if (tlsKey) {
                type = "SSLKey";
            } else {
                type = "SSHKey";
            }

            NSSInternal::TSecret secret = {
                .Type = type,
                .Validated = false,
                .Additional = {
                    {"sha1", NStringUtils::Sha1(content)},
                }};

            list.push_back(secret);
        }

        void TKeys::ProcessTlsKey(const TStringBuf content, const NTlsKey::TTlsKey* key, NSSInternal::TSecretList& list) {
            NSSInternal::TSecret secret = {
                .Type = "SSLKey",
                .Validated = false,
                .Additional = {
                    {"sha1", NStringUtils::Sha1(content)},
                    {"tls_subject", key->Subject()},
                    {"tls_serial", key->Serial()},
                    {"tls_chain", JoinSeq("->", key->Chain())},
                    {"tls_client_cert", (key->AllowClientAuth() ? "true" : "false")},
                    {"tls_server_cert", (key->AllowServerAuth() ? "true" : "false")}}};

            if (ctx.Validate && ctx.Validator && (key->AllowClientAuth() || key->AllowServerAuth())) {
                auto&& [valid, revoked] = ctx.Validator->CallSsl(
                    key->Serial(),
                    key->Chain(),
                    key->AllowClientAuth(),
                    key->AllowServerAuth());

                if (valid) {
                    secret.Validated = true;
                    secret.Additional["tls_revoked"] = revoked ? "true" : "false";
                }
            }

            if (secret.Validated || !ctx.ValidOnly) {
                list.push_back(secret);
            }
        }

        void TKeys::ProcessSshKey(const TStringBuf content, const NSshKey::TSshKey* key, NSSInternal::TSecretList& list) {
            const auto& legacyFp = key->FingerprintLegacy();
            NSSInternal::TSecret secret = {
                .Type = "SSHKey",
                .Validated = false,
                .Additional = {
                    {"sha1", NStringUtils::Sha1(content)},
                    {"ssh_fingerprint", legacyFp},
                    {"ssh_fingerprint_sha256", key->Fingerprint()},
                }};

            if (ctx.Validate && ctx.Validator) {
                const auto& info = ctx.Validator->Call("staff-ssh", legacyFp);
                if (info && info->Valid) {
                    secret.Type = info->Type;
                    secret.Owners = info->Owners;
                    secret.Validated = true;
                    secret.Additional["secret_users"] = info->Users;
                    secret.Additional["secret_validate_url"] = info->ValidationUrl;
                }
                if (!info) {
                    secret.ValidationError = true;
                }
            }

            if (secret.Validated || !ctx.ValidOnly) {
                list.push_back(secret);
            }
        }

        bool TKeys::IsAcceptablePath(const TFsPath& path) const {
            if (NMime::IsTextByExt(path.GetExtension())) {
                // Text files like *.py can't be ssh key files, probably...
                return false;
            }

            const auto& pathStr = path.GetPath();
            for (const auto& excl : kBlacklistPaths) {
                if (pathStr.StartsWith(excl)) {
                    return false;
                }
            }

            const auto& name = path.GetName();
            for (const auto& excl : kBlacklistNames) {
                if (excl == name) {
                    return false;
                }
            }
            return true;
        }

    }
}
