#include <library/cpp/testing/unittest/registar.h>

#include <security/ant-secret/secret-search/internal/searchers/ssh_key/ssh_key.h>

namespace {
    //TODO(buglloc): change to DATA macro

    const char* kOpenSSLRsaLegacyFinger = "0f:2e:b8:94:51:5e:61:76:c0:c7:fb:8f:5c:f6:12:58";
    const char* kOpenSSLRsaFinger = "SHA256:j9RuSxduIBo/vHwkcS9+krGF9K9g00Ci5ErIJ9XuuTE";
    const char* kOpenSSLRsaKey = R"(
-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAoVAkojU2G8/xoOo/CzGOPovOauTY8aVzR/oLQKXyR5iT6y11
UZfWwC0UfVLr4OmX/gMMjeMkoyV+4z8pxT5o80yw3IpTtByF70uUHlHscd9ZBS/d
wfWdSia2mo0k47wPLccgkxS7FLj+eJp7/SZS6WiaqQ5ALIXKv4YNKAoKMKcKdkpG
w8ZEq5IBOYzvrucJVAuSnkCJR2wamlA0pTomvoaOWyo1vSyrsHs1MpEWwH+hcLk4
M880PwrkfXZMAntJmxTtIPWJISsZqMQWa+iwZn5dVmVtGPZGimG8BpkX/5NE+MVJ
bOjWXreNEXWb+rkEkV57FzZCTzEji7C0tMi5qwIDAQABAoIBAD3mPuhver/ZX2Lc
AC0PawHdK+wgvI6daksoS8EW7I1Eox4PZ4FTvngn1EDy3MiFWmmCSgL4MkrYTIRu
7Fxfwm3587Ps8ImgmFRmsAOidBSOG969q16OHTki+db8ajdpJnnuezV1MwSUqDEn
NoTBe/dMyGwup7lt7YroqYu/roFUO94dggG031nztllV6mhuQRDH4e+qy2X6v3q5
oBhee4KhprkL2DW+0kuEJ6QZmiXJ4PuE3xDglE/01wCkr0OBAlQavm2oOrioFHo0
HYLAnFapXO5M9HuLc2Lwkm3YwdTkq4iEcR6979ea8bpA72izqrGvPQ75HtSOoMRn
IZ9+fKECgYEAy6CWji6JITI+W40hDXsfFXEXMOXErnYqPTL+YIpdTnvf6G6krRn5
L1tVW1zzTRqQ8XN8hCS8WKgeHhLA2+rj74htPU0jwjQF1Q1srslupPtY9NO8ZW9B
pH6eM+BIeW9wk+XdyEYFU0diQbiIL+pYnurQgSPRXEPAloW+3+smBdUCgYEAys12
en6cSbiSouTa7UECCxsthgF8RX5ADvGMmRU3m8gIzsc81m9qnmP8PqbmKcBG+ogH
dRbvn6/AcVcBA17jJuC5Ad88AkkZ5VeJ0VpCrRSFBT20MwYEIJN6/1mhCuY7CYtF
D1nMCBqvOpmFcyHlJaBtVSOAcm0w5MhMFaBOAX8CgYEArSs2nlfo+rg127nboBFS
z/NLG6RrbUrEhSM2vXct/q7OLmbZ2PQ7vBBaOMaFis7K8Oubuu1a17KxWjXsmZna
mjxVWkqd3GxNgXBXOxXG6LtwXVUr5L+G/ZNIiylzuQISpVRZFhamNLqCTw3Vk5oi
H5PmGc0SIPrlb5zyRI8/2LkCgYEAwtM1YaImJFBTVnfCtbAA9YhraFQe/We2xGRa
rGmMnXRCudr/YAVzCKyHE+0kKbHUvjNV7q9VDjQ6DOgHSiSq1ZuHP/wj2xI89Wzu
Mhei/EE9NPa9GVNLPbxQqeSB6EWbK6UGeCPM9XAlLId1g1BpoYcH4YWUjNjG4tJk
/VWd+qMCgYB/qc4pcdnE2fzWEktTsXTqjaz1vpGO0hH7cZHMFQpffx5ggZ9QB29o
M8A6KBDlcxOnRMSVdqKPW0xoZdAa7gXHPF2msYRYrV0o/a/aAeRRkhbLGyC9POxk
g+TFM7MYurvEYvOL9fyaPdYZXrkBJWP9HDIpbuEGWVChma3D2ouy8g==
-----END RSA PRIVATE KEY-----
)";

    const char* kOpenSSLDsaLegacyFinger = "08:ad:c2:49:3a:6a:10:46:fb:9a:58:d4:bb:ab:31:57";
    const char* kOpenSSLDsaFinger = "SHA256:FSazkicDu4hPpjBZjGN0eJKn09GczBx+YqaGoSWh5To";
    const char* kOpenSSLDsaKey = R"(
-----BEGIN DSA PRIVATE KEY-----
MIIBugIBAAKBgQCk5eBIT5qdH03UfEmtMsYvbU1tUTtaOKoCoOp2IhOmJBNvMPnj
RqzJBp/fylCvhBYVt4zYERl94mE2k7lS3MdBEOoEegK+IhzLjgpfXApDQ3euLLUX
BIzs1p6awHQlDNZMzVac2l11LPF0LQDaSCVZXjlVnfD5VvC3Bw62gHDPFwIVAJc+
wW9j8Qwrx07koMzVbEQFIh3pAoGAJaRSKd8nB1P0CLIJYyHUwyO00hJhue0iDIzS
MgXWkWEFBADv8w7YHae9Ze3IZSZsOhzIyROGeX8qX2fCZpyovLMNBOhic/xRMmkb
dtKB78+em7D/qq3tDDYdavY1YsBkk1pGLw5Ornf+uwDZUiL6L18KzZ9c6G9uyVGv
uL6qk8wCgYBXafvLbik8caAM+DULocYGlb66GlIQbFJpHIf1Zi8RK3QoNy1j5ulu
vJtZfpQPGjQBz27XaRjplf3iYJtS/dlgdQ6TMQOjNYOV5yThrZfKoAWSYehRgfXv
O2AtGKfuO/KBe6ZSfM7YUccfT6mpzC04WfN7tcjvyg9iKUSG19pQhAIUUlx/9CqI
jbK3DgknEUB1og5YLZc=
-----END DSA PRIVATE KEY-----
)";

    const char* kOpenSSLEcdsa256LegacyFinger = "e3:f0:57:78:f6:95:c8:f1:6e:76:26:78:5f:d6:19:26";
    const char* kOpenSSLEcdsa256Finger = "SHA256:J3oRg+8AvqFkYDn09791OGndMlJqf32snT3HG8Jp5D4";
    const char* kOpenSSLEcdsa256Key = R"(
-----BEGIN EC PRIVATE KEY-----
MHcCAQEEIJ2sT0PAZcUDDjCMcBHu36lRQs717ZipeX19HBDzwWOmoAoGCCqGSM49
AwEHoUQDQgAEesuJd/igMuU8z0Q0qh6s67G4/yQPijy+eTAkK0upU+4rjKWje+gf
SCdpRfl1hXleakS968RtdLlLgAjKkxJxWw==
-----END EC PRIVATE KEY-----
)";

    const char* kOpenSSLEcdsa384LegacyFinger = "f7:c7:2b:d6:dd:13:06:b1:96:1d:32:03:e7:38:6b:1c";
    const char* kOpenSSLEcdsa384Finger = "SHA256:ONoEGuI2nV6vGqA5A1bBW4trGkyDkD4Qea8/8SgdbPc";
    const char* kOpenSSLEcdsa384Key = R"(
-----BEGIN EC PRIVATE KEY-----
MIGkAgEBBDBmLRgcoe/NlUjv7ScWkCMB/8BQ3shBlJ822Tiojo4Fzs64Q2Gh9iJJ
EC8XcG3qBrygBwYFK4EEACKhZANiAAQBvlAnCzTmsRcGhwRhNK55TbD4u+BCfokW
oejMktkcJtE3IRFKHtD8b/jTCLbe3y6csdAPRLsbbx23uFBohPDJ/xnzkHJdCVWK
ahYO3LwzkgNFVDIxywz3u2RHE1hOpz4=
-----END EC PRIVATE KEY-----
)";

    const char* kOpenSSLEcdsa521LegacyFinger = "19:4a:fa:b2:40:0b:4e:cf:20:24:5d:f2:de:91:2d:ba";
    const char* kOpenSSLEcdsa521Finger = "SHA256:942rLscu1bJmISW5RWgmd4Mv7YRWMyXruI2WUjsfBJI";
    const char* kOpenSSLEcdsa521Key = R"(
-----BEGIN EC PRIVATE KEY-----
MIHcAgEBBEIBVS5TcRL74h3r0j2gAghkGtlXtFdgMz8J4fWIwd2m9l9apv9y1h3U
I3i/Cr+hMYnYL/WSiGNL+mf4YJjktbtnHFegBwYFK4EEACOhgYkDgYYABACilZMn
us0fF4kFKzU42sCwIOEK3BxVdI7rDCd2+6VvDI/69H51uJo2yvfy2HMMFxEqtyJS
nwvScb++5li+bRh5TQD3lq+0XjJfCqlyhhqRoDfxkm8+WCTURX5M2qA7LxnIJDfL
TU/grUyOejj3gZ72ZTZ0TEZklO3PvIccZ3MqfmFhKQ==
-----END EC PRIVATE KEY-----
)";

    const char* kOpenSSLEcdsaEncKey = R"(
-----BEGIN EC PRIVATE KEY-----
Proc-Type: 4,ENCRYPTED
DEK-Info: AES-128-CBC,AF843D8D6C79374C9F9FCD98B97DA3E3

gxs0qUOBP1Z7owNs84f6ac/ZmcKH9IyWa/i9ALfZbdhv/nubcv3mNYzo8WKzBFj1
TbNgiTVwwJDjt4x9aj2+5dinvKTF8QCQvJ365dfvA3qzc1XmLw+PKL2uzXzw22n7
i+1Ax4oPonatPP5Rgb94XTgL5GwZOpWRt90vDF8wyrs=
-----END EC PRIVATE KEY-----
)";

    const char* kOpenSSLUnknownKey = R"(
-----BEGIN FOO PRIVATE KEY-----
MHcCAQEEIJ2sT0PAZcUDDjCMcBHu36lRQs717ZipeX19HBDzwWOmoAoGCCqGSM49
AwEHoUQDQgAEesuJd/igMuU8z0Q0qh6s67G4/yQPijy+eTAkK0upU+4rjKWje+gf
SCdpRfl1hXleakS968RtdLlLgAjKkxJxWw==
-----END FOO PRIVATE KEY-----
)";

    const char* kOpenSSHEd25519KeyLegacyFinger = "2c:f8:5c:eb:6b:c0:29:af:91:fa:b4:34:1e:85:29:3e";
    const char* kOpenSSHEd25519KeyFinger = "SHA256:wcjZvU/56fl6WE6XKzMBfZOByNFZ8w7/0tNbxJuUeaA";
    const char* kOpenSSHEd25519Key = R"(
-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAAAMwAAAAtzc2gtZW
QyNTUxOQAAACB9ODjX+kG+qcthiVqYX3rDNV29ACygxkc95JMo8HN48QAAAJC7hPG9u4Tx
vQAAAAtzc2gtZWQyNTUxOQAAACB9ODjX+kG+qcthiVqYX3rDNV29ACygxkc95JMo8HN48Q
AAAEA0p4ON5rM+7W9d7YrABw2WbyKfsqV+y4v9ycPkxtKdlX04ONf6Qb6py2GJWphfesM1
Xb0ALKDGRz3kkyjwc3jxAAAADGJ1Z2xsb2NAYnBhZAE=
-----END OPENSSH PRIVATE KEY-----
)";

    const char* kOpenSSHEd25519EncKey = R"(
-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAACmFlczI1Ni1jdHIAAAAGYmNyeXB0AAAAGAAAABBzSnkHb3
wq0ejCcs0z7xGnAAAAEAAAAAEAAAAzAAAAC3NzaC1lZDI1NTE5AAAAIEmGmHmFLXcSt3f8
UDtnzKHPF4A/FpJIf5fbnauQC+n1AAAAkFysBBBKzR/qO2dnC1l+COZejFQ4pzSxXD+uLO
gCZ9K7eDiH0P0IZBsUBuGGbcvr32Nit3nms8UldpikUbxkUuj/aWBRQl0tNZ9Gv4jjixAi
5oNSRDmBg0ZN6kKbU4IW+PI24RJo3sU2h3J7De5hx8bv7UseJeXkV0suzoLRv0nj7hiHV0
6VxyKKtTlNW9IahQ==
-----END OPENSSH PRIVATE KEY-----
)";

    const char* kOpenSSHUnknownKey = R"(
-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAAAMwAAAAtzc2gtZWQyMjIy
OQAAACDROPhKHcKaIuMor37Uprh9NyBMa++Ck+WFdDto4tdruQAAAJBQTPzTUEz80wAAAAtzc2gt
ZWQyNTUxOQAAACDROPhKHcKaIuMor37Uprh9NyBMa++Ck+WFdDto4tdruQAAAECqfAYHeBAMYeBx
oqeRe0ql49YQs3lcs/FqeH1NvFZCc9E4+Eodwpoi4yivftSmuH03IExr74KT5YV0O2ji12u5AAAA
DGJ1Z2xsb2NAYnBhZAE=
-----END OPENSSH PRIVATE KEY-----
)";

    const char* kOpenSSHEcdsa521KeyLegacyFinger = "2b:f4:02:b9:8c:55:77:e9:62:c5:7d:9e:b6:b3:74:63";
    const char* kOpenSSHEcdsa521KeyFinger = "SHA256:hXYkqv8zNl38wJVfc6rKo4MT5fF/1AQGhY3p4yQGStU";
    const char* kOpenSSHEcdsa521Key = R"(
-----BEGIN OPENSSH PRIVATE KEY-----
b3BlbnNzaC1rZXktdjEAAAAABG5vbmUAAAAEbm9uZQAAAAAAAAABAAAArAAAABNlY2RzYS
1zaGEyLW5pc3RwNTIxAAAACG5pc3RwNTIxAAAAhQQB5b7DENoDwI9kTsCOjLrKdPR/62eQ
adgqdMwoM1F6BVYhPVVj/CuW8FY30OPl8Lxq2mAWSlaMThVrt8z8UGSWulQAh+U5+6HOLQ
96dcLPEbLmw39NvjZ+7nAsptGYuQhK/Ize2oC1G36Jx3haBvfePfFGCaq0OUPTSrOFnXik
HGqyLMsAAAEQt/p/prf6f6YAAAATZWNkc2Etc2hhMi1uaXN0cDUyMQAAAAhuaXN0cDUyMQ
AAAIUEAeW+wxDaA8CPZE7Ajoy6ynT0f+tnkGnYKnTMKDNRegVWIT1VY/wrlvBWN9Dj5fC8
atpgFkpWjE4Va7fM/FBklrpUAIflOfuhzi0PenXCzxGy5sN/Tb42fu5wLKbRmLkISvyM3t
qAtRt+icd4Wgb33j3xRgmqtDlD00qzhZ14pBxqsizLAAAAQgHysTLBEIP8D0NXhqG+jthn
6+a1Cb6TPZzGwdyKJe+P5lSFagfJhD8Clg4ZB0A70EoYemwKYAEJxBy8xzhLxEIsPwAAAA
xidWdsbG9jQGJwYWQBAgMEBQY=
-----END OPENSSH PRIVATE KEY-----
)";

    void parseValidKey(const TStringBuf rawKey, const TStringBuf finger, const TStringBuf legacyFinger) {
        using namespace NSshKey;

        const auto& key = TSshKey::FromPem(rawKey);
        UNIT_ASSERT(key);

        UNIT_ASSERT_STRINGS_EQUAL(key->FingerprintLegacy(), legacyFinger);
        UNIT_ASSERT_STRINGS_EQUAL(key->Fingerprint(), finger);
    }

    void parseInvalidKey(const TStringBuf rawKey) {
        using namespace NSshKey;

        const auto& key = TSshKey::FromPem(rawKey);
        UNIT_ASSERT(!key);
    }

}

Y_UNIT_TEST_SUITE(TSshKeyParser) {
    Y_UNIT_TEST(OpenSSL_RSA) {
        parseValidKey(
            kOpenSSLRsaKey,
            kOpenSSLRsaFinger,
            kOpenSSLRsaLegacyFinger);
    }

    Y_UNIT_TEST(OpenSSL_DSA) {
        parseValidKey(
            kOpenSSLDsaKey,
            kOpenSSLDsaFinger,
            kOpenSSLDsaLegacyFinger);
    }

    Y_UNIT_TEST(OpenSSL_EC) {
        parseValidKey(
            kOpenSSLEcdsa256Key,
            kOpenSSLEcdsa256Finger,
            kOpenSSLEcdsa256LegacyFinger);

        parseValidKey(
            kOpenSSLEcdsa384Key,
            kOpenSSLEcdsa384Finger,
            kOpenSSLEcdsa384LegacyFinger);

        parseValidKey(
            kOpenSSLEcdsa521Key,
            kOpenSSLEcdsa521Finger,
            kOpenSSLEcdsa521LegacyFinger);
    }

    Y_UNIT_TEST(OpenSSL_ENCRYPTED) {
        parseInvalidKey(kOpenSSLEcdsaEncKey);
    }

    Y_UNIT_TEST(OpenSSL_UNKNOWN) {
        parseInvalidKey(kOpenSSLUnknownKey);
    }

    Y_UNIT_TEST(OpenSSH_ED25519) {
        parseValidKey(
            kOpenSSHEd25519Key,
            kOpenSSHEd25519KeyFinger,
            kOpenSSHEd25519KeyLegacyFinger);
    }

    Y_UNIT_TEST(OpenSSH_ENCRYPTED) {
        parseInvalidKey(kOpenSSHEd25519EncKey);
    }

    Y_UNIT_TEST(OpenSSH_UNKNOWN) {
        parseInvalidKey(kOpenSSHUnknownKey);
    }

    Y_UNIT_TEST(OpenSSH_ECDSA) {
        parseValidKey(
            kOpenSSHEcdsa521Key,
            kOpenSSHEcdsa521KeyFinger,
            kOpenSSHEcdsa521KeyLegacyFinger);
    }
}
