#include <library/cpp/testing/unittest/registar.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>
#include <util/stream/file.h>

#include <security/ant-secret/secret-search/internal/searchers/tls_key/tls_key.h>

namespace {
    //TODO(buglloc): change to DATA macro

    const TStringBuf kTeamcitySubject =
        "/C=ru/ST=Moscow/L=Moscow/O=Yandex/OU=lpc/CN=teamcity@ld.yandex.ru/emailAddress=teamcity@yandex-team.ru";
    const TStringBuf kTeamcityPrivate = TStringBuf(R"(
-----BEGIN PRIVATE KEY-----
MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQCbWjp8Ac3UboLC
zTwrM+eR7RM15ygYobhytOPnh0vi4+WmbzViTUjEi/uxZL320fOBJOHqdmGF0ifF
cHDZWbNx0OiRWW6HrW0Y6HBOXtmmtfDxoM8/vEIDyfTRZNBRE7+HYvOYoa1IJ1DY
i/kTAtePTvB9QV3wWSLn3mgFLnSbjcRnDokEBCnb3r32QnGiHY3nL4o3UZFlO5u2
Z3wK+OoaDoHy3d26nqHruteSc6f7BNyXLNCR77wYJtiMQqlYCXqc7D9v91mWm3tr
gg/Qyn13aAL7MJCJjSL9IlSguu7JLeRkLFxjUwcBYT5Hmm90fi8qD+SYhRZ6a6se
ma4LVFC3AgMBAAECggEAB0oFgnoHMVd6FS/SCejmjadFBntxGy7DW8Z3Bw8HFX9B
8Dw5kgI0TtyOk7U5YJJbTYzweLLTQpcovI/ZC6m0zLtTnjZHBGCzxL5rPpIevj69
LmkwMDpiEpFwRj7ns1Ys32paKag0XDVUCsUOi8DPYm7+w7HEeZmFFMIPuWaDnCTJ
AJxLNeTK6F3cgY7vPd/SEQ5yFBY8FezyXgJ8COCFD8+z62iVtGIB+d6RH8SzsFGT
W1+DTMglTEafI/EE7HN43C1VzHrwdqKnx3eJ7HjICOmksxWwGn/027ySd3YWBi2o
8OQaHb045kSCw4rhD6/zL6CI9gwbaM35GzRmEGvOGQKBgQDNEsJsoz32GlnMPjwv
EBnVuBeTEOwRzQ9wpE5ahk5ECR8Sx0JOuEBTVXVu3o3qefBe78O7VT2L+B8Vyw2v
YpwvF+J3FVb34rUrMXcEm96fWnV3OxfpsTXoE20m9znv96/RkHhIIRQQSZwVZXGJ
ME4/RY/ytZDGSItxLMqKYBaJFQKBgQDB7onmFQO98Oiw1KBgTqiQO7SNIDWyiATZ
UiGecw3j5RTA6iErMViy/8oc/6vXj3XsN4fUhpY1awwOIllJF5xOdnPuVRIGfnEk
rPzbwI8/yw3ihYmxPGeZ2IWf4ZHa6ilQI0nupF82XWhUwLjgDvz/VFSa1vXbZkr2
fuEZcaRNmwKBgBZV7rsqZHtDj3hBZwixNj8i1956v1uI/B+GpcoSSr/6RlojPcOZ
KdhpvGqg7tAVC+srLiRfOl7eXXRgM1Ng6CXw8geKWXiSRq03rADigTLWBWlBFG+t
j3OBeif+UV8jmRC4LYGdqLPFcKyb6evJSRx3FSCYSfZuvFFNUZoUsR+pAoGARnKf
t2wE2vfrn2juBW4tf4+gy+9acktE76kNO2DyN8oc2uNyPx2C6itPQW+C/iBIJxIZ
DpQBdtA3Jd9Gbe3cwwf2nypaDO/brAYHKNDveK4LkjEOTsrxm4pzKBwp/ss3TLw0
/DLLv5T49Gi5//KR5zK8NS9HSncDXj3DCZolpGcCgYBiu1nnhJnrZU3KmmS3+u6F
qwvaEbK/cKwH2hUThMoqR9jTDtNCwDkSjDZTCHSnCxni/mBMx6kkePkHUWPxZKmU
wHRc5rOB46DThQnhTNj8hu+jpBXOv9b/imxuHVeZwaovn9UFNpV8YY9JDCczYMkW
5RlfLa7JilQwomvJ9S8uow==
-----END PRIVATE KEY-----
)");

    const TStringBuf kTeamcityCert = TStringBuf(R"(
-----BEGIN CERTIFICATE-----
MIIEiDCCA3CgAwIBAgIKEQw+rAACAAN36DANBgkqhkiG9w0BAQsFADBbMRIwEAYK
CZImiZPyLGQBGRYCcnUxFjAUBgoJkiaJk/IsZAEZFgZ5YW5kZXgxEjAQBgoJkiaJ
k/IsZAEZFgJsZDEZMBcGA1UEAxMQWWFuZGV4SW50ZXJuYWxDQTAeFw0xODA3MjQx
NTEyNDJaFw0xOTA3MjQxNTEyNDJaMIGWMQswCQYDVQQGEwJydTEPMA0GA1UECBMG
TW9zY293MQ8wDQYDVQQHEwZNb3Njb3cxDzANBgNVBAoTBllhbmRleDEMMAoGA1UE
CxMDbHBjMR4wHAYDVQQDDBV0ZWFtY2l0eUBsZC55YW5kZXgucnUxJjAkBgkqhkiG
9w0BCQEWF3RlYW1jaXR5QHlhbmRleC10ZWFtLnJ1MIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEAm1o6fAHN1G6Cws08KzPnke0TNecoGKG4crTj54dL4uPl
pm81Yk1IxIv7sWS99tHzgSTh6nZhhdInxXBw2VmzcdDokVluh61tGOhwTl7ZprXw
8aDPP7xCA8n00WTQURO/h2LzmKGtSCdQ2Iv5EwLXj07wfUFd8Fki595oBS50m43E
Zw6JBAQp29699kJxoh2N5y+KN1GRZTubtmd8CvjqGg6B8t3dup6h67rXknOn+wTc
lyzQke+8GCbYjEKpWAl6nOw/b/dZlpt7a4IP0Mp9d2gC+zCQiY0i/SJUoLruyS3k
ZCxcY1MHAWE+R5pvdH4vKg/kmIUWemurHpmuC1RQtwIDAQABo4IBEDCCAQwwCwYD
VR0PBAQDAgWgMB0GA1UdDgQWBBSFmgBMrs6glYMZopWR4l8j1khn9TAfBgNVHSME
GDAWgBSP3TKDCRNT3ZEaZumz1DzFtPJnSDBMBgNVHR8ERTBDMEGgP6A9hjtodHRw
Oi8vY3Jscy55YW5kZXgucnUvWWFuZGV4SW50ZXJuYWxDQS9ZYW5kZXhJbnRlcm5h
bENBLmNybDA9BgkrBgEEAYI3FQcEMDAuBiYrBgEEAYI3FQiG3KEmhZPVVYf9gS2D
+ecCg5XSBByCtbgVhOCSJgIBZAIBEzATBgNVHSUEDDAKBggrBgEFBQcDAjAbBgkr
BgEEAYI3FQoEDjAMMAoGCCsGAQUFBwMCMA0GCSqGSIb3DQEBCwUAA4IBAQCMkmxd
u+lMug7afgkr3UMx5vKgsoggO2fYLSI3U5qCT+viW6cWcDjpDQ5k5OBAPPSJBlQP
8Kq1zlbs4jiDYPO3lCxKGRLQWRHNYQD6Dr6XvLYuJyf/dU/7UYdP4P0m1cITyRd6
3S/F5HkcuCM8+BjZimD/YFLfZ/bxTjylzH1IfeLHjbuTUt/u9PYmfk5Gm+8PN5Hc
jySMltYVKS0mCGB7ZA2uh19HGfEEelhBw4Ix/xdOsTZCspI2yVqInP18rfCzQruu
27zpw6Jy3b0MlUdiN1vCdleR2iAymWupRyhKew+2WbXc/ofOnBdkls12P1KbjYEQ
ZeWx61u0dMZH0wNC
-----END CERTIFICATE-----
)");

    const TStringBuf kTeamcityChain = TStringBuf(R"(
-----BEGIN CERTIFICATE-----
MIIFZTCCA02gAwIBAgIKUlD06gAAAAAAGDANBgkqhkiG9w0BAQ0FADAfMR0wGwYD
VQQDExRZYW5kZXhJbnRlcm5hbFJvb3RDQTAeFw0xODA2MjgxMTE0NTdaFw0zMjA2
MjgxMTI0NTdaMFsxEjAQBgoJkiaJk/IsZAEZFgJydTEWMBQGCgmSJomT8ixkARkW
BnlhbmRleDESMBAGCgmSJomT8ixkARkWAmxkMRkwFwYDVQQDExBZYW5kZXhJbnRl
cm5hbENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAy6Sab1PCbISk
GSAUpr6JJKLXlf4O+cBhjALfQn2QpPL/cDjZ2+MPXuAUgE8KT+/mbAGA2rJID0KY
RjDSkByxnhoX8jwWsmPYXoAmOMPkgKRG9/ZefnMrK4oVhGgLmxnpbEkNbGh88cJ1
OVzgD5LVHSpDqm7iEuoUPOJCWXQ51+rZ0Lw9zBEU8v3yXXI345iWpLj92pOQDH0G
Tqr7BnQywxcgb5BYdywayacIT7UTJZk7832m5k7Oa3qMIKKXHsx26rNVUVBfpzph
OFvqkLetOKHk7827NDKr3I3OFXzQk4gy6tagv8PZNp+XGOBWfYkbLfI4xbTnjHIW
n5q1gfKPOQIDAQABo4IBZTCCAWEwEAYJKwYBBAGCNxUBBAMCAQIwIwYJKwYBBAGC
NxUCBBYEFNgaef9LcdQKs6qfsfiuWF5p/yqRMB0GA1UdDgQWBBSP3TKDCRNT3ZEa
Zumz1DzFtPJnSDBZBgNVHSAEUjBQME4GBFUdIAAwRjBEBggrBgEFBQcCARY4aHR0
cDovL2NybHMueWFuZGV4LnJ1L2Nwcy9ZYW5kZXhJbnRlcm5hbENBL3BvbGljaWVz
Lmh0bWwwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8G
A1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAUq7nF/6Hv5lMdMzkihNF21DdOLWow
VAYDVR0fBE0wSzBJoEegRYZDaHR0cDovL2NybHMueWFuZGV4LnJ1L1lhbmRleElu
dGVybmFsUm9vdENBL1lhbmRleEludGVybmFsUm9vdENBLmNybDANBgkqhkiG9w0B
AQ0FAAOCAgEAQnOiyykjwtSuCBV6rSiM8Q1rQIcfyqn1JBxSGeBMABc64loWSPaQ
DtYPIW5rwNX7TQ94bjyYgCxhwHqUED/fcBOmXCQ2iBsdy5LOcNEZaC2kBHQuZ7dL
0fSvpE98a41y9yY6CJGFXg8E/4GrQwgQEqT5Qbe9GHPadpRu+ptVvI6uLZG3ks2o
oodjOm5C0SIo1pY4OtPAYE/AzTaYkTFbAqYcPfEfXHEOigBJBeXnQs7cANxX/RaF
PnHEjZbGY57EtBP6p5ckndkfEmqp3PLXbsQteNOVpsUw5eVqEzinSisBmLc28nnr
5QEojRontAaZd7ZzB5zaGkVuE+0laUUWSNBhfGE1R3LrTJEK9L7FEsBBprOxIWww
CvLmAfglouwuNRc2TjRdfnZaEfPLD7NYIF4ahXPAMcfTii23Tlr2uB7LetNykSlX
Z9S5/yf61VFEKnxuipFPNgtKqPcFgFUxlEb+wOeOfYZ7ex8VlpMBWbadj3Go025b
KZUwKwHDQvgJ5pz9g3t+t5Xieu2pwyddWGu+1SItRohRhlyTiep7oW6yTps7Qt0e
8pdLuLG7ZF19h1Pxi+dVbeaeNcsGEAOdRuCk+RTZHNe+J4yC8tNJOepnfYDul6SB
RjFWthiFK45+TZRHAcsG9JuV8JNvgoKaL75v/GUsKaeJ3Cps3rBStfc=
-----END CERTIFICATE-----
)");

    const TStringBuf kSelfSignedCert = TStringBuf(R"(
-----BEGIN PRIVATE KEY-----
MIIBVQIBADANBgkqhkiG9w0BAQEFAASCAT8wggE7AgEAAkEAzXtq/qZpErZG9e5/
v/SbGVmWK921p2ctuJLW0g5RngXpqdjsFfBs2QLjs37ijBXYxVogS3wSMOPdGRai
N0O76wIDAQABAkBU6jrWD1j+pZr/iNb8MY5i1HXNnOqzylzHmM7zsX0jCtrmrNaa
5YmRv8iMTvnwyBowA+VOZNk9CIHEO4Qt+MXpAiEA6MIdteysMWWoHI7xsDVwcMTh
ROzG+T46HLiCY5Avay8CIQDiAA4KXuEcFbJXpCA+4lVpIpDytvaWWrVVJFh2tdqc
BQIhANh9/R6ChWLot+XgOwL3oZ7yMMloJmYbr/eY0oaCzOg3AiBiwoFgOMwIeNbB
g8BvmnqWgftzhU7Gr4b0WPirj5ajfQIhAOawgI09PBUkM0d7YIrd01HROpO0QmVA
phLxjGahI0xN
-----END PRIVATE KEY-----

-----BEGIN CERTIFICATE-----
MIICBjCCAbCgAwIBAgIJAJvBsd9N0r4gMA0GCSqGSIb3DQEBCwUAMF0xCzAJBgNV
BAYTAlJVMQ0wCwYDVQQIDARUZXN0MQ0wCwYDVQQHDARUZXN0MSEwHwYDVQQKDBhJ
bnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxDTALBgNVBAMMBHRlc3QwHhcNMTgwODI4
MDk0MzUxWhcNMTkwODI4MDk0MzUxWjBdMQswCQYDVQQGEwJSVTENMAsGA1UECAwE
VGVzdDENMAsGA1UEBwwEVGVzdDEhMB8GA1UECgwYSW50ZXJuZXQgV2lkZ2l0cyBQ
dHkgTHRkMQ0wCwYDVQQDDAR0ZXN0MFwwDQYJKoZIhvcNAQEBBQADSwAwSAJBAM17
av6maRK2RvXuf7/0mxlZlivdtadnLbiS1tIOUZ4F6anY7BXwbNkC47N+4owV2MVa
IEt8EjDj3RkWojdDu+sCAwEAAaNTMFEwHQYDVR0OBBYEFHXiLcQbcCnbUrfJG69k
X/PaK/HLMB8GA1UdIwQYMBaAFHXiLcQbcCnbUrfJG69kX/PaK/HLMA8GA1UdEwEB
/wQFMAMBAf8wDQYJKoZIhvcNAQELBQADQQCT11+hmYiqXNtzC3bYi0gNZrRxBzjf
2Fyd90sRIC5/y1B9FUPTGjYt9gtM/vSxflC5tRWrj0qUsSamk1LVGuQh
-----END CERTIFICATE-----
)");

    const TStringBuf kKeyOnly = TStringBuf(R"(
-----BEGIN PRIVATE KEY-----
MIIBVQIBADANBgkqhkiG9w0BAQEFAASCAT8wggE7AgEAAkEAzXtq/qZpErZG9e5/
v/SbGVmWK921p2ctuJLW0g5RngXpqdjsFfBs2QLjs37ijBXYxVogS3wSMOPdGRai
N0O76wIDAQABAkBU6jrWD1j+pZr/iNb8MY5i1HXNnOqzylzHmM7zsX0jCtrmrNaa
5YmRv8iMTvnwyBowA+VOZNk9CIHEO4Qt+MXpAiEA6MIdteysMWWoHI7xsDVwcMTh
ROzG+T46HLiCY5Avay8CIQDiAA4KXuEcFbJXpCA+4lVpIpDytvaWWrVVJFh2tdqc
BQIhANh9/R6ChWLot+XgOwL3oZ7yMMloJmYbr/eY0oaCzOg3AiBiwoFgOMwIeNbB
g8BvmnqWgftzhU7Gr4b0WPirj5ajfQIhAOawgI09PBUkM0d7YIrd01HROpO0QmVA
phLxjGahI0xN
-----END PRIVATE KEY-----
)");

    const TStringBuf kEncrypted = TStringBuf(R"(
-----BEGIN ENCRYPTED PRIVATE KEY-----
MIIBtDBOBgkqhkiG9w0BBQ0wQTApBgkqhkiG9w0BBQwwHAQI6OUpZCrE0U8CAggA
MAwGCCqGSIb3DQIJBQAwFAYIKoZIhvcNAwcECNK0yzWEcI1CBIIBYPkv9RSxJ1vs
PCnyQJVQdKDXyTFx/ODWBpsoGuhWar6s7L3PlyFsE+a1I8g2jXC3NDlS7pSzZ+Fk
fjecSZKQhux+vDsskHNNvMDPuFffrnASj49xz7hahK9/6ilULq2jGNxQfVZMDXt7
hZb6fIPJtNEE3ZIaIDZlnstwY4weW1tO6a+uxkZyQcQKKq/bPKlbamHLfxvhmuQ1
xBWiF0upTBSkOOgqmpjROtsX5Ct47Bstzj9MxaIAzD0eS79VxStG8Br1XaiubERd
fUgScC9XcXd8QW9/QcumjRLBSLiHgEJS5JY1CLmGEnExgvPjLjqxvEExv1KX75nF
lznpp+ar0oOu4Pbhbw6uaGvSUFEeo3EWs1yZebt+m7CA6C1iqnimPJWbGUwvO9bm
v6pm4iikgZ5f6ghBR7/G6c8G3gkn8k1qrBp+TrY3hS/8QDgPBnzow/PzoLhrppTb
vCcwU4mOTSA=
-----END ENCRYPTED PRIVATE KEY-----

-----BEGIN CERTIFICATE-----
MIIB6DCCAZKgAwIBAgIJANza2V/K3LHtMA0GCSqGSIb3DQEBCwUAME4xCzAJBgNV
BAYTAkFVMQ0wCwYDVQQIDARUZXN0MSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRnaXRz
IFB0eSBMdGQxDTALBgNVBAMMBFRlc3QwHhcNMTgwODMxMjA1NDQxWhcNMTkwODMx
MjA1NDQxWjBOMQswCQYDVQQGEwJBVTENMAsGA1UECAwEVGVzdDEhMB8GA1UECgwY
SW50ZXJuZXQgV2lkZ2l0cyBQdHkgTHRkMQ0wCwYDVQQDDARUZXN0MFwwDQYJKoZI
hvcNAQEBBQADSwAwSAJBALTtq71cLeg2nEkC8S6zohbX6laCZ32zs50YFB5TkSo/
ZKnETR6Y3FS1mS+Txw+9Ho6jkuMTPU3Ws8BaI239dYcCAwEAAaNTMFEwHQYDVR0O
BBYEFJsZhe0EiLKFafTudpCOLPzhMn55MB8GA1UdIwQYMBaAFJsZhe0EiLKFafTu
dpCOLPzhMn55MA8GA1UdEwEB/wQFMAMBAf8wDQYJKoZIhvcNAQELBQADQQCQIyfu
7x1TMDN6NQ/bXnTpDu/uCOZQpBdJyskSrBL6rw9oMZi1MNwf88bjlnDJ73omylM+
EwJdWTeQp4vMGUbd
-----END CERTIFICATE-----
)");

}

using namespace NTlsKey;

Y_UNIT_TEST_SUITE(TTlsKey) {
    Y_UNIT_TEST(ValidTeamcity) {
        const auto& key = TTlsKey::FromPem(TString::Join(kTeamcityPrivate, kTeamcityCert, kTeamcityChain));
        UNIT_ASSERT(key);

        UNIT_ASSERT(key->IsValid());
        UNIT_ASSERT(key->HaveCerts());
        UNIT_ASSERT(key->AllowClientAuth());
        UNIT_ASSERT(!key->IsSelfSigned());
        UNIT_ASSERT(!key->AllowServerAuth());
        UNIT_ASSERT_STRINGS_EQUAL(key->Subject(), kTeamcitySubject);

        const auto& chain = key->Chain();
        UNIT_ASSERT_EQUAL(chain.size(), 3);
        UNIT_ASSERT_STRINGS_EQUAL(chain[0], "teamcity@ld.yandex.ru");
        UNIT_ASSERT_STRINGS_EQUAL(chain[1], "YandexInternalCA");
        UNIT_ASSERT_STRINGS_EQUAL(chain[2], "YandexInternalRootCA");
    }

    Y_UNIT_TEST(ValidTeamcityFromFile) {
        TTempDir tempDir("tls");
        const auto& keyPath = tempDir.Path() / "teamcity.pem";
        TFileOutput secretsIgnore(keyPath);
        secretsIgnore.Write(kTeamcityPrivate);
        secretsIgnore.Write(kTeamcityCert);
        secretsIgnore.Write(kTeamcityChain);
        secretsIgnore.Flush();

        const auto& key = TTlsKey::FromFile(keyPath);
        UNIT_ASSERT(key);

        UNIT_ASSERT(key->IsValid());
        UNIT_ASSERT(key->HaveCerts());
        UNIT_ASSERT(key->AllowClientAuth());
        UNIT_ASSERT(!key->IsSelfSigned());
        UNIT_ASSERT(!key->AllowServerAuth());
        UNIT_ASSERT_STRINGS_EQUAL(key->Subject(), kTeamcitySubject);

        const auto& chain = key->Chain();
        UNIT_ASSERT_EQUAL(chain.size(), 3);
        UNIT_ASSERT_STRINGS_EQUAL(chain[0], "teamcity@ld.yandex.ru");
        UNIT_ASSERT_STRINGS_EQUAL(chain[1], "YandexInternalCA");
        UNIT_ASSERT_STRINGS_EQUAL(chain[2], "YandexInternalRootCA");
    }

    Y_UNIT_TEST(ValidTeamcitySeparatedFile) {
        TTempDir tempDir("tls");
        const auto& keyPath = tempDir.Path() / "teamcity.key";
        const auto& crtPath = tempDir.Path() / "teamcity.crt";
        {
            TFileOutput secretsIgnore(keyPath);
            secretsIgnore.Write(kTeamcityPrivate);
            secretsIgnore.Flush();
        }

        {
            TFileOutput secretsIgnore(crtPath);
            secretsIgnore.Write(kTeamcityCert);
            secretsIgnore.Write(kTeamcityChain);
            secretsIgnore.Flush();
        }

        const auto& key = TTlsKey::FromFile(keyPath);
        UNIT_ASSERT(key);

        UNIT_ASSERT(key->IsValid());
        UNIT_ASSERT(key->HaveCerts());
        UNIT_ASSERT(key->AllowClientAuth());
        UNIT_ASSERT(!key->IsSelfSigned());
        UNIT_ASSERT(!key->AllowServerAuth());
        UNIT_ASSERT_STRINGS_EQUAL(key->Subject(), kTeamcitySubject);

        const auto& chain = key->Chain();
        UNIT_ASSERT_EQUAL(chain.size(), 3);
        UNIT_ASSERT_STRINGS_EQUAL(chain[0], "teamcity@ld.yandex.ru");
        UNIT_ASSERT_STRINGS_EQUAL(chain[1], "YandexInternalCA");
        UNIT_ASSERT_STRINGS_EQUAL(chain[2], "YandexInternalRootCA");
    }

    Y_UNIT_TEST(ValidTeamcitySeparatedFiles) {
        TTempDir tempDir("tls");
        const auto& keyPath = tempDir.Path() / "teamcity.key";
        const auto& crtPath = tempDir.Path() / "teamcity.crt";
        const auto& intermediatePath = tempDir.Path() / "chain.pem";

        {
            TFileOutput secretsIgnore(tempDir.Path() / "0000");
            secretsIgnore.Write("NOT A PEM");
            secretsIgnore.Flush();
        }

        {
            TFileOutput secretsIgnore(intermediatePath);
            secretsIgnore.Write(kTeamcityChain);
            secretsIgnore.Flush();
        }

        {
            TFileOutput secretsIgnore(keyPath);
            secretsIgnore.Write(kTeamcityPrivate);
            secretsIgnore.Flush();
        }

        {
            TFileOutput secretsIgnore(crtPath);
            secretsIgnore.Write(kTeamcityCert);
            secretsIgnore.Flush();
        }

        const auto& key = TTlsKey::FromFile(keyPath);
        UNIT_ASSERT(key);

        UNIT_ASSERT(key->IsValid());
        UNIT_ASSERT(key->HaveCerts());
        UNIT_ASSERT(key->AllowClientAuth());
        UNIT_ASSERT(!key->IsSelfSigned());
        UNIT_ASSERT(!key->AllowServerAuth());
        UNIT_ASSERT_STRINGS_EQUAL(key->Subject(), kTeamcitySubject);

        const auto& chain = key->Chain();
        UNIT_ASSERT_EQUAL(chain.size(), 2);
        UNIT_ASSERT_STRINGS_EQUAL(chain[0], "teamcity@ld.yandex.ru");
        UNIT_ASSERT_STRINGS_EQUAL(chain[1], "YandexInternalCA");
    }

    Y_UNIT_TEST(InvalidSelfsigned) {
        const auto& key = TTlsKey::FromPem(kSelfSignedCert);
        UNIT_ASSERT(key);
        UNIT_ASSERT(key->HaveCerts());
        UNIT_ASSERT(key->IsSelfSigned());
        UNIT_ASSERT(!key->IsValid());
    }

    Y_UNIT_TEST(KeyOnly) {
        const auto& key = TTlsKey::FromPem(kKeyOnly);
        UNIT_ASSERT(key);
        UNIT_ASSERT(!key->IsValid());
        UNIT_ASSERT(!key->HaveCerts());
    }

    Y_UNIT_TEST(EncryptedShouldntRequirePass) {
        const auto& key = TTlsKey::FromPem(kEncrypted);
        UNIT_ASSERT(!key);
    }
}
