#include "jwt.h"

#include <security/ant-secret/internal/string_utils/common.h>

#include <library/cpp/string_utils/base64/base64.h>
#include <util/string/vector.h>
#include <util/string/split.h>

namespace NTokenValidators {
    namespace {
        bool isBase64Json(const TStringBuf data) {
            auto decoded = Base64DecodeUneven(data);
            return decoded.StartsWith('{') && decoded.EndsWith('}');
        }

    }

    TJwt::TJwt(NSSInternal::TContext& ctx)
        : TSimple(ctx, {R"([A-Za-z0-9\-_]{20,}\.[A-Za-z0-9\-_]{10,}(\.[A-Za-z0-9\-_]{20,})?)"})
    {
    }

    bool TJwt::IsTokenValid(size_t /*id*/, const TStringBuf token) {
        const auto& parts = SplitString(TString(token), ".");
        if (parts.size() < 2) {
            return false;
        }

        for (size_t i = 0; i < parts.size(); ++i) {
            if (!NStringUtils::IsBase64Url(parts[i])) {
                // must be base64 encoded
                return false;
            }

            if (i < 2 && !isBase64Json(parts[i])) {
                // must be json inside
                return false;
            }
        }
        return true;
    }

    TMaybe<TValidatorResult> TJwt::Validate(const TStringBuf data) {
        auto secret = Match(data);
        if (!secret) {
            return Nothing();
        }

        secret->Type = SecretType();
        secret->Additional = {
            {"have_sign", StringSplitter(secret->Secret).Split('.').Count() > 2 ? "true" : "false"},
        };
        return secret;
    }

    const TString TJwt::Name() const {
        return "jwt";
    }

    const TString TJwt::SecretType() const {
        return "JWT";
    }

    bool TJwt::CanValidate() {
        return false;
    }

}
