#include "simple.h"

#include <util/string/vector.h>

namespace NTokenValidators {
    namespace {
        constexpr char kAnchorsLeft[] = {'\'', '"', '\x20', '\t', '\r', '\n', '\f', '\v', '>', '=', '`', ':'};
        constexpr char kAnchorsRight[] = {'\'', '"', '\x20', '\t', '\r', '\n', '\f', '\v', '<', '&', '`', ';', '\\'};

    }

    TSimple::TSimple(NSSInternal::TContext& ctx, const TVector<TString>& rawPatterns)
        : ctx(ctx)
        , rawPatterns(rawPatterns)
    {
        InitializePatterns();
    }

    bool TSimple::InitializePatterns() {
        TVector<TString> searchPatterns;
        for (const auto& pattern : rawPatterns) {
            prefilterPatterns.push_back(R"((['"`\s>=:]|^))" + pattern + R"(([\\<\s`"'&;]|$))");
            searchPatterns.push_back("^" + pattern + "$");
        }

        re = CompilePatterns("simple_validator_db", searchPatterns, HS_FLAG_DOTALL | HS_FLAG_SINGLEMATCH);
        return true;
    }

    TVector<TString> TSimple::PrefilterPatterns() const {
        return prefilterPatterns;
    }

    TMaybe<TValidatorResult> TSimple::Match(const TStringBuf data) {
        const auto& token = FindToken(data);
        if (token.empty()) {
            return Nothing();
        }

        TValidatorResult secret = {
            .Secret = TString{token},
            .Type = SecretType(),
        };
        return secret;
    }

    TMaybe<TValidatorResult> TSimple::Validate(const TStringBuf data) {
        auto secret = Match(data);
        if (!ctx.Validator || !secret || !CanValidate()) {
            return Nothing();
        }

        const auto& info = ctx.Validator->Call(Name(), secret->Secret);
        if (!info) {
            secret->ValidationError = true;
        }
        if (!info || !info->Valid) {
            return secret;
        }

        secret->Validated = true;
        secret->Type = info->Type;
        secret->TokenUser = info->Users;
        secret->ValidateUrl = info->ValidationUrl;
        secret->Owners = info->Owners;
        secret->Additional = ParseAdditional(info->AdditionalInfo);
        return secret;
    }

    NSSInternal::TSecretAdditional TSimple::ParseAdditional(const NJson::TJsonValue& additionalInfo) {
        Y_UNUSED(additionalInfo);
        return {};
    }

    TStringBuf TSimple::FindToken(TStringBuf token) {
        for (const auto& ch : kAnchorsLeft) {
            if (token.StartsWith(ch)) {
                token.Skip(1);
                break;
            }
        }

        for (const auto& ch : kAnchorsRight) {
            if (token.EndsWith(ch)) {
                token.Chop(1);
                break;
            }
        }

        TStringBuf result;
        auto matcher = [&result, token, this](unsigned int id, unsigned long long /*from*/, unsigned long long /*to*/) {
            if (IsTokenValid(id, token)) {
                result = token;
            }
        };

        NHyperscan::ScanPtr(re.db, re.scratch, token, matcher);
        return result;
    }

}
