#include "xiva.h"

#include <security/ant-secret/internal/string_utils/common.h>
#include <util/string/join.h>
#include <util/generic/set.h>

namespace NTokenValidators {
    namespace {
        TString JoinSet(const TSet<TString>& set) {
            TVector<TString> result;
            for (const auto& item : set) {
                if (item.empty()) {
                    continue;
                }
                result.push_back(item);
            }
            return JoinSeq(",", result);
        }

    }

    TXiva::TXiva(NSSInternal::TContext& ctx)
        : TSimple(ctx, {R"([a-f0-9]{40})"})
    {
    }

    NSSInternal::TSecretAdditional TXiva::ParseAdditional(const NJson::TJsonValue& additional) {
        if (Y_UNLIKELY(!additional.IsDefined() || !additional.IsMap())) {
            return {};
        }

        const auto& environments = additional["environments"];
        if (!environments.IsDefined() || !environments.IsMap()) {
            return {};
        }

        TSet<TString> xivaEnvironments;
        TSet<TString> xivaServices;
        TSet<TString> xivaTypes;
        TSet<TString> xivaOwners;
        for (auto&& [name, env] : environments.GetMapSafe()) {
            xivaEnvironments.insert(name);
            xivaServices.insert(env["service"].GetStringSafe());
            xivaTypes.insert(env["type"].GetStringSafe());
            xivaOwners.insert(env["owner"].GetStringSafe());
        }
        return NSSInternal::TSecretAdditional{
            {"xiva_environments", JoinSet(xivaEnvironments)},
            {"xiva_services", JoinSet(xivaServices)},
            {"xiva_types", JoinSet(xivaTypes)},
            {"xiva_owners", JoinSet(xivaOwners)}};
    }

    bool TXiva::IsTokenValid(size_t /*id*/, const TStringBuf token) {
        return NStringUtils::IsHash(token);
    }

    const TString TXiva::Name() const {
        return "xiva";
    }

    const TString TXiva::SecretType() const {
        return "Xiva";
    }

    bool TXiva::CanValidate() {
        return true;
    }

}
