#include "yandex_oauth.h"

#include <security/ant-secret/internal/string_utils/common.h>
#include <security/ant-secret/internal/blackbox/oauth.h>

#include <util/string/join.h>
#include <util/string/vector.h>


namespace NTokenValidators {
    /*
We support 3 types of oauth tokens:
  1. old legacy token - md5 hash, e.g.:
    03c7c0ace395d80182db07ae2c30f034
  2. token with embedded info - base64url(uid, clid, shard, random), [a-zA-Z0-9_-]+, e.g.:
    AQAD-qJSJpcjAAADwHcUZ2Serk5EmBjzGpemfJQ
  3. stateless token - ver.uid.client_id.expires.token_id.key_id.iv.data.tag, e.g.:
    1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q
*/
    TYandexOAuth::TYandexOAuth(NSSInternal::TContext& ctx)
        : TSimple(
              ctx,
              {
                  // legacy
                  {R"([a-f0-9]{32})"},
                  // embedded - remove prefix?
                  {R"([A-Za-z0-9_\-]{39})"},
                  // new embed
                  {R"(y[0-9]_[A-Za-z0-9_\-]{55})"},
                  // stateless, for now version is always 1
                  {R"(1\.\d+\.\d+\.\d+\.\d+\.\d+\.[\w\-]{16}\.[\w\-]+\.[\w\-]{22})"},
              }) {
    }

    bool TYandexOAuth::IsTokenValid(size_t id, const TStringBuf token) {
        switch (id) {
            case 0:
                // legacy
                return NStringUtils::IsHash(token);
            case 1:
                return NBlackBox::IsEmbeddedV2Token(token);
            case 2:
                return NBlackBox::IsEmbeddedV3Token(token);
            case 3:
                // stateless
                // passport/infra/daemons/blackbox/src/oauth_processor.cpp:
                // token.size() > 70 && NPassport::NUtils::beginsWith(token, "1.")
                return token.size() > 70;
            default:
                ythrow TSystemError() << "unexpected token pattern id: " << id;
        }
    }

    const TString TYandexOAuth::Name() const {
        return "yandex-oauth";
    }

    const TString TYandexOAuth::SecretType() const {
        return "YandexOAuth";
    }

    bool TYandexOAuth::CanValidate() {
        return true;
    }

    NSSInternal::TSecretAdditional TYandexOAuth::ParseAdditional(const NJson::TJsonValue& additional) {
        if (Y_UNLIKELY(!additional.IsDefined() || !additional.IsMap())) {
            return {};
        }

        return NSSInternal::TSecretAdditional{
            {"oauth_token_id", additional["token_id"].GetStringSafe(TString())},
            {"oauth_client_id", additional["client_id"].GetStringSafe(TString())},
            {"oauth_client_name", additional["client_name"].GetStringSafe(TString())},
            {"oauth_device_id", additional["device_id"].GetStringSafe(TString())},
            {"oauth_device_name", additional["device_name"].GetStringSafe(TString())},
            {"oauth_create_time", additional["create_time"].GetStringSafe(TString())},
            {"oauth_issue_time", additional["issue_time"].GetStringSafe(TString())},
            {"oauth_scopes", JoinSeq(",", SplitString(additional["scopes"].GetStringSafe(TString()), " "))}};
    }

}
