#include "secret.h"

#include <security/libs/cpp/log/log.h>

#include <util/charset/utf8.h>
#include <util/string/builder.h>

namespace NSSInternal {
    namespace {
        THashMap<TString, TVector<std::pair<TString, TString>>> kRelatedInfo{
            {"StaffSSH", {
                {"users", "secret_users"},
                {"fingerprint", "ssh_fingerprint"},
            }},
            {"SSLKey", {
                {"subject", "tls_subject"},
                {"client_auth", "tls_client_cert"},
                {"server_auth", "tls_server_cert"},
            }},
            {"YOAuthInternal", {
                {"users", "secret_users"},
                {"client_id", "oauth_client_id"},
                {"client_name", "oauth_client_name"},
            }},
            {"YOAuthExternal", {
                {"users", "secret_users"},
                {"client_id", "oauth_client_id"},
                {"client_name", "oauth_client_name"},
            }},
            {"YSessionInternal", {
                {"users", "secret_users"},
            }},
            {"YSessionExternal", {
                {"users", "secret_users"},
            }},
            {"Xiva", {
                {"environments", "xiva_environments"},
                {"services", "xiva_services"},
                {"owners", "xiva_owners"},
            }},
            {"GithubInternal", {
                {"users", "secret_users"},
            }},
            {"GithubExternal", {
                {"users", "secret_users"},
            }},
            {"YCApiKey", {
                {"account_id", "account_id"},
                {"folder_id", "folder_id"},
                {"subject", "kind"},
                {"environment", "env"},
            }},
            {"YCToken", {
                {"account_id", "account_id"},
                {"folder_id", "folder_id"},
                {"subject", "kind"},
                {"environment", "env" },
            }},
            {"YCCookie", {
                {"account_id", "account_id"},
                {"folder_id", "folder_id"},
                {"subject", "kind"},
                {"environment", "env"},
            }},
        };

        THashMap<TString, TString> kSecretTypes{
            {"StaffSSH", "SSH key from staff"},
            {"SSLKey", "TLS private key"},
            {"YOAuthInternal", "Yandex team OAuth token"},
            {"YOAuthExternal", "Yandex OAuth token"},
            {"YSessionInternal", "Yandex team session"},
            {"YSessionExternal", "Yandex session"},
            {"YCApiKey", "Yandex.Cloud API key"},
            {"YCToken", "Yandex.Cloud IAM token"},
            {"YCCookie", "Yandex.Cloud IAM cookie"},
            {"Xiva", "Xiva token"},
            {"GithubInternal", "Github (internal) auth token"},
            {"GithubExternal", "Github (external) auth token"},
        };

    }

    void TSecret::ToJson(NJson::TJsonWriter& json) const {
        json.OpenMap();
        if (Type) {
            json.Write("type", Type);
        }
        if (Secret) {
            json.Write("secret", Secret);
        }

        json.Write("validated", Validated);
        json.Write("line_no", LineNo);
        if (!SourceLine.empty()) {
            if (IsUtf(SourceLine)) {
                json.Write("source_line", SourceLine);
            } else {
                NSecurityHelpers::LogWarn("Skip non UTF-8 source line", "source_line", SourceLine);
            }
        }

        if (!Owners.empty()) {
            json.OpenArray("owners");
            for (const auto& owner: Owners) {
                json.Write(owner);
            }
            json.CloseArray();
        }

        if (!Additional.empty()) {
            json.OpenMap("additional");
            for (const auto& item : Additional) {
                if (!item.second.empty()) {
                    json.Write(item.first, item.second);
                }
            }
            json.CloseMap();
        }
        json.CloseMap();
    }

    const TString TSecret::ShortAdditional() const {
        if (!kRelatedInfo.contains(Type)) {
            return TString();
        }

        TStringBuilder result;
        bool first = true;
        for (auto&& [name, key] : kRelatedInfo[Type]) {
            const auto it = Additional.find(key);
            if (it != Additional.end()) {
                if (first) {
                    first = false;
                } else {
                    result << " ";
                }

                result << name << "=" << it->second;
            }
        }

        return result;
    }

    const TString TSecret::HumanType() const {
        if (kSecretTypes.contains(Type)) {
            return kSecretTypes[Type];
        }

        return Type;
    }

}
